<?php

namespace Hilco\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use DB;
use Auth;
use Elasticquent\ElasticquentTrait;
use Hilco\Facades\HilcoAssets;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Class WebPart
 * @package Hilco\Models
 * @property int $id
 * @property string $name
 * @property string $part_detail
 * @property string $safe_name
 * @property string $override_name
 * @property-read Part $part
 * @property-read WebPartFamily $family
 * @property-read WebPartAttributeValue[] $attributeValues
 * @method static Builder latest()
 * @property string $part_number
 * @property integer $part_id
 * @property integer $webpartfamily_id
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property-read \Hilco\Models\WebPartFamily $webPartFamily
 * @property-write mixed $web_part_attributes
 * @property-read mixed $part_name
 * @property-read mixed $family_name
 * @property-read mixed $context
 * @property-read mixed $default_context
 * @mixin \Eloquent
 * @property integer $webfamily_id
 * @method static WebPart firstOrNew(array $attributes)
 */
class WebPart extends WebModel
{
    protected $table = "WebParts";
    protected $fillable = ['min_quantity', 'quantity_step', 'quantity_presets', 'name', 'webAttributes', 'keywords', 'webfamily_id', 'part_id', 'part_number', 'is_visible', 'is_family_image', 'in_stock_label', 'out_of_stock_label'];
    protected $casts = [
        ['is_visible' => 'boolean' , 'is_family_image' => 'boolean']
    ];

    use AlgoliaEloquentTrait;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
            'part_number',
            'nodash_part_number',
        ],
        'attributesForFaceting' => [
            'webSiloIds',
            'validPlants',
            'is_rx',
        ]
    ];

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

//        DB::enableQueryLog();
        static::has('webFamily.webCollections.webCategories.webGroups.webSilos')->whereHas('part', function ($query) {
            $query->whereHas('inventoryItems', function ($query) {
                $query->isValid();
            });
        })
        ->where('is_visible', 1)
        ->with('part.inventoryItems', 'webFamily.webCollections.webCategories.webGroups.webSilos.plants')->chunk(100, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
            /** @var \AlgoliaSearch\Index $index */
//            dd(DB::getQueryLog());
//            DB::flushQueryLog();
            foreach ($indicesTmp as $index) {
                $records         = [];
                $recordsAsEntity = [];

                foreach ($models as $model) {
                    if ($modelHelper->indexOnly($model, $index->indexName)) {
                        $records[] = $model->getAlgoliaRecordDefault($index->indexName);

                        if ($onInsert && is_callable($onInsert)) {
                            $recordsAsEntity[] = $model;
                        }
                    }
//                    dd(DB::getQueryLog());
                }

                $index->addObjects($records);

                if ($onInsert && is_callable($onInsert)) {
                    call_user_func_array($onInsert, [$recordsAsEntity]);
                }
            }

        });

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }
    }

    public function indexOnly() {
        if (!$this->is_visible) return false;
        if (!$this->webFamily->is_visible) return false;

        return true;
    }

    public function getAlgoliaRecord() {
        $record = array_only($this->toArray(), [
            'id',
            'part_id',
            'name',
            'part_number',
            'webFamily',
        ]);
        $record['nodash_part_number'] = str_replace('-', '', $this->part_number);
        $paths = $this->hierarchyPaths;
        $silos = [];
        foreach ($paths as $path => $info) {
            $silos[$info['webSilo']->id] = $info['webSilo']->id;
        }
        $record['webSiloIds'] = array_keys($silos);
        $record['webFamily']['slug'] = !is_null($this->webFamily) ? $this->webFamily->slug : null;
        $record['validPlants'] = !is_null($this->part) && !is_null($this->part->inventoryItems) ? $this->part->inventoryItems->pluck('plant') : [];
        $record['webCollectionIds'] = !is_null($this->webFamily) ? $this->webFamily->webCollections->pluck('id') : [];
        $record['is_rx'] = $this->part->is_rx;
        return $record;
    }

    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

//    use ElasticquentTrait;

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords){
        $params=[
            'multi_match' => [
                'query'     => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields'    => ['_all']
            ]
        ];
        return $params;
    }

//	use KeyWordsFunctional;

	public static function clearIsFamilyImage($familyId, $notIn=[]){
		self::where('webfamily_id', $familyId)->where('is_family_image',1)->whereNotIn('id',$notIn)->update(['is_family_image'=>0]);
	}

    /**
     * @description index all the entries in an Eloquent model (if needed)
     */
    public static function elasticIndexingIfNeeded(){
//		$elasticClient = ClientBuilder::create()->setHosts(config('elasticquent.config.hosts'))->build();
//		$instance = new static;

//		$basicParams = $instance->getBasicEsParams();
//		$indexParams = ['index' => $basicParams['index']];
//
//		$result = $elasticClient->indices()->exists($indexParams);
//		if (!$result)

//		self::createIndex();
        $mappingExists = self::mappingExists();
        if (!$mappingExists) {
            self::createIndex();
            self::putMapping();
            self::addAllToIndex();
        }

//		if(!$result || !$mappingExists)
//		return (!$result || !$mappingExists);
	}
	
	public function part() {
		return $this->belongsTo(Part::class, 'part_id', 'id');
	}

	/**
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function webFamily() {
		return $this->belongsTo(WebFamily::class, 'webfamily_id', 'id');
	}

	public function scopeJoinDiscount($query){
		$customerId = Auth::user()->customer_id;
		return $query->select(['WebParts.*',DB::raw('SUM(CustomerDiscounts.disc_val) as disc_val')])->leftJoin('CustomerDiscounts', function($query)use($customerId){
			return $query->on('CustomerDiscounts.part_id','=','WebParts.id')
						 ->orOn('CustomerDiscounts.productfamily_id','=','WebParts.webfamily_id')
						 ->where('CustomerDiscounts.customer_id','=',$customerId);
		})->groupBy('id');
	}

	public function scopeIsFamilyImage($query){
		$query->where('is_family_image',1)->visible()->whereHas('part', function ($query) {
            return $query->whereHas('inventoryItems', function ($query) {
                return $query->isValid();
            });
        });
	}

	public function scopePriceList($query, $priceList, $currency, $quantityLevel = 1) {
		return $query
			->leftJoin('PriceLists', 'PriceLists.part_no', '=', 'WebParts.part_number')
			->where('price_list', $priceList)
			->where('currency', $currency)
			->where('quantity_level', '>=', $quantityLevel)
			->select('WebParts.*', 'PriceLists.price')
		;
	}

	public function webAttributes() {
		return $this->belongsToMany(WebAttribute::class, 'WebAttribute_WebPart', 'webpart_id', 'webattribute_id')
			->withPivot(['id','attribute_value']);
	}

	public function webAttributeValue($webAttributeId) {
		if (!$this->relationLoaded('webAttributes')) $this->load('webAttributes');
		$webAttributeValue = $this->webAttributes->find($webAttributeId);
		if (!is_null($webAttributeValue)) return $webAttributeValue->pivot->attribute_value;
		return null;
	}

	public function setWebAttributeValue($webAttributeId, $value) {
		$this->webAttributes()->sync([$webAttributeId => ['attribute_value' => $value]], false);
	}
	public function setWebAttributesAttribute($attributes) {
		$this->save();
		foreach ($attributes as $webAttributeId => $attributeValue) {
			$this->setWebAttributeValue($webAttributeId, $attributeValue);
		}
	}

    public function relatedParent() {
        return $this->morphToMany(WebFamily::class, 'related', 'WebFamily_Related', 'related_id', 'webfamily_id')->withPivot(['related_category']);
    }

	public function getPartNameAttribute() {
		$part = $this->part;
		if (is_null($part)) return null;
		return $part->part_desc;
	}
	
	public function getFamilyNameAttribute() {
		$family = $this->webFamily;
		if (is_null($family)) return null;
		return $family->name;
	}

	public function getListPriceAttribute() {
	    return $this->part->list_price;
    }

    public function getCustomerPriceAttribute() {
        return $this->part->customer_price;
    }

    public function getCustomerPrice($quantity = 1) {
        return $this->part->getCustomerPrice($quantity);
    }

    public function customerPrices() {
        return $this->part->customerPrices();
    }

    public function getCustomerDiscount() {
	    return $this->part->getCustomerDiscount();
    }

    public function getImageLink($width = null, $height = null, $avoidCache = false) {
        return $this->primaryImage($width ? $width : false, $height ? $height : false);
    }

//	public static function getImageUrl($partNumber, $width = null, $height = null, $avoidCache=false) {
//		$url = config('hilco.images.url');
//		if(!isset($url)) $url = config('hilco-b2b.images.url');
//		$url .= '/' . $partNumber;
//		if ($width) $url .= "/$width";
//		if ($height) $url .= "/$height";
//
//		return ($avoidCache)?\ImgHelper::avoidCache($url):$url;
//	}

    public function getImageUrl($width = false, $height = false) {
        return HilcoAssets::part($this, $width, $height);
    }

	public function scopePriceRange($query, $min, $max) {
		return $query->whereHas('part', function($part) use ($min, $max){
			$part->whereBetween('list_price', [$min, $max]);
		});
	}

	public function scopeManagedBy($query, $manager) {
		return $query->whereHas('part', function ($query) use ($manager) {
			return $query->where('pf_prod_mgr', $manager);
		});
	}

	public function scopeInFamily($query, $webFamilyId) {
		return $query->whereHas('webFamily', function ($query) use ($webFamilyId) {
			return $query->where('webfamily_id', $webFamilyId);
		});
	}

    public function scopeInCollection($query, $webCollectionId) {
        return $query->whereHas('webFamily', function ($query) use ($webCollectionId) {
            return $query->inCollection($webCollectionId);
        });
    }

    public function scopeInCategory($query, $webCategoryId) {
        return $query->whereHas('webFamily', function ($query) use ($webCategoryId) {
            return $query->inCategory($webCategoryId);
        });
    }

	public function scopeInGroup($query, $webGroupId) {
		return $query->whereHas('webFamily', function ($query) use ($webGroupId) {
			return $query->inGroup($webGroupId);
		});
	}

	public function scopeVisible(Builder $query, $webSiloId = false) {
	    if ($webSiloId === false) {
	        $webSiloId = b2b()->activeWebSilo()->id;
        }
		return $query
			->where('WebParts.is_visible', '1')
			->whereHas('part', function ($query) use ($webSiloId) {
//				$user = Auth::user();
//				$userCustomer = (isset($user))?$user->customer:null;
//				$pharmancyInfo = null;
//				if(isset($userCustomer)){
//					$pharmancyInfo = $userCustomer->pharmacyInfo()->first();
//				}
//				if(!isset($pharmancyInfo) || !$pharmancyInfo->checkRx()){
//					$query = $query->isNotRX();
//				}

                $query->where('private_lbl', '0');
                if (b2b()->activeCountry() != 'US') {
                    $query->isNotRX();
                }

				return $query->whereHas('inventoryItems', function ($query) use ($webSiloId) {
                    $activeCountry = b2b()->activeCountry();
                    if ($activeCountry == 'US') $query->inPlant('PLAINVILLE');
                    return $query->inSilo($webSiloId);
				});
			})
		;
	}

    public function getIsRXAttribute() {
	    $part = $this->part;
	    if (is_null($part)) return null;

        return (in_array($part->product_family_no, Part::$rxProductFamilyNumbers));
    }

	public function visibleInSilo($webSilo) {
	    if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);
//	    $webSiloId = ($webSilo instanceof WebSilo) ? $webSilo->id : $webSilo;
        $inSilo = false;
        foreach ($this->part->inventoryItems as $inventoryItem) {
            foreach ($webSilo->plants as $plant) {
                $flags = explode(',', $plant->pivot->web_flags);
                $inPlant = $inventoryItem->plant_id == $plant->id;
                $hasFlag = in_array($inventoryItem->web_flag, $flags);
                $isNotObsolete = ($inventoryItem->part_stat != 'Phase-out' || $inventoryItem->max_avail_qty > 0);
                $inSilo = $inPlant && $hasFlag && $isNotObsolete;
            }

        }
        return $inSilo;
    }

    public function getHasValidInventoryAttribute() {
        $status = 'none';
        $hasAll = true;
        if ($this->part) {
            foreach ($this->part->inventoryItems as $inventoryItem) {
                if ($inventoryItem->is_valid) {
                    $status = 'some';
                } else {
                    $hasAll = false;
                }
            }
        }

        if ($hasAll) $status = 'all';
        return $status;
    }

    public function getHierarchyPathsAttribute() {
        $isVisible = true;
        if (!$this->is_visible) $isVisible = 'Manually Hidden';
        if (is_null($this->part)) $isVisible = 'Not in Point.man';
        if (is_null($this->webFamily)) $isVisible = 'Not in a family';
        if ($this->has_valid_inventory == 'none') $isVisible = 'No valid inventory';

        $paths = [];
        if (!is_null($this->webFamily)) {
            foreach ($this->webFamily->webCollections as $webCollection) {
                foreach ($webCollection->webCategories as $webCategory) {
                    foreach ($webCategory->webGroups as $webGroup) {
                        foreach ($webGroup->webSilos as $webSilo) {
                            if ($isVisible === true) {
                                if (!$this->visibleInSilo($webSilo)) $isVisible = 'Excluded due to silo rules';
                            }
                            if (!$this->webFamily->is_visible) $isVisible = 'Parent Family is not visible';
                            if (!$webCollection->is_visible) $isVisible = 'Parent Collection is not visible';
                            if (!$webCategory->is_visible) $isVisible = 'Parent Category is not visible';
                            if (!$webGroup->is_visible) $isVisible = 'Parent Group is not visible';
                            $paths[] = [
                                'webSilo' => $webSilo,
                                'webGroup' => $webGroup,
                                'webCategory' => $webCategory,
                                'webCollection' => $webCollection,
                                'webFamily' => $this->webFamily,
                                'isVisible' => $isVisible,
                            ];
                        }
                    }
                }
            }
        }
        return $paths;
    }

    public function hierarchyPaths() {
//        $data['webPart'] = $this;
//        $isVisible = true;
//        if (!$this->is_visible) $isVisible = 'Manually Hidden';
//        if (is_null($this->part)) $isVisible = 'Not in Point.man';
//        if (is_null($this->webFamily)) $isVisible = 'Not in a family';
//        if ($this->has_valid_inventory == 'none') $isVisible = 'No valid inventory';
//        $data['isVisible'] = $isVisible;
        $paths = ['webPart' => $this];
        if ($this->webFamily) $paths = $this->webFamily->hierarchyPaths($paths);
        return $paths;
    }


    public function setWebfamilyIdAttribute($value) {
        if (!$value || $value < 1) $this->attributes['webfamily_id'] = null;
        else $this->attributes['webfamily_id'] = $value;
    }

    public function getInStockAttribute() {
        return $this->part->in_stock;
    }

    public function getInStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'in_stock_label');
        return strlen($modelValue) ? $modelValue : config('hilco.inStockLabels.0', 'In Stock');
    }

    public function getOutOfStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'out_of_stock_label');
        return strlen($modelValue) ? $modelValue : config('hilco.outOfStockLabels.0', 'Out of stock');
    }

    public function getStockStatusAttribute() {
        return $this->part->stock_status;
    }

    public function getStockLabelAttribute() {
        $status = $this->stock_status;
        switch ($status) {
            case Part::IN_STOCK:
                return $this->in_stock_label;
            case Part::OUT_OF_STOCK:
                return $this->out_of_stock_label;
            case Part::DELAYED:
                return "Ships in 3-5 days";
        }
    }

	use HasAssets, RevisionableTrait;

}
