<?php

namespace Hilco\Models;

use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Collection;

/**
 * Hilco\Models\VisionWebOrder
 * @property integer $id
 * @property string $visionweb_orderid
 * @property string $order_date
 * @property string $po_number
 * @property string $cust_no
 * @property string $shipto_addr_no
 * @property string $frame_sku
 * @property string $right_lens_sku
 * @property string $left_lens_sku
 * @property boolean $is_processed
 * @property string $last_error_message
 * @property string $last_exception
 * @property Collection|VisionWebOrderAddOn[] $visionWebOrderAddOns
 * @method static Builder|VisionWebOrder[] unprocessed()
 */
class VisionWebOrder extends WebModel {
    protected $table = 'VisionWebOrders';
    protected $fillable = [
        'visionweb_orderid',
        'order_date',
        'po_number',
        'cust_no',
        'shipto_addr_no',
        'frame_sku',
        'right_lens_sku',
        'left_lens_sku',
        'is_processed',
        'last_error_message',
        'last_exception',
    ];
    protected $casts = ['is_processed' => 'boolean',];

    /**
     * VisionWebOrderAddOn relationship
     * @return HasMany
     */
    public function visionWebOrderAddOns(): HasMany {
        return $this->hasMany(VisionWebOrderAddOn::class, 'visionweborder_id', 'id');
    }

    /**
     * @param Builder $query
     * @return Builder
     */
    public function scopeUnprocessed (Builder $query): Builder {
        return $query->where('is_processed', '=', 0);
    }

    /**
     * @return string
     */
    public function getVisionWebOrderId(): string {
        return $this->visionweb_orderid;
    }

    /**
     * @return string
     */
    public function getOrderDate(): string {
        return $this->order_date;
    }

    /**
     * @return string
     */
    public function getPONumber(): string {
        return $this->po_number;
    }

    /**
     * @return string
     */
    public function getCustNo(): string {
        return $this->cust_no;
    }

    /**
     * @return string
     */
    public function getShipToAddrNo(): string {
        return $this->shipto_addr_no;
    }

    /**
     * @return string
     */
    public function getFrameSku(): string {
        return $this->frame_sku;
    }

    /**
     * @return string
     */
    public function getRightLensSku(): string {
        return$this->right_lens_sku;
    }

    /**
     * @return string
     */
    public function getLeftLensSku(): string {
        return $this->left_lens_sku;
    }

    /**
     * @return bool
     */
    public function isProcessed(): bool {
        return $this->is_processed;
    }

    /**
     * @return string
     */
    public function getLastErrorMessage(): string {
        return $this->last_error_message;
    }

    /**
     * @return string
     */
    public function getLastException(): string {
        return $this->last_exception;
    }

    /**
     * @param string $errorMessage
     * @return void
     */
    public function recordLatestError (string $errorMessage): void {
        $this->last_error_message = $errorMessage;
        $this->save();
    }

    /**
     * @param Exception $exception
     * @param bool $recordError
     * @return void
     */
    public function recordLatestException (Exception $exception, bool $recordError = true): void {
        $this->last_exception = $exception->getTraceAsString();
        if ($recordError) {
            $this->last_error_message = $exception->getMessage();
        }
        $this->save();
    }

    /**
     * @return void
     */
    public function markProcessed() {
        $this->is_processed = 1;
        $this->save();
    }
}