<?php

namespace Hilco\Models;

use Illuminate\Database\Eloquent\Relations\MorphToMany;

trait HasAssets {
    /**
     * @return MorphToMany
     */
    public function assets() {
        return $this->morphToMany(WebAsset::class, 'assetable', 'Assetables', 'assetable_id', 'webasset_id')->withPivot(['sequence', 'assetable_join_type', 'alt_text'])->wherePivot('deleted_at', '=', '0000-00-00 00:00:00');
    }

    public function assetsByType($type) {
        return $this->assets->where('pivot.assetable_join_type', $type);
    }

    /**
     * Returns the primary image asset for the model.
     *
     * @return WebAsset
     *
     */
    public function getPrimaryImage() {
        return $this->assetsByType('primary')->first();
    }

    public function getFeaturedImage() {
        $asset = $this->assetsByType('featured')->first();
        if (!is_null($asset)) return $asset;

        return $this->getPrimaryImage();
    }

    /**
     * Returns a URL for the primary image asset for the model, or nopic if it doesn't exist.
     *
     * @param $width
     * @param $height
     * @return string
     */
    public function primaryImage($width = false, $height = false) {
        $asset = $this->getPrimaryImage();
        if ($height === false && is_string($width)) return WebAsset::getPredefinedSizeUrl($asset, $width);
        return WebAsset::getSizedUrl($asset, $width, $height);
    }

    public function featuredImage($width = false, $height = false) {
        $asset = $this->getFeaturedImage();
        return WebAsset::getSizedUrl($asset, $width, $height);
    }



    /**
     * Attaches an asset to this model. If the unique parameter is true, it will detach all other assets of the specified type first.
     * @param $webAsset
     * @param $type
     * @param bool $unique
     * @return $this
     */
    public function setAssetByType($webAsset, $type, $unique = false) {
        if ($unique) $this->assets()->wherePivot('assetable_join_type', $type)->detach();
        $this->assets()->syncWithoutDetaching([$webAsset->id => ['assetable_join_type' => $type]]);
        $this->save();
        return $this;
    }

    public function setPrimaryImage($webAsset) {
        return $this->setAssetByType($webAsset, 'primary', true);
    }

    /**
     * Returns a URL for the primary image asset for the model, or false if it doesn't exist.
     * @param $width
     * @param $height
     * @return bool|string
     */
    public function primaryImageOrFalse($width = false, $height = false) {
        $asset = $this->getPrimaryImage();
        return is_null($asset) ? false : WebAsset::getSizedUrl($asset, $width, $height);
    }
}