<?php
/**
 * Created by PhpStorm.
 * User: brian
 * Date: 4/2/19
 * Time: 10:19 AM
 */

namespace Hilco\Models;

use Illuminate\Database\Eloquent\Model;
use Hilco\Models\WebModelTranslation;

/**
 * Trait HasModelTranslations
 * @package Hilco\Models
 */
trait HasModelTranslations {

    protected $queuedTranslations = [];

    public static function bootHasModelTranslations() {
        static::created (function (Model $translatableModel) {
            foreach ($translatableModel->queuedTranslations as $translationInfo) {
                $translatableModel->setTranslation($translationInfo['language'], $translationInfo['column_name'], $translationInfo['translation']);
            }
            $translatableModel->queuedTranslations = [];
        });
    }

    public function translations() {
        return $this->morphMany(WebModelTranslation::class, 'translatable');
    }

    public function getTranslation($columnName, $language = false, $default = false) {
        if ($language === false) $language = b2b()->activeLanguage();
        if ($default === false) $default = array_get($this->attributes, $columnName, '');
        $translation = $this->translations->where('language', $language)->where('column_name', $columnName)->first();
        return array_get($translation, "translation", $default);
    }

    public function deleteTranslation($columnName, $language){
        $this->translations->where('language', $language)->where('column_name', $columnName)->first()->delete();
    }

    public function setEnNameAttribute($name) {
        $this->setTranslation('en', 'name', $name);
    }

    public function setEnDetailAttribute($name){
        $this->setTranslation('en', 'detail' ,$detail);
    }

    public function setFrNameAttribute($name) {
        $this->setTranslation('fr', 'name', $name);
    }
    public function setFrDetailAttribute($detail){
        $this->setTranslation('fr', 'detail' ,$detail);
    }

    public function getEnNameAttribute() {
        return $this->getTranslation('name', 'en');
    }

    public function getEnDetailAttribute(){
        return $this->getTranslation('detail', 'en');
    }

    public function getFrNameAttribute() {
        return $this->getTranslation('name', 'fr');
    }

    public function getFrDetailAttribute(){
        return $this->getTranslation('detail', 'fr');
    }

    public function setTranslation($language, $columnName, $translation, $type = null) {
        if (!$this->exists) {
            $this->queuedTranslations[] = [
                'language' => $language,
                'column_name' => $columnName,
                'translation' => $translation
            ];
        } else {
            $modelTranslation = $this->translations()->withTrashed()->where('language', $language)->where('column_name', $columnName)->first();
            if(!empty($modelTranslation)){
                $modelTranslation->restore();
            }
            if(is_null($type)){
                $type = lcfirst(class_basename($this));
            }
            $this->translations()
                ->updateOrCreate(
                    [
                        'language' => $language,
                        'column_name' => $columnName
                    ],
                    [
                        'translatable_type' => $type,
                        'translation' => $translation
                    ]
                )
            ;
        }
    }

    protected static $morphMap = [
        'feedBackSubject'   => FeedbackSubject::class,
        'feedBackType'      => FeedbackType::class,
        'webAttribute'      => WebAttribute::class,
        'webLink'           => WebLink::class,
        'webLinkCategory'   => WebLinkCategory::class,
        'webPart'           => WebPart::class,
        'webFamily'         => WebFamily::class,
        'webCategory'       => WebCategory::class,
        'webCollection'     => WebCollection::class,
        'webGroup'          => WebGroup::class,
        'webSilo'           => WebSilo::class
    ];

    /**
     * Get the morph mapping of trigger/action types to their actual class models
     * @return array
     */
    public static function getMorphMap() {
        return self::$morphMap;
    }

    public static function getModelLangDetails($id, $model, $lang){
        $detail = WebModelTranslation::where('translatable_id', '=', $id)
            ->where('translatable_type', '=', $model)
            ->where('language', '=', $lang)
            ->where('column_name', '=', 'detail')
            ->first();

        if(empty($detail)){
            return '';
        }
        return $detail->translation;
    }

    public static function getSiloShippingNoteTranslation($id, $lang){
        $note = WebModelTranslation::where('translatable_id', '=', $id)
            ->where('translatable_type', '=', 'webSilo')
            ->where('language', '=', $lang)
            ->where('column_name', '=', 'shipping_note')
            ->first();

        if(empty($note)){
            return '';
        }
        return $note->translation;
    }
}