<?php

namespace Hilco\Models;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

/**
 * Model Class StagedOrderLine
 * @property integer id
 * @property integer stagedorder_id
 * @property integer line_number
 * @property integer exportedline_number
 * @property integer part_id
 * @property integer quantity
 * @property float unit_price
 * @property float ext_tax
 * @property boolean nocharge_flag
 * @property boolean override_price
 * @property boolean taxable
 * @property boolean is_contract
 * @property boolean is_kit
 * @property string pricelist
 * @property integer plant_id
 * @property string customer_reference
 * @property string line_comment_all
 * @property string line_comment_external
 * @property string line_comment_internal
 * @property string line_comment_display
 * @property string line_comment_picklist
 * @property string line_comment_deliverynote
 * @property string line_comment_manufacturing
 * @property string line_comment_invoice
 * @property mixed salesorderline_id
 * @property StagedOrder stagedOrder
 * @property Part part
 * @property Plant shipFromPlant
 * @property Plant shipPlant
 * @property SalesOrderLine salesOrderLine
 * @property Collection|StagedOrderDiscLine[] stagedOrderDiscLines
 * @property Collection|StagedOrderDiscLine[] discLines
 */
class StagedOrderLine extends StagingModel {
    protected $table = "StagedOrderLines";
    protected $fillable = [
        'stagedorder_id',
        'line_number',
        'part_id',
        'quantity',
        'unit_price',
        'ext_tax',
        'taxable',
        'is_contract',
        'is_kit',
        'pricelist',
        'customer_reference',
    ];

    protected static $allowedCommentVisibilities = [
        "all",
        "deliverynote",
        "display",
        "external",
        "internal",
        "invoice",
        "manufacturing",
        "picklist",
    ];

    public function getLineNumber(): int {
        return $this->line_number;
    }

    public function getCustomerReference(): ?string {
        return $this->customer_reference;
    }

    public function getTrayInfo() {
        return $this->customer_reference;
    }

    /**
     * @return float|null
     */
    public function getOverridePrice(): ?float {
        return $this->override_price;
    }

    public function getQuantity(): int {
        return $this->quantity;
    }

    /**
     * @return float
     */
    public function getTaxExtension(): float {
        return is_null($this->ext_tax) ? 0.0 : $this->ext_tax;
    }

    public function getUnitPrice() {
        return $this->unit_price;
    }

    public function getComment ($visibility): ?string {
        if (in_array($visibility, self::$allowedCommentVisibilities)) {
            $commentColumn = "line_comment_".$visibility;
            return $this->$commentColumn;
        }
        return null;
    }

    public function getPriceList(): ?string {
        return $this->pricelist;
    }

    /**
     * @return bool
     */
    public function isKit(): bool {
        return $this->is_kit;
    }

    /**
     * @return bool
     */
    public function isOnContractPricing(): bool {
        return $this->is_contract;
    }

    /**
     * @return bool
     */
    public function isNoCharge(): bool {
        return $this->nocharge_flag;
    }

    /**
     * @return bool
     */
    public function isTaxable(): bool {
        return $this->taxable;
    }

    public function getLinePrice() {
        return $this->unit_price * $this->quantity;
    }

    public function getDiscountUnitPriceExtension() {
        return $this->getLinePrice() - $this->getTotalDiscount();
    }

    public function getTotalDiscount() {
        $totalDiscount = 0;
        foreach ($this->stagedOrderDiscLines as $discLine) {
            $totalDiscount += $discLine->discount_amount;
        }
        return $totalDiscount;
    }

    public function discLines(): HasMany {
        return $this->hasMany(StagedOrderDiscLine::class, "stagedorderline_id", "id");
    }

    /**
     * @return BelongsTo
     */
    public function part(): BelongsTo {
        return $this->belongsTo(Part::class, 'part_id', 'id');
    }

    /**
     * @return BelongsTo
     */
    public function salesOrderLine(): BelongsTo {
        return $this->belongsTo(SalesOrderLine::class, "salesorderline_id", "id");
    }

    public function shipFromPlant(): BelongsTo {
        return $this->belongsTo(Plant::class, "plant_id", "id");
    }

    public function shipPlant(): BelongsTo {
        return $this->belongsTo(Plant::class, "plant_id", "id");
    }

    /**
     * @return HasMany
     */
    public function stagedOrderDiscLines(): HasMany {
        return $this->hasMany(StagedOrderDiscLine::class, "stagedorderline_id", "id");
    }

    /**
     * @return BelongsTo
     */
    public function stagedOrder(): BelongsTo {
        return $this->belongsTo(StagedOrder::class, "stagedorder_id", "id");
    }
}