<?php

namespace Hilco;

use Debugbar;
use Hilco\Models\AustraliaFreightPostCode;
use Hilco\Models\Carrier;
use Hilco\Models\Customer;
use Hilco\Models\CustomerCreditProfile;
use Hilco\Models\Part;
use Hilco\Models\Plant;
use Hilco\Models\Promotion;
use Hilco\Models\PromotionCodeTrigger;
use Hilco\Models\PromotionTrigger;
use Hilco\Models\RewardsProductCategoryGroupDiscount;
use Hilco\Models\Translations_WebCollection;
use Hilco\Models\Translations_WebFamily;
use Hilco\Models\Translations_WebGroup;
use Hilco\Models\VATTax;
use Hilco\Models\WebCart;
use Hilco\Models\WebCategory;
use Hilco\Models\WebCollection;
use Hilco\Models\WebFamily;
use Hilco\Models\WebFeedbackResponse;
use Hilco\Models\WebGroup;
use Hilco\Models\WebPart;
use Hilco\Models\WebRole;
use Hilco\Models\WebSilo;
use Hilco\Models\WebUrl;
use Hilco\Shipments\Rate;
use HilcoB2B\AuthorizeNetWrapper;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Hilco\Facades\RateShop;

class B2BHelper {
    protected $activeCustomer;
    protected $activeWebSilo;
    protected $activeShippingAddress;
    protected $activeBillingCustomer;
    protected $activeCarrier;
    protected $activeRate;
    protected $activePlant;
    protected $activePlants;
    protected $activeLocale;
    protected $activeLanguage;
    protected $activePromoCode;
    protected $rewardsProductCategoryGroupDiscounts;
    protected $activeWebGroupsForLayout;
    protected $activeWebTranslations;

    public function __construct() {
        $this->activeShippingAddress = null;
        $this->activeBillingCustomer = null;
        $this->activeCarrier = null;
        $this->activeRate = null;
        $this->activePlant = null;
        $this->activePlants = null;
        $this->activePromoCode = null;
        $this->activeWebGroupsForLayout = null;
        $this->activeWebTranslations = null;

        $this->loadActiveCustomer();
        if ($this->activeCustomer) {
            $this->loadActiveBillingCustomer();
            $this->loadActiveShippingAddress();
            $this->loadActiveCarrier();
            $this->loadActiveRate();
        }

        $this->loadActiveLocale();
        $this->loadActiveLanguage();
        $this->loadActiveWebSilo();
        $this->loadActivePlants();
        $this->loadRewardsProductCategoryGroupDiscounts();
        if($this->activeLanguage != "en"){
            $this->loadActiveWebTranslations();
        }
    }

//  ******************************** LOAD ********************************
    protected function loadActiveCustomer() {
        $this->activeCustomer = false;

        $aliasedCustomerId = session()->get('b2b.aliasedCustomerId', false);
        if ($aliasedCustomerId) {
            $this->activeCustomer = Customer::with('segments.address.plant', 'discounts', 'availableShippingAddresses', 'webSilo')->find($aliasedCustomerId);
        } else {
            $user = auth()->user();
            if ($user) $this->activeCustomer = $user->customer;
        }
    }

    protected function loadActiveBillingCustomer() {
        $this->activeBillingCustomer = false;

        $activeBillingCustomerId = session()->get('b2b.activeBillingCustomerId', false);
        if ($activeBillingCustomerId) {
            $this->activeBillingCustomer = Customer::with('segments.address.plant', 'discounts', 'availableShippingAddresses')->find($activeBillingCustomerId);
        } else {
            $this->activeBillingCustomer = array_get($this->activeCustomer, 'billToCustomer', false);
        }
    }

    protected function loadActiveShippingAddress() {
        $this->activeShippingAddress = $this->activeCustomer->default_shipping_address;

        $activeShippingAddressId = session()->get('b2b.activeShippingAddressId', false);
        if ($activeShippingAddressId) {
            foreach ($this->activeCustomer->availableShippingAddresses as $address) {
                if ($address->id == $activeShippingAddressId) {
                    $this->activeShippingAddress = $address;
                    break;
                }
            }
        }
    }

    protected function loadActiveCarrier() {
        $this->activeCarrier = array_get($this->activeShippingAddress, 'defaultCarrier', false);

        $activeCarrierId = session()->get('b2b.activeCarrierId', false);
        if ($activeCarrierId) {
            $this->activeCarrier = Carrier::find($activeCarrierId);
        }
    }

    protected function loadActiveRate() {
        $this->activeRate = session()->get('b2b.activeRate', false);
    }

    protected function loadActiveWebSilo() {
        $this->activeWebSilo = false;
        $user = auth()->user();

        $activeWebSiloId = !is_null($user) ? session()->get('b2b.activeWebSiloId', false) : false;

        if (!$activeWebSiloId) {
            if ($user) {
                if ($this->activeCustomer) {
                    $defaultWebSilo = $this->activeCustomer->webSilo;
                    if ($defaultWebSilo && $defaultWebSilo->is_visible) {
                        $activeWebSiloId = $defaultWebSilo->id;
                    }
                }

                if (!$activeWebSiloId) {
                    $userSilo = $user->defaultWebSilo;
                    if ($userSilo) {
                        $activeWebSiloId = $userSilo->id;
                    }
                }
            }
            if (!$activeWebSiloId) {
                $query = WebSilo::with('webLandingPageWebSiloJoins');

                if ($this->activeCustomer) {
                    $cust_cat_summary = $this->activeCustomer->customerCategory->cust_cat_summary;
                    $divisions = $this->activeCustomer->divisions->pluck('id');
                    $query->whereHas('divisionRules', function ($query) use ($divisions) {
                        $query->whereIn('Divisions.id', $divisions);
                    })->whereHas('customerCategorySummaryRules', function ($query) use ($cust_cat_summary) {
                        $query->where('cust_cat_summary', $cust_cat_summary);
                    });
                } else {
                    $webUrl = WebUrl::current()->first();
                    if (!is_null($webUrl)) {
                        $query->whereHas('webUrls', function ($query) use ($webUrl) {
                            $query->where('id', '=', $webUrl->id);
                        });
                    }
                }
                $result = $query->first();
                if (!is_null($result)) $activeWebSiloId = $result->id;
            }

            if (!$activeWebSiloId) $activeWebSiloId = WebSilo::first()->id;
        }
        $this->activeWebSilo = WebSilo::with('webLandingPageWebSiloJoins')->find($activeWebSiloId);
        session()->put('b2b.activeWebSiloId', $activeWebSiloId);

        return $this->activeWebSilo;
    }

    public function loadActiveLanguage() {
        $this->activeLanguage = false;

        $activeLanguage = session()->get('b2b.activeLanguage', false);
        if ($activeLanguage) {
            $this->activeLanguage = $activeLanguage;
        } else {
            switch ($this->activeCountry()) {
                case 'GB':
                    $this->activeLanguage = 'en_GB';
                    break;
                default:
                    $this->activeLanguage = config('app.locale', 'en');
                    break;
            }
        }
        if (!in_array($this->activeLanguage, ['en', 'fr', 'ts'])) $this->activeLanguage = 'en';
        $oldLanguage = App::getLocale();

        if ($oldLanguage != $this->activeLanguage) {
            Debugbar::addMessage("Setting language from $oldLanguage to {$this->activeLanguage}");
            App::setLocale($this->activeLanguage);
        }
    }

    public function loadActiveWebTranslations(){
        $this->activeWebTranslations = false;
        // Fetch the WebGroups.

        // For some reason I kept getting "unable to process request" issues when I was initially calling down the web silo.
        $webGroupTranslations = Translations_WebGroup::where('language', '=', $this->activeLanguage)->get();
        $webCollectionTranslations = Translations_WebCollection::where('language', '=', $this->activeLanguage)->get();
        $activeWebTranslations = ['webCollectionTranslations' => $webCollectionTranslations];
        $this->activeWebTranslations = $activeWebTranslations;
    }

    public function loadActiveLocale() {
        $this->activeLocale = false;

        $activeLocale = session()->get('b2b.activeLocale', false);
        if ($activeLocale) {
            $this->activeLocale = $activeLocale;
        } else {
            switch ($this->activeCountry()) {
                case 'GB':
                    $this->activeLocale = 'en_GB';
                    break;
                default:
                    $this->activeLocale = $this->defaultLocale();
                    break;
            }
        }
        Debugbar::addMessage("Setting locale to {$this->activeLocale}");
    }

    public function loadActivePlants() {
        $this->activePlants = false;
//        $this->activePlant = false;

        $activePlantsIds = session()->get('b2b.activePlantsIds', false);

        if ($activePlantsIds) {
            $this->activePlants = Plant::findMany($activePlantsIds);
        } else {
            $activePlantNames = [];

            if ($this->activeWebSilo()->id == config('hilco.hilcoOutdoorWebSiloId')) {
                $activePlantNames = ['PLAINVILLE', 'BOZEMAN', 'MONTREAL'];
            } else {
                $activeSegment = $this->activeSegment();
                $customerPlantName = array_get($activeSegment, 'address.plant.plant', array_get(Plant::defaultPlant(), 'plant'));
                $activePlantNames[] = $customerPlantName;
                // Extra check to make sure Plainville is added to the active plants list because Montreal is technically shut down
                if ($customerPlantName == 'MONTREAL') {
                    $activePlantNames[] = 'PLAINVILLE';
                }

                // Extra check because the Kaiser microsite technically has some stock in Dallas
                // use 'strpos() !== false' because strpos() returns offset str position if found (so 0 is technically valid) and false if not found at all
                if (strpos($this->activeWebSilo()->slug, 'kaiser') !== false) {
                    $activePlantNames[] = 'DALLAS';
                }
            }
            $this->activePlants = Plant::whereIn('plant', $activePlantNames)->get();
        }

        $activePlantsIds = [];
        foreach ($this->activePlants as $plant) {
            $activePlantsIds[] = $plant->id;
        }
        session()->put('b2b.activePlantsIds', $activePlantsIds);
    }

    public function loadRewardsProductCategoryGroupDiscounts(){
        $rewardsProductCategoryGroupDiscounts = session()->get('b2b.rewardsProductCategoryGroupDiscounts', false);
        $rewardsProductCategoryGroupDiscounts = RewardsProductCategoryGroupDiscount::all();
        $this->rewardsProductCategoryGroupDiscounts = $rewardsProductCategoryGroupDiscounts;
    }
//  ******************************** LOAD ********************************


//  ******************************** SETTERS ********************************
    public function setActiveShippingAddress($activeShippingAddressId = false) {
        if ($activeShippingAddressId) {
            session()->put('b2b.activeShippingAddressId', $activeShippingAddressId);
        } else {
            session()->forget('b2b.activeShippingAddressId');
        }
        $this->loadActiveShippingAddress();
    }

    public function setActiveBillingCustomer($billingCustomerId = false) {
        if ($billingCustomerId) {
            session()->put('b2b.activeBillingCustomerId', $billingCustomerId);
        } else {
            session()->forget('b2b.activeBillingCustomerId');
        }
        $this->loadActiveBillingCustomer();
    }

    public function setActiveCarrier($activeCarrierId = false) {
        if ($activeCarrierId) {
            session()->put('b2b.activeCarrierId', $activeCarrierId);
        } else {
            session()->forget('b2b.activeCarrierId');
        }
        $this->loadActiveCarrier();
    }

    public function setActiveRate($activeRate = false) {
        if ($activeRate) {
            session()->put('b2b.activeRate', $activeRate);
        } else {
            session()->forget('b2b.activeRate');
        }
        $this->activeRate = $activeRate;
    }

    public function setActiveLanguage($language) {
        session()->put('b2b.activeLanguage', $language);
        $this->loadActiveLanguage();
    }

    public function setActiveLocale($locale) {
        session()->put('b2b.activeLocale', $locale);
        $this->loadActiveLocale();
    }

    public function setActivePromoCode($promoCode) {
        session()->put('b2b.activePromoCode', $promoCode);
        $this->activePromoCode = $promoCode;
    }

    public function setRewardsProductCategoryGroupDiscounts($rewardsProductCategoryGroupDiscounts){
        session()->put('b2b.rewardsProductCategoryGroupDiscounts', $rewardsProductCategoryGroupDiscounts);
        $this->rewardsProductCategoryGroupDiscounts = $rewardsProductCategoryGroupDiscounts;
    }
//  ******************************** SETTERS ********************************


//  ******************************** GETTERS ********************************
    public function activeCustomer() {
        return $this->activeCustomer;
    }

    public function activeBillingCustomer() {
        return $this->activeBillingCustomer;
    }

    public function activeShippingAddress() {
        return $this->activeShippingAddress;
    }

    public function activeCarrier() {
        return $this->activeCarrier;
    }

    /**
     * @return Rate
     */
    public function activeRate() {
        return $this->activeRate;
    }

    public function activeCountry() {
        if ($this->activeShippingAddress) return $this->activeShippingAddress()->country;

        return null;
    }

    public function activePriceList() {
        $activePriceList = 'Catalog';
        if ($this->activeCustomer) {
            $customerPriceList = $this->activeCustomer->default_price_list;
            if ($customerPriceList) $activePriceList = $customerPriceList;
        }

        return $activePriceList;
    }

    public function activeCurrency() {
        $activeCurrency = 'USD';

        if ($this->activeCustomer) {
            $activeCurrency = $this->activeCustomer->currency;
        }

        return $activeCurrency;
    }

    public function activeWebHierarchy() {
        return $this->activeWebSilo()->webHierarchy;
    }

    /** @return WebSilo */
    public function activeWebSilo() {
        return $this->activeWebSilo;
    }

    public function activeLocale() {
        return $this->activeLocale;
    }

    public function activeLanguage() {
        return $this->activeLanguage;
    }

    public function activeSegment() {
        $activeCustomer = $this->activeCustomer();

        return array_get($activeCustomer, 'activeSegment');
    }

    public function activePlant() {
        $activePlant = null;
        $activeSegment = $this->activeSegment();

        $activePlant = array_get($activeSegment, 'address.plant', Plant::defaultPlant());

        return $activePlant;
    }

    public function activePlants() {
        return $this->activePlants;
//        if (b2b()->activeWebSilo()->id == config('hilco.hilcoOutdoorWebSiloId')) {
//            $activePlants = Plant::whereIn('plant', ['PLAINVILLE', 'BOZEMAN'])->get();
//        } else {
//            $activePlants = collect([b2b()->activePlant()]);
//        }
//        return $activePlants;
    }

    public function activePlantNames() {
        $plantNames = [];
        foreach ($this->activePlants as $activePlant) {
            $plantNames[] = $activePlant->plant;
        }
        return $plantNames;
    }

    public function activePromoCode() {
        return $this->activePromoCode;
    }
    public function rewardsProductCategoryGroupDiscounts()
    {
        return $this->rewardsProductCategoryGroupDiscounts;
    }
    public function activeWebTranslations(){
        return $this->activeWebTranslations;
    }
//  ******************************** GETTERS ********************************


    public function aliasAs($customerId = false) {
        session()->forget('b2b');

        if ($customerId) {
            session()->put('b2b.aliasedCustomerId', $customerId);
        }

        $this->loadActiveCustomer();
    }

    public function isAliased() {
        return (session()->get('b2b.aliasedCustomerId', false));

    }

    public function stopAliasing() {
        $this->aliasAs(false);
    }

    public function currencySymbol($currencyCode = false) {
        if ($currencyCode === false) $currencyCode = $this->activeCurrency();
        if ($currencyCode == 'GBP') return '£';

        return '$';
    }

    public function formatPrice($price, $decimals = 2, $symbol = false) {
        if ($symbol === false) $symbol = $this->currencySymbol();
        if ($decimals == -1) {
            if ($price == round($price)) $decimals = 0;
            else $decimals = 2;
        }
        return $symbol . number_format($price, $decimals);
    }

    public function dateFormatString($division = false) {
        if ($division === false) $division = $this->activeCustomer()->activeSegment->division;

        $format = 'm/d/Y';
        switch ($division) {
            case 'HILCO-UK':
                $format = 'd/m/Y';
                break;
            case 'HILCO-US':
            case 'WILSON-US':
                $format = 'm/d/Y';
        }

        return $format;
    }

    public function isOrderable(Model $model) {
        if ($model instanceof WebFamily) return $this->isWebFamilyOrderable($model);
        else if ($model instanceof WebPart) return $this->isWebPartOrderable($model);
        else if ($model instanceof Part) return $this->isPartOrderable($model);

        return false;
    }

    public function isWebFamilyOrderable(WebFamily $webFamily) {
        $isOrderable = true;
        foreach ($webFamily->visibleWebParts as $webPart) {
            if (!$this->isWebPartOrderable($webPart)) $isOrderable = false;
        }
        return $isOrderable;
    }

    public function isWebPartOrderable(WebPart $webPart) {
        $part = $webPart->part;
        if (is_null($part)) return false;
        return $this->isPartOrderable($part);
    }

    public function isPartOrderable(Part $part) {
        if ($this->activeCountry() != 'US' && $this->activeCountry() != 'GB') {
            if ($part->isRX) return false;
        }
        return true;
    }

    public function allowCreditCardOrders() {
        return (bool)
            !$this->activeRate()->isError() &&
            $this->activeWebSilo->allow_credit_cards &&
            $this->isBillingDirect() &&
            strlen(array_get($this->authorizeNetCredentials(), 'clientKey', ''));
    }

    public function allowDirectBilling() {
        if (!$this->isBillingDirect() && !$this->allowBillingChange()) {
            return false;
        }

        return (bool) $this->activeWebSilo->allow_direct_billing;
    }

    public function allowGroupBilling() {
        if ($this->isBillingDirect() && !$this->allowBillingChange()) {
            return false;
        }
        return (bool) $this->activeWebSilo->allow_group_billing;
    }

    public function allowBillingChange() {
        if ($this->activeCountry() == 'GB') return false;
        else return true;
    }

    public function requireApprovalSiloParts() {
        return (bool) array_get($this->activeCustomer,'webSilo.pivot.require_approval_silo_parts', false);
    }

    public function requireApprovalHilcoParts() {
        return (bool) array_get($this->activeCustomer,'webSilo.pivot.require_approval_hilco_parts', false);
    }

    public function allowPOOrders() {
        $terms = array_get($this->activeBillingCustomer, 'activeSegment.def_pmt_terms');
        return !in_array($terms, ['CR CARD', 'CR CARD+BAL']);
    }

    public function isPONumberRequired() {
        if ($this->activeBillingCustomer->activeSegment->division == 'HILCO-US') return false;

        return true;
    }

    public function authorizeNetCredentials() {
        $clientKey = '';
        $apiLoginID = '';

        try {
            $credentials = AuthorizeNetWrapper::getCredentials($this->activeBillingCustomer->activeSegment->division, $this->activeBillingCustomer->currency);
            $clientKey = $credentials['CLIENT_KEY'];
            $apiLoginID = $credentials['API_LOGIN'];
        } catch (\AuthorizeNetException $e) {

        }

        return ['clientKey' => $clientKey, 'apiLoginId' => $apiLoginID];
    }

    public function creditProfiles() {
        $creditProfiles = [];
        try {
            $authorizeNet = AuthorizeNetWrapper::create($this->activeBillingCustomer->activeSegment->division, $this->activeBillingCustomer->currency);
            $customerCreditProfile = CustomerCreditProfile::where('customer_id', $this->activeCustomer->id)
                ->whereNotNull('profileId')
                ->whereRaw('LENGTH(`profileId`) > 0')
                ->where('deleted_at', '0000-00-00 00:00:00')
                ->first();
            if ($customerCreditProfile) {
                foreach ($authorizeNet->getCreditProfiles($authorizeNet->getCustomerProfile($customerCreditProfile->profileId)) as $creditProfile) {
                    $creditProfiles[] = [
                        'id' => $creditProfile->getCustomerPaymentProfileId(),
                        'text' => $creditProfile->getPayment()->getCreditCard()->getCardNumber(). ' - ' .$creditProfile->getPayment()->getCreditCard()->getExpirationDate(),
                        'last4' => substr($creditProfile->getPayment()->getCreditCard()->getCardNumber(), -4),
                        'expDate' => $creditProfile->getPayment()->getCreditCard()->getExpirationDate(),
                        'cardType' => $creditProfile->getPayment()->getCreditCard()->getCardType(),
                    ];
                }
                $creditProfiles[] = ['id' => 0, 'text' => 'Use new Card'];
            }
        } catch (\AuthorizeNetException $e) {

        }

        return $creditProfiles;
    }

    public function isRewardsEnabled() {
        if (!config('rewards.rewardsEnabled')) return false;
        return (bool) $this->activeWebSilo->allow_rewards;
    }

    public function isBillingDirect() {
        return $this->activeCustomer->id == $this->activeBillingCustomer->id;
    }

    public function isCustomerTaxable() {
        return array_get($this->activeBillingCustomer, 'activeSegment.cust_txbl', false);
    }

    public function availableShippingAddresses() {
        $defaultAddress = $this->activeCustomer->default_shipping_address;
        $addresses = collect([$defaultAddress]);
        if ($this->activeWebSilo()->allow_drop_shipping) {
            $addresses = $this->activeCustomer->availableShippingAddresses->sort(function ($a, $b) use ($defaultAddress) {
                if ($a->id == $defaultAddress->id) return -1;
                if ($b->id == $defaultAddress->id) return 1;

                return strcmp($a->cust_name, $b->cust_name);
            });
        }

        return $addresses;
    }

    public function isValidPromoCode($promoCode) {
        // does promo code even belong to an existing codeTrigger
        $codeTriggers = PromotionCodeTrigger::where('code', '=', $promoCode)->where('deleted_at', '=', '0000-00-00 00:00:00')->get();
        if($codeTriggers == null){
            return false;
        }
        /**
         * Original Case: We pull up a list of codetriggers, and just pull from the first.
         * Now, we need to conisder ALL of the CodeTriggers that have this code.
         * To do this successfully, we need to evaluate any and all promotions assosciated with this code.
         * If at least one of them are valid, let it go through.
         * Otherwise, return invalid.
         */

        foreach($codeTriggers as $codeTrigger) {
            $promotion = $codeTrigger->promotion()->first();
            // Is this an active Promo?
            if (!$this->existsInActivePromos($promotion)) {
                continue;
            }
            $promotionIsValid = true;
            foreach($promotion->triggers as $trigger) {
                if ($trigger->trigger_type != 'codeTrigger') {
                    // Is this triggered?
                    if (!$trigger->details->isTriggered($this->activeCustomer, WebCart::getWebCartItems(auth()->user()))) {
                        // Then mark it off. This promo isn't acceptable,
                        // Unless of course it has an Almost Qualifying Field.
                        if($trigger->details->hasAlmostQualifyingField()){
                            continue;
                        }
                        $promotionIsValid = false;
                    }
                }
            }
            if($promotionIsValid){
                return true;
            }

        }

        return false;
    }

    // TODO: Come up with a better name for this, probably.
    // Checks for promotions that are code triggered, but still have un-triggered triggers (ugh)
    // This requires a seperate display for the checkout controller to indicate this when the code is input.
    // We could possible expand this to display the kind of promotion it is and the specifics if needed.
    public function nonTriggeredCodeTriggeredPromotions($customerTriggeredPromotions, $triggeredPromotions){
        $nonTriggeredCodeTriggeredPromotions = [];
        $billingCustomer = b2b()->activeBillingCustomer;
        $webCartItems = WebCart::getWebCartItems(auth()->user());
        
        foreach($customerTriggeredPromotions as $customerTriggeredPromotion){
            $codeTrigger = $customerTriggeredPromotion->codeTriggers()->first();
            if(is_null($codeTrigger)){
                continue;
            }
            foreach($triggeredPromotions as $triggeredPromotion){
                if($customerTriggeredPromotion->id === $triggeredPromotion->id){
                    continue 2;
                }
            }
            // Compatible for the event that we ever let them have multiple promo codes. Probably.
            if($codeTrigger->isTriggered($billingCustomer, $webCartItems)){
                $nonTriggeredCodeTriggeredPromotions[$codeTrigger->code] = $customerTriggeredPromotion;
            }

        }
        return $nonTriggeredCodeTriggeredPromotions;

    }

    private function existsInActivePromos($promotion) {
        foreach (Promotion::getActivePromotions() as $activePromotion) {
            if ($activePromotion->id === $promotion->id) return true;
        }
        return false;
    }

//  ******************************** RATE SHOP ********************************
    /**
     * @return Collection|Rate[]
     */
    public function rateShop() {
        if ($this->activeWebSilo()->allow_rate_shopping) {
            $oldRates = $this->oldRateShop();
        } else {
            $oldRates = [];
        }


        $siloCodesString = $this->activeWebSilo()->allowed_carrier_codes;
        $siloCodes = [];
        if (strlen($siloCodesString)) {
            $siloCodes = explode(',', $siloCodesString);
            foreach ($oldRates as $index => $oldRate) {
                if (!in_array(array_get($oldRate, 'hilco_carrier_code'), $siloCodes)) {
                    unset($oldRates[$index]);
                }
            }
        }



        $carrierCodes = [];
        foreach ($oldRates as $oldRate) {
            $code = array_get($oldRate, 'hilco_carrier_code', false);
            if ($code) {
                if (!count($siloCodes) || in_array($code, $siloCodes)) {
                    $carrierCodes[$code] = $code;
                }
            }
        }
        $carriers = Carrier::codes($carrierCodes)->get()->keyBy('carrier_code');

        $activeRate = $this->activeRate();
        $activeRateCode = ($activeRate) ? $activeRate->code() : false;
        $wasRateSelected = false;

        $rates = new Collection();
        foreach ($oldRates as $oldRate) {
            $code = array_get($oldRate, 'hilco_carrier_code', false);
            if ($code === false) continue;
            $carrier = array_get($carriers, $code);
            $rate = array_get($oldRate, 'rate', 0);
            $isDefault = array_get($oldRate, 'is_default', false);
            $discount = array_get($oldRate, 'discount_amount', 0);
            $tax = array_get($oldRate, 'tax_amount', 0);

            $rate = new Rate($carrier, $rate, $isDefault, $discount, $tax);
            $rate->setText(array_get($oldRate, 'carrier_desc'));

            if ($code == $activeRateCode) {
                $rate->selected = true;
                $wasRateSelected = true;
            }
            $rates->put($code, $rate);
        }

        $rates = $rates->sort(function (Rate $a, Rate $b) {
            $aSpecial = $a->isDefault() || $a->isDiscounted();
            $bSpecial = $b->isDefault() || $b->isDiscounted();

            if ($aSpecial && !$bSpecial) return -1; //a is special but b isn't, a goes first
            if (!$aSpecial && $bSpecial) return 1; //b is special but a isn't, b goes first

            //both or neither are special
            if ($a->rate() > $b->rate()) return 1;
            if ($a->rate() < $b->rate()) return -1;

            return 0;
        });

        if ($rates->count() == 0) {
            $rates->put('-1', new Rate(null, -1, true));
        }

        if ($activeRateCode === false || $wasRateSelected === false) {
            $firstRate = $rates->shift();
            $firstRate->selected = true;
            $this->setActiveRate($firstRate);
            $rates->prepend($firstRate, $firstRate->code());
        }

        return $rates;
    }

    protected function rateShopGroupCode() {
        $shippingAddress = $this->activeShippingAddress;
        $countryString = array_get($shippingAddress, 'country');
        return $countryString != null && $countryString == "CA" ? 2 : 3;
    }

    protected function getFromCode($shippingAddressObj, $groupCode) {
        if ($groupCode === 2) {
            return array_get($shippingAddressObj, 'plant.plantAddress.canada_zip');
        } else {
            return array_get($shippingAddressObj, 'plant.plantAddress.zip');
        }
    }

    public function allInStock($cartItems, $plant){
        foreach ($cartItems as $cartItem){
            $part = $cartItem['part'];
            $inventoryItem = $part->inventoryItems->where('plant', $plant)->first();
            if(!isset($inventoryItem) || (isset($inventoryItem) && (!$inventoryItem->inStock || $cartItem['quantity'] > $inventoryItem->net_avail))){
                return false;
            }
        }
        return true;
    }

    public function allStockable($cartItems, $plant){
        foreach ($cartItems as $cartItem) {
            $part = $cartItem['part'];
            $inventoryItem = $part->inventoryItems->where('plant', $plant)->first();
            if (!isset($inventoryItem)){
                return false;
            }
        }
        return true;
    }

    protected function oldRateShop() {
        $shippingAddress = $this->activeShippingAddress;
        $countryString = array_get($shippingAddress, 'country');
        $groupCode = $this->rateShopGroupCode();
        $fromCode = $this->getFromCode($shippingAddress, $groupCode);//array_get($shippingAddress, 'plant.plantAddress.zip');
        $defaultCarrierCode = array_get($shippingAddress, 'def_carr_cd');

        $cartStatus = App::make('cartSingleton')->getCartStatus();

        $cartItems = $cartStatus['items'];
        $excludedItems = [];
        $rewardsItems = [];
        $cartTotal = 0;
        $weights = [];
        $excludedWeights = [];
        $rewardsWeights = [];
        $volume = 0;
        $excludedVolume = 0;
        $rewardsVolume = 0;
        $hideAir = false;
        foreach ($cartItems as $cartItem) {
            $quantity = $cartItem['quantity'];
            $part = $cartItem['part'];
            if($part->isRewardsExcluded()){
                $excludedItems[$cartItem['id']] = $cartItem;
            }
            else{
                $rewardsItems[$cartItem['id']] = $cartItem;
            }

            $key = 0;
            if (array_get($shippingAddress, 'def_ship_from') == 'MONTREAL') {
                $inventoryItem = $part->inventoryItems->where('plant','MONTREAL')->first();
                if (isset($inventoryItem) && $inventoryItem->inStock) {
                    $key = 1;
                }
            }

            if(in_array($cartItem, $excludedItems)){
                if(!isset($excludedWeights[$key])){
                    $excludedWeights[$key] = $part->weight_lbs * $quantity;
                }
                else{
                    $excludedWeights[$key] += $part->weight_lbs * $quantity;
                }
            }
            else{
                if (!isset($rewardsWeights[$key])){
                    $rewardsWeights[$key] = $part->weight_lbs * $quantity;
                }
                else{
                    $rewardsWeights[$key] += $part->weight_lbs * $quantity;
                }
            }

            if (!isset($weights[$key])) {
                $weights[$key] = $part->weight_lbs * $quantity;
            } else {
                $weights[$key] += $part->weight_lbs * $quantity;
            }

            if(in_array($cartItem, $excludedItems)){
                $excludedVolume += $part->vol_ci * $quantity;
            }
            else {
                $rewardsVolume += $part->vol_ci * $quantity;
            }
            $volume += $part->vol_ci * $quantity;
            $hideAir = $hideAir || ($part->inventoryItems->where('plant_id', $shippingAddress->default_shipfrom_id)->count() > 0 && $part->inventoryItems->where('plant_id', $shippingAddress->default_shipfrom_id)->first()->hazmat_code === 'ORM-D');
            $cartTotal += $cartItem['extendedPrice'];
        }

        if (count($weights) > 0) {
            if ($shippingAddress->plant->plant == 'AUSTRALIA') {
                /* AUS REQUESTED NO RATE SHOPPING */
                return [['id' => -1, 'text' => trans('hilco::app.unable_to_calculate_shipping_rates')]];
            } else if ($countryString == 'GB') {
                /* UK REQUESTED NO RATE SHOPPING */
                return [['id' => -1, 'text' => trans('hilco::app.unable_to_calculate_shipping_rates')]];
            } else if ($groupCode == 3) {
                $carrierCodes = DB::connection()
                    ->table('RateShopGroups')
                    ->where('group_code', '=', $groupCode)
                    ->pluck('hilco_carrier_code');

                $isDefaultNewRow = false;
                $defaultCarrier = null;
                if ($shippingAddress->def_carr_cd != null) {
                    if (!in_array($shippingAddress->def_carr_cd, $carrierCodes)) {
                        $defaultCarrier = DB::connection()
                            ->table('Carriers')
                            ->leftJoin('RateShopAPIDetails', 'Carriers.carrier_code', '=', 'RateShopAPIDetails.hilco_carrier_code')
                            ->where('carrier_code', '=', $shippingAddress->def_carr_cd)
                            ->select()
                            ->first();

                        // Yes this is intentional. No it isn't our data. Yes if they decide to fix it without telling us it would be bad.
                        if ($defaultCarrier->bill == 'Hico' || $defaultCarrier->bill == 'Hilco') {
                            $carrierCodes[] = $shippingAddress->def_carr_cd;
                        } else {
                            $isDefaultNewRow = true;
                        }
                    }
                }

                if(($shippingAddress->def_carr_cd == '1413' || $shippingAddress->def_carr_cd == '1414' || $shippingAddress->def_carr_cd == '1415')
                    && !in_array($shippingAddress->def_carr_cd, $carrierCodes)){
                    array_push($carrierCodes, $shippingAddress->def_carr_cd);
                }

                if ($shippingAddress->def_carr_cd != null && ($shippingAddress->def_carr_cd == '1400' || (isset($defaultCarrier->super_code) && $defaultCarrier->super_code == '1400'))) {
                    if (($key = array_search('1000', $carrierCodes)) !== false) {
                        unset($carrierCodes[$key]);
                    }
                } else {
                    if (($key = array_search('1400', $carrierCodes)) !== false) {
                        unset($carrierCodes[$key]);
                    }
                }

                if ($hideAir) {
                    $airCodes = DB::connection()
                        ->table('RateShopAPIDetails')
                        ->whereIn('hilco_carrier_code', $carrierCodes)
                        ->where('is_air', '=', true)
                        ->select('hilco_carrier_code')
                        ->get();

                    foreach ($airCodes as $airCode) {
                        if (($key = array_search($airCode->hilco_carrier_code, $carrierCodes)) !== false) {
                            unset($carrierCodes[$key]);
                        }
                    }
                }

                try {
                    if(count($excludedItems)){
                        $excludedShipVia = RateShop::rateShopUSAWithAccount($carrierCodes, $excludedWeights[0], $excludedVolume, $shippingAddress->state, $shippingAddress->postal_cd, $fromCode);
                    }
                    if(count($rewardsItems)){
                        $rewardsShipVia = RateShop::rateShopUSAWithAccount($carrierCodes, $rewardsWeights[0], $rewardsVolume, $shippingAddress->state, $shippingAddress->postal_cd, $fromCode);
                    }
                    $shipVia = RateShop::rateShopUSAWithAccount($carrierCodes, $weights[0], $volume, $shippingAddress->state, $shippingAddress->postal_cd, $fromCode);
                } catch (\Exception $e) {
                    Log::critical('Exception caught while getting shipping rates: ' . $e->getMessage(), $e->getTrace());
                }

                if ($isDefaultNewRow) {
                    if (isset($defaultCarrier->super_code) && isset($defaultCarrier->code_type)) {
                        if ($shippingAddress->def_ship_via != null && strlen($shippingAddress->def_ship_via) > 0) {
                            $codes = DB::connection()
                                ->table('RateShopAPIDetails as myCode')
                                ->join('RateShopAPIDetails as otherCode', 'myCode.code_type', '=', 'otherCode.code_type')
                                ->join('Carriers', 'Carriers.carrier_code', '=', 'otherCode.hilco_carrier_code')
                                ->where('myCode.hilco_carrier_code', '=', $shippingAddress->def_carr_cd)
                                ->whereIn('otherCode.super_code', $carrierCodes)
                                ->select(['otherCode.hilco_carrier_code', 'Carriers.carrier_desc', 'otherCode.super_code'])
                                ->get();

                            foreach ($codes as $code) {
                                if(isset($excludedShipVia) and count($excludedShipVia)){
                                    $excludedShipVia[$code->hilco_carrier_code] = [
                                        'api_carrier' => null,
                                        'residential' => null,
                                        'hilco_carrier_code' => $code->hilco_carrier_code,
                                        'carrier_desc' => $code->carrier_desc,
                                        'rate' => 2,
                                        'text' => $code->carrier_desc . ' - ' . $this->formatPrice(2),
                                        'service' => null,
                                        'delivery_date' => null,
                                        'delivery_time' => null,
                                        'guaranteed' => null,
                                        'def_ship_via' => $shippingAddress->def_ship_via,
                                        'is_default' => $code->hilco_carrier_code == $defaultCarrierCode,
                                    ];
                                }
                                if(isset($rewardsShipVia) and count($rewardsShipVia)){
                                    $rewardsShipVia[$code->hilco_carrier_code] = [
                                        'api_carrier' => null,
                                        'residential' => null,
                                        'hilco_carrier_code' => $code->hilco_carrier_code,
                                        'carrier_desc' => $code->carrier_desc,
                                        'rate' => 2,
                                        'text' => $code->carrier_desc . ' - ' . $this->formatPrice(2),
                                        'service' => null,
                                        'delivery_date' => null,
                                        'delivery_time' => null,
                                        'guaranteed' => null,
                                        'def_ship_via' => $shippingAddress->def_ship_via,
                                        'is_default' => $code->hilco_carrier_code == $defaultCarrierCode,
                                    ];
                                }
                                $shipVia[$code->hilco_carrier_code] = [
                                    'api_carrier' => null,
                                    'residential' => null,
                                    'hilco_carrier_code' => $code->hilco_carrier_code,
                                    'carrier_desc' => $code->carrier_desc,
                                    'rate' => 2,
                                    'text' => $code->carrier_desc . ' - ' . $this->formatPrice(2),
                                    'service' => null,
                                    'delivery_date' => null,
                                    'delivery_time' => null,
                                    'guaranteed' => null,
                                    'def_ship_via' => $shippingAddress->def_ship_via,
                                    'is_default' => $code->hilco_carrier_code == $defaultCarrierCode,
                                ];
                            }
                        }
                    } else {
                        if(isset($excludedShipVia) and count($excludedShipVia)){
                            $excludedShipVia[$shippingAddress->def_carr_cd] = [
                                'api_carrier' => null,
                                'residential' => null,
                                'hilco_carrier_code' => $shippingAddress->def_carr_cd,
                                'carrier_desc' => $defaultCarrier->carrier_desc,
                                'rate' => 2,
                                'text' => $defaultCarrier->carrier_desc . ' - ' . $this->formatPrice(2),
                                'service' => null,
                                'delivery_date' => null,
                                'delivery_time' => null,
                                'guaranteed' => null,
                                'def_ship_via' => $shippingAddress->def_ship_via,
                                'is_default' => true,
                            ];
                        }
                        if(isset($rewardsShipVia) and count($rewardsShipVia)){
                            $rewardsShipVia[$shippingAddress->def_carr_cd] = [
                                'api_carrier' => null,
                                'residential' => null,
                                'hilco_carrier_code' => $shippingAddress->def_carr_cd,
                                'carrier_desc' => $defaultCarrier->carrier_desc,
                                'rate' => 2,
                                'text' => $defaultCarrier->carrier_desc . ' - ' . $this->formatPrice(2),
                                'service' => null,
                                'delivery_date' => null,
                                'delivery_time' => null,
                                'guaranteed' => null,
                                'def_ship_via' => $shippingAddress->def_ship_via,
                                'is_default' => true,
                            ];
                        }
                        $shipVia[$shippingAddress->def_carr_cd] = [
                            'api_carrier' => null,
                            'residential' => null,
                            'hilco_carrier_code' => $shippingAddress->def_carr_cd,
                            'carrier_desc' => $defaultCarrier->carrier_desc,
                            'rate' => 2,
                            'text' => $defaultCarrier->carrier_desc . ' - ' . $this->formatPrice(2),
                            'service' => null,
                            'delivery_date' => null,
                            'delivery_time' => null,
                            'guaranteed' => null,
                            'def_ship_via' => $shippingAddress->def_ship_via,
                            'is_default' => true,
                        ];
                    }
                }
            } else {
                $shipmentArray = [];
                $excludedShipmentArray = [];
                $rewardsShipmentArray = [];
                if(count($excludedItems)){
                    foreach ($excludedWeights as $excludedWeight){
                        $excludedShipmentArray[] = ['weight' => $excludedWeight, 'volume' => $excludedVolume, 'toCode' => $shippingAddress->postal_cd, 'fromCode' => $fromCode];
                    }
                    $excludedShipVia = RateShop::rateShopCanada($groupCode, $excludedShipmentArray);
                }
                if(count($rewardsItems)){
                    foreach ($rewardsWeights as $rewardsWeight){
                        $rewardsShipmentArray[] = ['weight' => $rewardsWeight, 'volume' => $rewardsVolume, 'toCode' => $shippingAddress->postal_cd, 'fromCode' => $fromCode];
                    }
                    $rewardsShipVia = RateShop::rateShopCanada($groupCode, $rewardsShipmentArray);
                }
                foreach ($weights as $weight) {
                    $shipmentArray[] = ['weight' => $weight, 'volume' => $volume, 'toCode' => $shippingAddress->postal_cd, 'fromCode' => $fromCode];
                }
                $shipVia = RateShop::rateShopCanada($groupCode, $shipmentArray);
            }
        }

        if(!isset($excludedShipVia)) $excludedShipVia = [];
        foreach ($excludedShipVia as $key => $value) {
            $excludedShipVia[$key]['id'] = $value['hilco_carrier_code'];
            if (isset($value['rate']) && $value['rate'] > 0) {
                $excludedShipVia[$key]['rate_string'] = number_format($value['rate'], 2);
                $excludedShipVia[$key]['text'] = $value['carrier_desc'] . ' - ' . $this->formatPrice($value['rate']);
                $excludedShipVia[$key]['is_default'] = $value['hilco_carrier_code'] == $defaultCarrierCode;
            } else {
                if (is_null($excludedShipVia[$key]['rate'])) unset($excludedShipVia[$key]);
            }
        }

        if(!isset($rewardsShipVia)) $rewardsShipVia = [];
        foreach ($rewardsShipVia as $key => $value) {
            $rewardsShipVia[$key]['id'] = $value['hilco_carrier_code'];
            if (isset($value['rate']) && $value['rate'] > 0) {
                $rewardsShipVia[$key]['rate_string'] = number_format($value['rate'], 2);
                $rewardsShipVia[$key]['text'] = $value['carrier_desc'] . ' - ' . $this->formatPrice($value['rate']);
                $rewardsShipVia[$key]['is_default'] = $value['hilco_carrier_code'] == $defaultCarrierCode;
            } else {
                if (is_null($rewardsShipVia[$key]['rate'])) unset($rewardsShipVia[$key]);
            }
        }

        if(!isset($shipVia)) $shipVia = [];
        foreach ($shipVia as $key => $value) {
            $shipVia[$key]['id'] = $value['hilco_carrier_code'];
            if (isset($value['rate']) && $value['rate'] > 0) {
                $shipVia[$key]['rate_string'] = number_format($value['rate'], 2);
                $shipVia[$key]['text'] = $value['carrier_desc'] . ' - ' . $this->formatPrice($value['rate']);
                $shipVia[$key]['is_default'] = $value['hilco_carrier_code'] == $defaultCarrierCode;
            } else {
                if (is_null($shipVia[$key]['rate'])) unset($shipVia[$key]);
            }
        }
        if(isset($shipVia['1413'])){
            $shipVia['1413']['rate'] = 0;
            $shipVia['1413']['rate_string'] = number_format($shipVia['1413']['rate'], 2);;
            $shipVia['1413']['text'] = $shipVia['1413']['carrier_desc'] . ' - ' . $this->formatPrice($shipVia['1413']['rate']);;
        }

        $baseRates = collect();
        $excludedBaseRates = collect();
        $rewardsBaseRates = collect();

        if (isset($shipVia['1100'])) {
            $baseRates['1100'] = array_get($shipVia, '1100.rate', 0);
        }

        if (isset($shipVia['1000'])) {
            $baseRates['1000'] = array_get($shipVia, '1000.rate', 0);
        } else if (isset($shipVia['1400'])) {
            $baseRates['1400'] = array_get($shipVia, '1400.rate', 0);
        }

        if (isset($excludedShipVia['1100'])) {
            $excludedBaseRates['1100'] = array_get($excludedShipVia, '1100.rate', 0);
        }

        if (isset($excludedShipVia['1000'])) {
            $excludedBaseRates['1000'] = array_get($excludedShipVia, '1000.rate', 0);
        } else if (isset($excludedShipVia['1400'])) {
            $excludedBaseRates['1400'] = array_get($excludedShipVia, '1400.rate', 0);
        }

        if (isset($rewardsShipVia['1100'])) {
            $rewardsBaseRates['1100'] = array_get($rewardsShipVia, '1100.rate', 0);
        }

        if (isset($rewardsShipVia['1000'])) {
            $rewardsBaseRates['1000'] = array_get($rewardsShipVia, '1000.rate', 0);
        } else if (isset($rewardsShipVia['1400'])) {
            $rewardsBaseRates['1400'] = array_get($rewardsShipVia, '1400.rate', 0);
        }

        $netOrderTotal = $cartStatus['discountedTotal']; //'discounts.rewards', 0) - array_get($cartStatus, 'discounts.promotion', 0);
        $netRewardsOrderTotal = 0;

        if(count($rewardsItems)) {
            if (count($excludedItems)) {
                foreach ($rewardsItems as $rewardsItem) {
                    $netRewardsOrderTotal += $rewardsItem['extendedPrice'];
                }
                $rewardsCarrierCodes = rewards()->carrierCode($netRewardsOrderTotal, $rewardsBaseRates, $cartStatus['rewardsTier']);
            } else {
                $rewardsCarrierCodes = rewards()->carrierCode($netOrderTotal, $baseRates, $cartStatus['rewardsTier']);
            }

            if ($rewardsCarrierCodes) {
                foreach ($rewardsCarrierCodes as $rewardsCarrierCode) {
                    if (count($excludedItems)) {
                        $baseRate = array_get($rewardsBaseRates, $rewardsCarrierCode['base_code'], 0) +
                            array_get($excludedBaseRates, $rewardsCarrierCode['base_code'], 0);
                        if ($baseRate > 0) {
                            $shipVia[$rewardsCarrierCode['code']] = [
                                'id' => $rewardsCarrierCode['code'],
                                'api_carrier' => null,
                                'residential' => null,
                                'hilco_carrier_code' => $rewardsCarrierCode['code'],
                                'carrier_desc' => $rewardsCarrierCode['description'],
                                'rate' => $baseRate,
                                'rate_string' => number_format($rewardsCarrierCode['rate'], 2),
                                'text' => $rewardsCarrierCode['description'] . ' - ' . $this->formatPrice($rewardsCarrierCode['rate']),
                                'service' => null,
                                'delivery_date' => null,
                                'delivery_time' => null,
                                'guaranteed' => null,
                                'base_rate' => $baseRate,
                                'base_rate_string' => number_format($baseRate, 2),
                                'discount_amount' => $rewardsCarrierCode['rate'],
                                'is_default' => $rewardsCarrierCode['code'] == $defaultCarrierCode,
                            ];
                        }
                    } else {
                        $baseRate = array_get($baseRates, $rewardsCarrierCode['base_code'], 0);
                        if ($baseRate > 0) {
                            $shipVia[$rewardsCarrierCode['code']] = [
                                'id' => $rewardsCarrierCode['code'],
                                'api_carrier' => null,
                                'residential' => null,
                                'hilco_carrier_code' => $rewardsCarrierCode['code'],
                                'carrier_desc' => $rewardsCarrierCode['description'],
                                'rate' => $baseRate,
                                'rate_string' => number_format($rewardsCarrierCode['rate'], 2),
                                'text' => $rewardsCarrierCode['description'] . ' - ' . $this->formatPrice($rewardsCarrierCode['rate']),
                                'service' => null,
                                'delivery_date' => null,
                                'delivery_time' => null,
                                'guaranteed' => null,
                                'base_rate' => $baseRate,
                                'base_rate_string' => number_format($baseRate, 2),
                                'discount_amount' => $rewardsCarrierCode['rate'],
                                'is_default' => $rewardsCarrierCode['code'] == $defaultCarrierCode,
                            ];
                        }
                    }
                }
            }
        }

        $customer = b2b()->activeCustomer();
        $customerCategory = $customer->overrideCustomerCategory;
        $category = $customerCategory->cust_category;
        if ($category === 'OPTICIAN' || $category === 'OPTOMETRIST' || $category === 'OPHTHALMOLOGIST' || $category === 'LEASED DOCTOR') {
            if(count($excludedItems) && count($rewardsItems)){
                if($netRewardsOrderTotal >= 150){
                    if(isset($rewardsBaseRates['1000']) && isset($excludedBaseRates['1000'])){
                        $baseRate = array_get($rewardsBaseRates, '1000');
                        $rate = round($baseRate / 4, 2);
                        $baseRate += array_get($excludedBaseRates, '1000');
                        $shipVia['1409'] = [
                            'id' => '1409',
                            'api_carrier' => null,
                            'residential' => null,
                            'hilco_carrier_code' => '1409',
//                            'carrier_desc' => 'Promotional 25% off Ground Shipping',
                            'carrier_desc' => 'Promotional Ground Shipping',
                            'rate' => $baseRate,
                            'rate_string' => number_format($rate, 2),
//                            'text' => 'Promotional 25% off Ground Shipping' . ' - ' . $this->formatPrice($rate),
                            'text' => 'Promotional Ground Shipping' . ' - ' . $this->formatPrice($rate),
                            'service' => null,
                            'delivery_date' => null,
                            'delivery_time' => null,
                            'guaranteed' => null,
                            'base_rate' => $baseRate,
                            'base_rate_string' => number_format($baseRate, 2),
                            'discount_amount' => $rate,
                            'is_default' => 1409 == $defaultCarrierCode,
                        ];
                    }
                }
            } else if(count($rewardsItems)){
                if ($netOrderTotal >= 150) {
                    // If baserates doens't exist, don't offer 1409

                    if (isset($baseRates['1000'])) {
                        $baseRate = array_get($baseRates, '1000');
                        $rate = round($baseRate / 4, 2);
                        $shipVia['1409'] = [
                            'id' => '1409',
                            'api_carrier' => null,
                            'residential' => null,
                            'hilco_carrier_code' => '1409',
//                            'carrier_desc' => 'Promotional 25% off Ground Shipping',
                            'carrier_desc' => 'Promotional Ground Shipping',
                            'rate' => $baseRate,
                            'rate_string' => number_format($rate, 2),
//                            'text' => 'Promotional 25% off Ground Shipping' . ' - ' . $this->formatPrice($rate),
                            'text' => 'Promotional Ground Shipping' . ' - ' . $this->formatPrice($rate),
                            'service' => null,
                            'delivery_date' => null,
                            'delivery_time' => null,
                            'guaranteed' => null,
                            'base_rate' => $baseRate,
                            'base_rate_string' => number_format($baseRate, 2),
                            'discount_amount' => $rate,
                            'is_default' => 1409 == $defaultCarrierCode,
                        ];

                    }

                }
            }
        }

        if ($this->activeCountry() == 'US') { // hacky wrapper if-statement to force promotion to only apply to US customers (per Bob Woyton's email at 3:14pm on 12/0117)
            $baseRate = array_get($baseRates, '1000', 0);
            // check if order qualifies for shipping discount promotions and add them to shipVia list
            $totalShippingDiscounts = 0;
            $triggeredPromotions = array_get($cartStatus, 'triggeredPromotions');
            foreach ($triggeredPromotions as $triggeredPromotion) {
                $shippingDiscountActions = $triggeredPromotion->shippingDiscountActions;
                foreach ($shippingDiscountActions as $shippingDiscountAction) {
                    if ($totalShippingDiscounts < $baseRate) {
                        if ($shippingDiscountAction->is_percent) {
                            $totalShippingDiscounts += round(($baseRate * $shippingDiscountAction->amount / 100), 2);
                        } else {
                            $totalShippingDiscounts += $shippingDiscountAction->amount;
                        }
                    }
                }
            }
            if ($totalShippingDiscounts > 0) {
                if ($totalShippingDiscounts > $baseRate) {
                    $totalShippingDiscounts = $baseRate;
                }
                $shipVia['5001'] = [
                    'id' => '5001',
                    'api_carrier' => null,
                    'residential' => null,
                    'hilco_carrier_code' => '5001',
                    'carrier_desc' => 'Promotional Ground Shipping',
                    'rate' => $baseRate,
                    'rate_string' => number_format($totalShippingDiscounts, 2),
                    'text' => 'Promotional Ground Shipping' . ' - ' . $this->formatPrice($totalShippingDiscounts),
                    'service' => null,
                    'delivery_date' => null,
                    'deliver_time' => null,
                    'guaranteed' => null,
                    'base_rate' => $baseRate,
                    'base_rate_string' => number_format($baseRate, 2),
                    'discount_amount' => $totalShippingDiscounts,
                    'is_default' => 5001 == $defaultCarrierCode,
                ];
            }
        }

        if (count($shipVia) == 0) {
            $shipVia[0] = ['id' => -1, 'text' => trans('hilco::app.unable_to_calculate_shipping_rates')];
        }

        return array_values($shipVia);
    }
//  ******************************** RATE SHOP ********************************


    public function hasSentFeedback() {
        $user = auth()->user();
        $has = ($user) ? auth()->user()->webFeedbackResponses()->count() > 0 : false;
        return $has;
    }

    public function getShipmentRates(CustomerShippingAddress $customerShippingAddress) {
        $cartComposer = App::make('cartSingleton');
        $cartStatus = $cartComposer->getCartStatus();
        $rateType = array_get($this->activeWebSilo, 'shipment_rate_type', 'api');
        $rates = new Collection;

        if ($rateType == 'api'){
            $carriers = array_get($this->activeWebSilo, 'carriers', []);
            $package = new Package(array_get($cartStatus, 'totalWeight', 0), array_get($cartStatus, 'totalVolume', 0), $customerShippingAddress->postal_cd, $customerShippingAddress->country);
            foreach ($carriers as $carrier) {

            }
        } else {
            $rates = array_get($this->activeWebSilo, 'shipmentFlatRates', false);
            krsort($rates);
            $cartValue = 0;
            if ($rateType == 'flatByWeight') $cartValue = array_get($cartStatus, 'totalWeight', 0);
            if ($rateType == 'flatByPrice') $cartValue = array_get($cartStatus, 'discountedTotal', 0);
            foreach ($rates as $threshold => $rate) {
                if ($cartValue >= $threshold) {
                    $rates->push([
                        'rate' => $rate,
                    ]);
                    break;
                }
            }
        }

        return $rates;
    }

    public function getAPIShipmentRates($cartStatus, $customerShippingAddress) {

    }

    public function allowBannerCarousel($lpModel) {
        if ($this->activeWebSilo->id == '2') {
            if ($lpModel instanceof WebGroup || $lpModel instanceof WebCategory || $lpModel instanceof WebCollection) {
                return false;
            }
        }
        return true;
    }

    /**
     * Retrieves product models from an array of productType, productId pairs
     *
     * @param $products
     */
    public function retrieveProductModels($products) {
        $idsByType = [];
        foreach ($products as $product) {
            $type = array_get($product, 'productType');
            $id = array_get($product, 'productId');
            array_set($idsByType, "$type.$id", $id);
        }

        $map = Relation::morphMap();
        $models = collect();
        foreach ($idsByType as $type => $ids) {
            $type = normalizeProductType($type);
            $className = array_get($map, $type, false);
            if ($className === false) continue;
            $classModels = $className
                ::hasVisibleChildren()
                ->hasVisibleParents()
                ->isVisible()
                ->findMany($ids)
            ;
            foreach ($classModels as $model) {
                $models->push($model);
            }
        }

        return $models;
    }

    public function defaultRole() {
        $defaultRoleSlug = config('hilco.defaultWebRoleSlug');
        $webRole = WebRole::where('slug', '=', $defaultRoleSlug)->first();

        return $webRole;
    }

    public function catalogsWebLink() {
        $activeWebSilo = $this->activeWebSilo();
        $activePlantId = $this->activePlant()->id;
        $catalogsWebLinkId = $activeWebSilo->catalogs_weblink_id;
        $linkJoins = $activeWebSilo->webLinkWebSiloJoins->where('weblink_id', $catalogsWebLinkId)->keyBy('plant_id');
        $validLink = array_get($linkJoins, "$activePlantId", array_get($linkJoins, '-1'));
//        if (array_get($validLink, 'is_visible')) {
//
//        }
        return $validLink;

    }
    function promosHaveCartHeaders($cartStatus){
        $triggeredPromotions = $cartStatus['triggeredPromotions'];
        $almostQualifiedPromotionsInfo = $cartStatus['almostQualifiedPromotionsInfo'];
        if(!is_null($triggeredPromotions)){
            foreach($triggeredPromotions as $promotion){
                if(strlen($promotion->cart_header)){
                    return true;
                }
            }
        }
        if(!is_null($almostQualifiedPromotionsInfo)){
            foreach($almostQualifiedPromotionsInfo as $key => $almostQualifiedPromoInfo){
                if(isset($almostQualifiedPromoInfo['promotion'])){
                    $promotion = $almostQualifiedPromoInfo['promotion'];
                    if(strlen($promotion->almost_header) > 0){
                        return true;
                    }
                }
            }
        }
        return false;

    }

    function promosHaveCheckoutHeaders($cartStatus){
        $triggeredPromotions = $cartStatus['triggeredPromotions'];
        $almostQualifiedPromotionsInfo = $cartStatus['almostQualifiedPromotionsInfo'];
        if(!is_null($triggeredPromotions)){
            foreach($triggeredPromotions as $promotion){
                if(strlen($promotion->checkout_header)){
                    return true;
                }
            }
        }
        if(!is_null($almostQualifiedPromotionsInfo)){
            foreach($almostQualifiedPromotionsInfo as $key => $almostQualifiedPromoInfo){
                if(isset($almostQualifiedPromoInfo['promotion'])){
                    $promotion = $almostQualifiedPromoInfo['promotion'];
                    if(strlen($promotion->almost_header) > 0){
                        return true;
                    }
                }
            }
        }
        return false;

    }

    public function shippingErrorMessage() {
        $note = array_get($this->activeWebSilo(), 'shipping_note');
        if (strlen($note)) {
            return $note;
        } else {
            return trans('messages.shippingError');
        }
    }

    public function defaultLocale() {
        return 'us_EN';
    }

    public function defaultLanguage() {
        return 'en';
    }


//  ******************************** PASSWORDS ********************************
    public function getPasswordValidationMinLength() {
        return 8;
    }
    public function getPasswordValidationRegex() {
        return '/^(?=.*[a-zA-Z])(?=.*\d).+$/';
    }

    public function getPasswordValidationString() {
        $passwordValidationMinLength = $this->getPasswordValidationMinLength();
        $passwordValidationRegex = $this->getPasswordValidationRegex();
        $passwordValidationString = "min:$passwordValidationMinLength|regex:$passwordValidationRegex";
        return $passwordValidationString;
    }

    public function generateRandomPasswordString($length = null, $passwordValidationRegex = null) {
        if (!isset($length)) {
            $length = $this->getPasswordValidationMinLength();
        }

        if (!isset($passwordValidationRegex)) {
            $passwordValidationRegex = $this->getPasswordValidationRegex();
        }

        $allowedCharacters = '346789abcdefghkmnpqrtuvwxyzABCDEFGHJKMNPRTUVWXY';
        $allowedCharactersLength = strlen($allowedCharacters);
        $randomPasswordString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomPasswordString .= $allowedCharacters[rand(0, $allowedCharactersLength - 1)];
        }

        if (!preg_match($passwordValidationRegex, $randomPasswordString)) {
            $randomPasswordString = $this->generateRandomPasswordString($length, $passwordValidationRegex);
        }

        return $randomPasswordString;
    }
//  ******************************** PASSWORDS ********************************


}
