<?php

namespace Hilco\Models;

use HilcoB2B\M3Request;
use Illuminate\Database\Eloquent\Builder;
use Log;

/**
 * Hilco\Models\InventoryItem
 *
 * @property mixed $id
 * @property string $plant
 * @property mixed $plant_id
 * @property string $part_no
 * @property mixed $part_id
 * @property string $part_stat
 * @property string $part_src
 * @property string $part_type
 * @property boolean $part_taxable
 * @property boolean $kit_flag
 * @property boolean $on_bom
 * @property boolean $has_bom
 * @property integer $mfg_atp
 * @property float $first_avail_qty
 * @property string $first_avail_dt
 * @property float $max_avail_qty
 * @property string $max_avail_dt
 * @property string $date_created
 * @property string $date_modified
 * @property string $deleted_at
 * @property-read mixed $id_string
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem wherePlant($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem wherePlantId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem wherePartNo($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem wherePartId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem wherePartStat($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem wherePartSrc($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem wherePartType($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem wherePartTaxable($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereKitFlag($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereOnBom($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereHasBom($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereMfgAtp($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereFirstAvailQty($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereFirstAvailDt($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereMaxAvailQty($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereMaxAvailDt($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereDateCreated($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereDateModified($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereDeletedAt($value)
 * @mixin \Eloquent
 * @property integer $lot_control
 * @property string $date_uploaded
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereLotControl($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\InventoryItem whereDateUploaded($value)
 */
class InventoryItem extends WebModel {
    // TODO: change this to InventoryItems when we're for it on CST as well
    protected $table = 'InventoryItems_tmp';

    const STATUS_DONOTREORDER   = 'DoNotReorder';
    const STATUS_OPEN           = 'Open';
    const STATUS_OBSOLETE       = 'Obsolete';

    /**
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     * @deprecated shouldn't be necessary anymore, use Part::partBOMs()
     */
    public function partBOMs() {
        return $this->hasMany(PartBOM::class, 'main_part_id', 'part_id');
    }

    public function part() {
        return $this->belongsTo(Part::class, 'part_id', 'id');
    }

    public function scopeInPlant(Builder $query, $plant) {
        return $query->where('plant', '=', $plant);
    }

    /**
     * @param Builder $query
     * @param $flags
     * @return mixed
     * @deprecated web_flag is no longer a concept, this and any references to it need to be removed
     */
    public function scopeWebFlagIn(Builder $query, $flags) {
        if (!is_array($flags)) $flags = [$flags];
        return $query->whereIn('web_flag', $flags);
    }

    public function scopeAvailableInPlant(Builder $query, $activePlants = false) {
        if ($activePlants === false) $activePlants = b2b()->activePlants();
        $query->where(function ($query) use ($activePlants) {
            foreach ($activePlants as $activePlant) {
                if (!strlen($activePlant->plant)) continue;
                $query->orWhere(function ($query) use ($activePlant) {
                    return $query
                        ->where('plant', $activePlant->plant)
                        ->where(function ($query) {
                            $query
                                ->where('part_stat', '!=', InventoryItem::STATUS_OBSOLETE)
                            ;
                        })
                    ;
                });
            }
            return $query;
        });
        return $query;
    }
    
    public function scopeAvailableInAnyPlant (Builder $query) {
        return $query->where('part_stat', '!=', InventoryItem::STATUS_OBSOLETE);
    }

    public function scopeInSilo(Builder $query, $plant = false) {
        if ($plant === false) $plant = b2b()->activePlant();
        $query->where(function ($query) use ($plant) {
            return $query
                ->where('plant', array_get($plant, 'plant'))
                ->where(function ($query) {
                    $query
                        ->where('part_stat', '!=', InventoryItem::STATUS_OBSOLETE)
                    ;
                })
            ;
        });
        return $query;
    }

    public function scopeIsValid(Builder $query) {
        $query->where('part_stat', '!=', InventoryItem::STATUS_OBSOLETE);
    }

    public function getIsValidAttribute() {
        return $this->part_stat != InventoryItem::STATUS_OBSOLETE;
    }

    public function getInStockAttribute() {
        // First check if the current active customer is default ship-from Las Vegas...
        $activeCustomer = b2b()->activeCustomer();
        $activeSegment = array_get($activeCustomer, 'activeSegment', false);
        if($activeSegment && $activeSegment->def_ship_from == Plant::LASVEGAS_CODE) {
            // ...because if so we need to only use the Las Vegas logic plants
            $activePlantNames = Plant::lasVegasLogicPlants();
        } else {
            // ...otherwise just use the current active plants
            $activePlantNames = b2b()->activePlants()->pluck('plant')->toArray();
        }

        // Don't bother considering this inventoryitem record if it isn't one of the active plants
        if (!(in_array($this->plant, $activePlantNames))) {
            return false;
        }

        // Don't bother considering this inventoryitem record if the part is obsolete at this plant
        if ($this->part_stat == InventoryItem::STATUS_OBSOLETE) {
            return false;
        }

        // Now that we've made sure this inventoryitem record is one of the active plants and the part is not obsolete
        // check the ATP value for this part at this plant (the old check was "return net_avail > 0;", for reference)
        $atp = self::getATP($this->part_no, $this->plant);
        return
            !empty($atp) ?
                ($atp > 0) :
                ($atp === false ? $atp : null)
        ;
    }

    public static function getATP ($partNumber, $warehouse) {
        if(auth()->user()) {
            $guzzleClient = null;
            $guzzleM3Request = new M3Request();
            if (empty($guzzleM3Request->client)) {
                Log::error(
                    'Exception Caught while performing ' . __FUNCTION__ .
                    ': Guzzle Http Client does not exist. Base URI may be missing!')
                ;
                return null;
            } else {
                $guzzleClient = $guzzleM3Request->client;
            }

            $atpRequestData = [];

            $deliveryDueDate = Carbon::now()->addDay()->format('Ymd');
            $atpRequestData[] = [
                'item_number' => $partNumber,
                'warehouse' => $warehouse,
                'requested_delivery_date' => $deliveryDueDate
            ];
            $response = json_decode(
                $guzzleClient
                    ->get(
                        'v4/' . __FUNCTION__,
                        [
                            'query'     => [
                                'requestFrom'           => 'b2b',
                                'itemsATPRequestInfo'   => $atpRequestData
                            ],
                            'headers'   => [
                                'Accept' => 'application/json'
                            ]
                        ]
                    )
                    ->getBody()
                    ->getContents(),
                true
            );

            foreach ($response['atps'] as $atp) {
                if (count($atp['details']) == 1) {
                    Log::error('Exception Caught while performing ' . __FUNCTION__ . ': ' . $atp['details']);
                    return null;
                } else if (
                    array_has($atp['details'], 'MIRecord') &&
                    array_has($atp['details']['MIRecord'][0], 'NameValue')) {

                    $results = $atp['details']['MIRecord'][0]['NameValue'];
                    foreach ($results as $result) {
                        if ($result['Name'] == 'AVTQ') {
                            return round(trim($result['Value'], ' '));
                        }
                    }

                } else {
                    Log::error('Exception Caught while performing ' . __FUNCTION__ . ': ' . $atp['details']['Message']);
                    return null;
                }
            }
        }

        return null;
    }
}

