<?php

namespace Hilco\Models;

use Auth;
use DB;
use Illuminate\Database\Eloquent\Model;

class WebCart extends WebModel
{
    protected $table = 'WebCarts';
    protected $fillable = ['quantity', 'comment'];

    public function webPart() {
        return $this->hasOne(WebPart::class, 'part_number', 'part_number');
    }

    public function webUser() {
        return $this->hasOne(WebUser::class, 'webuser_id', 'id');
    }

    public function priceList() {
//        return $this->hasOne(PriceList::class)
    }

    public function scopeForUser($query, $webuserId) {
        return $query->where('webuser_id', $webuserId);
    }

    public function scopePartNumber($query, $partNumber) {
        return $query->where('part_number', $partNumber);
    }

//    public function product() {
//        return $this->belongsTo(Product::class);
//    }

    public function getExtendedPriceAttribute() {
        return $this->webPart->getCustomerPrice($this->quantity) * $this->quantity;
    }

    public static function add(WebPart $webPart, $quantity = 1) {
        $webUserId = Auth::id();
        $webCart = static::firstOrNew([
            'webuser_id' => $webUserId,
            'part_number' => $webPart->part_number,
        ]);

        if (array_get($webCart, 'quantity', 0)) {
            $webCart->quantity += $quantity;
        } else {
            $webCart->webuser_id = $webUserId;
            $webCart->part_number = $webPart->part_number;
            $webCart->quantity = $quantity;
        }

        $webCart->save();
        return $webCart;
    }

    public static function remove($userId, $partNumber)
    {
        $instance = new static;
        return $instance::where(['part_number' => $partNumber, 'webuser_id' => $userId])->delete();
    }

    public static function cartStatus(WebSilo $webSilo = null, WebUser $webUser = null, Customer $soldToCustomer = null) {
        if (is_null($webSilo)) $webSilo = b2b()->activeWebSilo();
        if (is_null($webUser)) $webUser = auth()->user();
        if (is_null($soldToCustomer)) $soldToCustomer = b2b()->activeCustomer();

        if (is_null($webSilo) || is_null($webUser) || is_null($soldToCustomer)) return [];

        $status = [
            'itemsCount' => 0,
            'quantityCount' => 0,
            'baseTotal' => 0,
            'discountedTotal' => 0,
            'discounts' => [
                'rewards' => 0,
                'customer' => 0,
            ],
            'productCategoryTotals' => [],
            'rewardsUpgradeSpending' => 0,
//            'rewardsTier' => rewards()->tier($soldToCustomer),
            'rewardsTier' => null,
            'taxAmount' => 0,
            'taxPercent' => null,
            'shippingAmount' => 0,
            'shippingDiscountAmount' => 0,
            'shippingTaxAmount' => 0,
            'requireApproval' => false,
            'hasNetItem' => false,
        ];

        $items = [];

        $customerPriceListName = array_get($soldToCustomer, 'activeSegment.def_price_list');
        $currencyCode = $soldToCustomer->currency;
        $siloPriceListName = $webSilo->priceList;
        $isCustomerTaxable = b2b()->isCustomerTaxable();

        $webCartItems = self
            ::where('webuser_id', ($webUser->id))
            ->get()
        ;

        $distinctPartQuantities = [];
        foreach ($webCartItems as $webCartItem) array_increment($distinctPartQuantities, $webCartItem->webPart->part->id, $webCartItem->quantity);


        foreach ($webCartItems as $webCartItem) {
            /**
             * @var Part $part
             */
            $part = $webCartItem->webPart->part;

            $isContractPrice = false;
            $usedPriceList = 'PART_TABLE_LIST_PRICE';
            $discounts = [];

            $priceLevel = $part->getPriceLevelInformationForQuantity($webCartItem->quantity);
            $basePrice = array_get($priceLevel, 'genericPrice');

            $extendedBasePrice = $basePrice * $webCartItem->quantity;
            $discountedPrice = $part->getCustomerPrice($webCartItem->quantity);
            $baseDiscountAmount = $basePrice - $discountedPrice;
            $extendedDiscountAmount = $baseDiscountAmount * $webCartItem->quantity;
            if ($webCartItem->quantity > 1 && $basePrice > 0) {
                $discountPercent = $discountedPrice != $basePrice ? round((1 - $discountedPrice/$basePrice) * 100, 2) : 0;
                $extendedPrice = $extendedBasePrice - round($extendedBasePrice * round(1 - $discountedPrice/$basePrice, 2), 2);
//                $extendedPrice = $discountedPrice * $webCartItem->quantity;
            } else {
                $discountPercent = 0;
                $extendedPrice = $discountedPrice;
            }
            $itemTaxAmount = 0;

            if ($isCustomerTaxable) {
                $partTax = $part->vat_percentage;
                $status['taxPercent'] = $partTax;
                if ($partTax) {
                    $itemTaxAmount += round($extendedPrice * ($partTax / 100), 2);
                }
            }

            $itemRequiresApproval = $webCartItem->webPart->requireApproval;
            if ($itemRequiresApproval) {
                $status['requireApproval'] = true;
            }


            $items[$webCartItem->id] = collect([
                'id' => $webCartItem->id,
                'webCart' => $webCartItem,
                'webPart' => $webCartItem->webPart,
                'part' => $webCartItem->webPart->part,
                'basePrice' => $basePrice,
                'quantity' => $webCartItem->quantity,
                'extendedBasePrice' => $extendedBasePrice,
                'discountedPrice' => $discountedPrice,
                'baseDiscountAmount' => $baseDiscountAmount,
                'discountPercent' => $discountPercent,
                'extendedPrice' => $extendedPrice,
                'extendedDiscountAmount' => $extendedDiscountAmount,
                'displayDiscount' => array_get($priceLevel, 'discount', $discountPercent),
                'discounts' => $discounts,
                'isContractPrice' => $isContractPrice,
                'usedPriceList' => $usedPriceList,
                'taxAmount' => $itemTaxAmount,
                'requireApproval' => $itemRequiresApproval,
                'isDiscounted' => $part->getIsDiscounted($webCartItem->quantity),
                'isNetItem' => isset($priceLevel['netItem']) ? $priceLevel['netItem'] : false,
            ]);
            array_increment($status, 'itemsCount');
            array_increment($status, 'quantityCount', $webCartItem->quantity);
            array_increment($status, 'baseTotal', $extendedBasePrice);
            array_increment($status, 'discountedTotal', $extendedPrice);
            array_increment($status, 'taxAmount', $itemTaxAmount);

            $status['hasNetItem'] = $status['hasNetItem'] || $items[$webCartItem->id]['isNetItem'];

            foreach ($discounts as $discountType => $discountInfo) {
                array_increment($status, "discounts.$discountType", $discountInfo['extendedAmount']);
            }

//            $productCategoryGroupId = $part->productFamily->productCategory->productCategoryGroup->id;
//            array_increment($status, "productCategoryTotals.$productCategoryGroupId", $extendedPrice);
        }

        if (rewards()->isRegistered() && b2b()->isBillingDirect() && rewards()->isEligible($soldToCustomer, false)) {
            $rewardsUpgradeSpending = rewards()->calculateUpgradeSpending($status, $soldToCustomer);
            $status['rewardsUpgradeSpending'] = $rewardsUpgradeSpending;

            if ($status['rewardsUpgradeSpending'] <= 0) { //Tier Upgrade
                $status['baseRewardsTier'] = $status['rewardsTier'];
                $status['rewardsTier'] = min(4, rewards()->tier() + 1);
                $status['upgradedTier'] = min(4, rewards()->tier() + 1);

                $rewardsDiscountTotal = 0;
                foreach ($items as $itemKey => $item) {
//                if (array_has($item, 'discounts.rewards')) {
                    $discountPercent = rewards()->discountForPartTier($item['part']->id, $status['upgradedTier'], $soldToCustomer);
                    $discountAmount = round($item['basePrice'] * ($discountPercent / 100), 2);
                    $extendedDiscountAmount = $discountAmount * $item['quantity'];
                    $baseDiscountAmount = round($item['basePrice'] * ($discountPercent / 100), 2);
                    $discountedPrice = $item['basePrice'] - $baseDiscountAmount;
                    $extendedPrice = $discountedPrice * $item['quantity'];
                    $rewardsDiscountTotal += $extendedDiscountAmount;

                    $item->put('discounts.rewards.percent', $discountPercent);
                    $item->put('discounts.rewards.baseAmount', $discountAmount);
                    $item->put('discounts.rewards.extendedAmount', $extendedDiscountAmount);
                    $item->put('discountPercent', $discountPercent);
                    $item->put('baseDiscountAmount', $discountAmount);
                    $item->put('extendedDiscountAmount', $extendedDiscountAmount);
                    $item->put('discountedPrice', $discountedPrice);
                    $item->put('extendedPrice', $extendedPrice);
//                }
                }
                $status['discounts']['rewards'] = $rewardsDiscountTotal;
            }
        }


        $webSiloShippingTier = $webSilo->webSiloShippingTiers()->where('minAmount', '<=', $status['discountedTotal'])->orderBy('minAmount', 'DESC')->first();
        if (isset($webSiloShippingTier)) {
            $status['shippingAmount'] = $webSiloShippingTier->shippingRate;
        } else {
            $shippingFee = ShippingFee::country(b2b()->activeCountry())->first();
            if ($status['discountedTotal'] <= array_get($shippingFee, 'order_limit')) {
                $status['shippingAmount'] = array_get($shippingFee, 'fee', 0);
            }
        }

        $status['orderTotal'] = $status['discountedTotal'] + $status['taxAmount'] + $status['shippingAmount'];

        $webUserWebSilo = $webUser->webSilo()->where('websilo_id', b2b()->activeWebSilo()->id)->first();
        $webCustomerWebSilo = $webUser->customer->webSilo()->where('websilo_id', b2b()->activeWebSilo()->id)->first();
        $aliasWebCustomerWebSilo = b2b()->activeCustomer()->webSilo()->where('websilo_id', b2b()->activeWebSilo()->id)->first();

        $spendingLimit = array_get($webUserWebSilo, 'pivot.spending_limit', -1);
        $minimumOrder = array_get($webUserWebSilo, 'pivot.minimum_order', -1);

        if($spendingLimit <= 0) $spendingLimit = array_get($webCustomerWebSilo, 'pivot.spending_limit', -1);
        if($minimumOrder <= 0) $minimumOrder = array_get($webCustomerWebSilo, 'pivot.minimum_order', -1);

        if($spendingLimit <= 0) $spendingLimit = array_get($aliasWebCustomerWebSilo, 'pivot.spending_limit', -1);
        if($minimumOrder <= 0) $minimumOrder = array_get($aliasWebCustomerWebSilo, 'pivot.minimum_order', -1);

        if($spendingLimit <= 0) $spendingLimit = array_get($webSilo, 'default_spending_limit', -1);
        if($minimumOrder <= 0) $minimumOrder = array_get($webSilo, 'default_minimum_order', -1);

        if ($spendingLimit > 0 && $status['orderTotal'] > $spendingLimit) $status['requireApproval'] = true;
        if ($minimumOrder > 0 && $status['orderTotal'] < $minimumOrder) $status['requireApproval'] = true;

        $status['items'] = $items;
        return $status;
    }
}
