<?php

namespace Hilco;

use Hilco\Models\Customer;
use Hilco\Models\CustomerRewardsStatus;
use Hilco\Models\Part;
use Hilco\Models\ProductCategoryGroup;
use Hilco\Models\WebGroup;

class RewardsHelper {

    public function tierMap($id = false) {
        if ($id === false) return config('rewards.tierMap');
        return config("rewards.tierMap.$id", $id);
    }

    public function isEligible(Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;

        $customerCategory = $customer->overrideCustomerCategory;
        $customerCategoryId = is_null($customerCategory) ? null : $customerCategory->id;
        $validCustomerCategory = in_array($customerCategoryId, config('rewards.eligibleCustomerCategoryIds', []));

        $validMemberGroup = !in_array($customer->member_group, config('rewards.excludedMemberGroups', []));
        $validBilling = $customer->billsDirect;
        $validPriceList = in_array($customer->activeSegment->def_price_list, config('rewards.eligiblePriceLists', []));
        $validCurrency = in_array($customer->currency, config('rewards.eligibleCurrencyCodes', []));

        $validAffiliates = true;
        foreach ($customer->affiliateCustomer->affiliateChildren as $affiliate) {
            if (!$affiliate->billsDirect) $validAffiliates = false;
        }

        return ($validCustomerCategory && $validMemberGroup && $validBilling && $validAffiliates && $validPriceList && $validCurrency);
    }

    public function isRegistered(Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;

        $registration = $customer->rewardsRegistration;
        return !is_null($registration);
    }

    public function isQualified(Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;
        $tier = $customer->rewardsTier;

        return is_null($tier) ? false : $tier->tier > 1;
    }

    public function model(Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;
        $customerCategory = $customer->overrideCustomerCategory;
        $customerCategoryId = is_null($customerCategory) ? null : $customerCategory->id;
        return in_array($customerCategoryId, [7,14]) ? 1 : 2;
    }

    public function spendingThreshold(Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;

        $threshold = 0;
        $model = $this->model($customer);
        $categories = $this->qualifyingCategories($customer);
//        $tier = $this->tier($customer);
        if ($model == 1) {
            if ($categories >= 5) {
                $threshold = 800;
            } else if ($categories >= 3) {
                $threshold = 2000;
            } else {
                $threshold = 5000;
            }
        } else {
            if ($categories >= 5) {
                $threshold = 2500;
            } else if ($categories >= 3) {
                $threshold = 6000;
            } else {
                $threshold = 14000;
            }
        }

        return $threshold;
    }

    public function recalculate(Customer $customer = null) {

    }

    public function qualifyingCategories(Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;

        return $customer->rewardsTier->category_mix;
    }

    public function totalSpending(Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;

        return $customer->rewardsTier->total_spending;
    }

    public function tier(Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;

        $rewardsTier = $customer->rewardsTier;

        return is_null($rewardsTier) ? 0 : $rewardsTier->tier;
    }

    public function tierName(Customer $customer = null) {
        return $this->tierMap($this->tier($customer));
    }

    public function joinDate(Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;

        $registration = $customer->rewardsRegistration;
        if (is_null($registration)) return false;

        return $registration->date_created->toDateString();
    }

    public function discountForPart($partId, Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;

        $part = ($partId instanceof Part) ? $partId : Part::find($partId);
        $groupName = $part->productFamily->productCategory->productCategoryGroup->product_category_group;
        $currentTier = $this->tier($customer);
        $discount = config("rewards.productCategoryGroups.$groupName.discountPercentByTier.$currentTier", 0);
        return $discount;
    }

    public function calculateUpgrade($currentOrderSpending, Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;

        $status = $customer->rewardsStatus()->with('productCategoryGroup')->get();

        $currentTotal = $newTotal = $currentMix = $newMix = 0;
        $newSpending = collect();

        foreach ($status as $group) {
            $currentTotal += $group->spending;
            if ($group->spending >= 200) $currentMix++;

            $productCategoryGroupName = $group['productCategoryGroup']['product_category_group'];

            $groupTotalAfterOrder = ($group->spending + (float)array_get($currentOrderSpending, $productCategoryGroupName, 0));
            $newTotal += $groupTotalAfterOrder;
            if ($groupTotalAfterOrder >= 200) $newMix++;

            $newGroup = [
                'spending_before' => $group->spending,
                'spending_after' => $groupTotalAfterOrder,
                'qualifying_before' => $group->qualifying,
                'qualifying_after' => ($groupTotalAfterOrder >= 200),
                'product_category_group' => $productCategoryGroupName,
            ];

            $newSpending->put($group['productcategorygroup_id'], $newGroup);
        }



        return $newSpending;
    }

    public function calculateTier($spending, $categories, Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;
    }

    public function productCategoryGroups(Customer $customer = null) {
        if (is_null($customer)) $customer = auth()->user()->activeCustomer;

        $groupsConfig = config('rewards.productCategoryGroups');
        $map = collect();
        foreach ($groupsConfig as $name => $info) {
            $map->put($info['webGroupSlug'], $name);
        }

        $status = $customer->rewardsStatus;
        $groups = ProductCategoryGroup::orderBy('product_category_group')->get()->keyBy('id');
        $webGroups = WebGroup::whereIn('slug', $map->keys())->get();

        $data = collect();
        foreach ($webGroups as $webGroup) {
            $groupName = $map[$webGroup->slug];
            $group = $groups->where('product_category_group', $groupName)->first();
            if (is_null($group)) continue;
            $theStatus = $status->where('productcategorygroup_id', $group->id)->first();
            if (is_null($theStatus)) {
                $theStatus = new CustomerRewardsStatus();
                $theStatus->productcategorygroup_id = $group->id;
                $theStatus->spending = 0;

            }
            $theStatus['displayName'] = $webGroup->name;
//            $theStatus['product_category_group'] = $groupName;
            $theStatus['webGroupId'] = $webGroup->id;
            $theStatus['webGroupSlug'] = $webGroup->slug;
            $theStatus['spendingPercent'] = 100 * (min(200, $theStatus['spending']) / 200);
            $theStatus['discounts'] = array_get($groupsConfig, "$groupName.discountPercentByTier");
            $data->put($group->id, $theStatus);
        }

        return $data;
    }

    public function info(Customer $customer) {
        $productCategoryGroups = ProductCategoryGroup::get()->keyBy('id');
        $tierLevel = 0;
        $tierName = rewards()->tierMap(0);

        $registration = $customer->rewardsRegistration;
        $isRegistered = !is_null($registration);

        $rewardsStatus = $customer->rewardsStatus;
        if (!is_null($rewardsStatus)) {
            $spending = $rewardsStatus;
        } else {
            $spending = $productCategoryGroups->map(function ($item) {
                $item->spending = 0;
                $item->qualifying = 0;
            });
        }

        $rewardsTier = $customer->rewardsTier;
        if (!is_null($rewardsTier)) {
            $tierLevel = $rewardsTier->tier;
            $tierName = rewards()->tierMap($rewardsTier->tier);
        }

        return [
            'tierLevel' => $tierLevel,
            'tierName' => $tierName,
            'registered' => $isRegistered,
            'eligible' => $customer->rewardsEligibility == Customer::REWARDS_ELIGIBLE,
            'qualified' => $tierLevel > 1,
            'spending' => $spending,
        ];
    }

    public function tierUpgradeSQL($singleAccount = false) {
        $singleAccountString = '';
        if ($singleAccount) {
            $singleAccountString = ' AND Customers.id = :customer_id';
        }
        $sql = <<<SQLEND
INSERT IGNORE INTO CustomerRewardsTiers (id, customer_id, first_qualified_date, latest_qualified_date, tier, total_spending, category_mix, commit_sequence)
SELECT
  UuidToBin(UUID()),
  newTier.customer_id,
  COALESCE(
      IF(newTier.tier = currentTier.tier, currentTier.first_qualified_date, DATE(NOW())),
      DATE(NOW())
  ) as first_qualified_date,
  DATE(NOW()) as latest_qualified_date,
  newTier.tier as tier,
  newTier.total_spending as total_spending,
  newTier.category_mix as category_mix,
  :commit_sequence as commit_sequence
FROM (SELECT
        CustomerRewardsStatus.customer_id,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7)
          THEN (
            CASE
            WHEN (
              (SUM(qualifying) >= 5 AND SUM(spending) >= 801) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 2001) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 5000)
            )
              THEN 4
            WHEN (
              (SUM(qualifying) >= 5) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 801) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 2001)
            )
              THEN 3
            WHEN (
              (SUM(qualifying) >= 3) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 801)
            )
              THEN 2
            ELSE 1
            END
          )
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (18)
          THEN (
            CASE
            WHEN (
              (SUM(qualifying) >= 5 AND SUM(spending) >= 2501) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 6001) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 14000)
            )
              THEN 4
            WHEN (
              (SUM(qualifying) >= 5) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 2501) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 6001)
            )
              THEN 3
            WHEN (
              (SUM(qualifying) >= 3) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 2501)
            )
              THEN 2
            ELSE 1
            END
          )
        ELSE 0
        END                                       AS tier,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7, 18)
          THEN SUM(spending)
        ELSE 0
        END                                       AS total_spending,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7, 18)
          THEN SUM(qualifying)
        ELSE 0
        END                                       AS category_mix,
        CASE WHEN CustomerRewardsRegistration.id IS NOT NULL
          THEN 1
        ELSE 0 END                                AS isRegistered,
        COALESCE(qualifiedCustomers.qualified, 0) AS isQualified
      FROM CustomerRewardsStatus
        JOIN Customers ON (Customers.id = CustomerRewardsStatus.customer_id)
        LEFT JOIN CustomerOverrides ON (CustomerOverrides.customer_id = Customers.id)
        LEFT JOIN CustomerRewardsRegistration
          ON (CustomerRewardsRegistration.customer_id = CustomerRewardsStatus.customer_id)
        LEFT JOIN (
                    SELECT
                      Customers.id AS customer_id,
                      CASE WHEN SUM(affiliateGroups.notDirectBilled) = 0
                        THEN 1
                      ELSE 0 END   AS qualified
                    FROM Customers
                      JOIN CustomerRewardsRegistration ON (CustomerRewardsRegistration.customer_id = Customers.id)
                      JOIN CustomerSegments ON (CustomerSegments.customer_id = Customers.id)
                      LEFT JOIN CustomerOverrides ON (CustomerOverrides.customer_id = Customers.id)
                      LEFT JOIN (
                                  SELECT
                                    aff_customer_id,
                                    SUM(CASE WHEN CustomerSegments.billto_customer_id != CustomerSegments.customer_id
                                      THEN 1
                                        ELSE 0 END) notDirectBilled
                                  FROM Customers
                                    JOIN CustomerSegments ON (CustomerSegments.customer_id = Customers.id)
                                  WHERE
                                    CustomerSegments.active = 1 AND
                                    CustomerSegments.billto_customer_id = CustomerSegments.customer_id
                                  GROUP BY aff_customer_id
                                ) affiliateGroups
                        ON (affiliateGroups.aff_customer_id = COALESCE(Customers.aff_customer_id, Customers.id))
                    WHERE
                      CustomerSegments.active = 1 AND
                      CustomerSegments.billto_customer_id = CustomerSegments.customer_id AND
                      COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (7, 14, 18) AND
                      Customers.member_group NOT IN ('5653107', 'TX1554') AND
                      CustomerSegments.def_price_list IN ('Catalog', 'Tier 1', 'Tier 2', 'Tier 3', 'Tier 4', 'VolumeDisc') AND 
                      Customers.currency = 'USD'$singleAccountString
                    GROUP BY Customers.id
                  ) qualifiedCustomers ON (qualifiedCustomers.customer_id = Customers.id)
      GROUP BY CustomerRewardsStatus.customer_id) newTier
LEFT JOIN CustomerRewardsTiers currentTier ON (currentTier.customer_id = newTier.customer_id)
WHERE
  newTier.isQualified = 1 AND
  (currentTier.tier IS NULL OR currentTier.tier <= newTier.tier)
ON DUPLICATE KEY UPDATE
  latest_qualified_date = DATE(NOW()),
  tier = VALUES(tier),
  total_spending = VALUES(total_spending),
  category_mix = VALUES(category_mix),
  commit_sequence = VALUES(commit_sequence)
;
SQLEND;
        return $sql;
    }

    public function tierDowngradeSQL($singleAccount = false) {
        $singleAccountString = '';
        if ($singleAccount) {
            $singleAccountString = ' AND Customers.id = :customer_id';
        }

        $sql = <<<SQLEND
INSERT INTO CustomerRewardsTiers (id, customer_id, first_qualified_date, latest_qualified_date, tier, total_spending, category_mix, commit_sequence)
  SELECT
    UuidToBin(UUID()),
    newTier.customer_id,
    DATE(NOW()) as first_qualified_date,
    DATE(NOW()) as latest_qualified_date,
    newTier.tier as tier,
    newTier.total_spending as total_spending,
    newTier.category_mix as category_mix,
    :commit_sequence as commit_sequence
  FROM (SELECT
        CustomerRewardsStatus.customer_id,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7)
          THEN (
            CASE
            WHEN (
              (SUM(qualifying) >= 5 AND SUM(spending) >= 801) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 2001) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 5000)
            )
              THEN 4
            WHEN (
              (SUM(qualifying) >= 5) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 801) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 2001)
            )
              THEN 3
            WHEN (
              (SUM(qualifying) >= 3) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 801)
            )
              THEN 2
            ELSE 1
            END
          )
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (18)
          THEN (
            CASE
            WHEN (
              (SUM(qualifying) >= 5 AND SUM(spending) >= 2501) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 6001) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 14000)
            )
              THEN 4
            WHEN (
              (SUM(qualifying) >= 5) OR
              (SUM(qualifying) >= 3 AND SUM(spending) >= 2501) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 6001)
            )
              THEN 3
            WHEN (
              (SUM(qualifying) >= 3) OR
              (SUM(qualifying) >= 1 AND SUM(spending) >= 2501)
            )
              THEN 2
            ELSE 1
            END
          )
        ELSE 0
        END                                       AS tier,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7, 18)
          THEN SUM(spending)
        ELSE 0
        END                                       AS total_spending,
        CASE
        WHEN COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN (14, 7, 18)
          THEN SUM(qualifying)
        ELSE 0
        END                                       AS category_mix,
        CASE WHEN CustomerRewardsRegistration.id IS NOT NULL
          THEN 1
        ELSE 0 END                                AS isRegistered,
        COALESCE(qualifiedCustomers.qualified, 0) AS isQualified
      FROM CustomerRewardsStatus
        JOIN Customers ON (Customers.id = CustomerRewardsStatus.customer_id)
        JOIN CustomerRewardsRegistration ON (CustomerRewardsRegistration.customer_id = CustomerRewardsStatus.customer_id)
        LEFT JOIN CustomerOverrides ON (CustomerOverrides.customer_id = Customers.id)
        LEFT JOIN (
                    SELECT
                      Customers.id AS customer_id,
                      CASE WHEN SUM(affiliateGroups.notDirectBilled) = 0
                        THEN 1
                      ELSE 0 END   AS qualified
                    FROM Customers
                      JOIN CustomerSegments ON (CustomerSegments.customer_id = Customers.id)
                      LEFT JOIN CustomerOverrides ON (CustomerOverrides.customer_id = Customers.id)
                      LEFT JOIN (
                                  SELECT
                                    aff_customer_id,
                                    SUM(CASE WHEN CustomerSegments.billto_customer_id != CustomerSegments.customer_id
                                      THEN 1
                                        ELSE 0 END) notDirectBilled
                                  FROM Customers
                                    JOIN CustomerSegments ON (CustomerSegments.customer_id = Customers.id)
                                  WHERE
                                    CustomerSegments.active = 1 AND
                                    CustomerSegments.billto_customer_id = CustomerSegments.customer_id
                                  GROUP BY aff_customer_id
                                ) affiliateGroups
                        ON (affiliateGroups.aff_customer_id = COALESCE(Customers.aff_customer_id, Customers.id))
                    WHERE
                      CustomerSegments.active = 1 AND
                      CustomerSegments.billto_customer_id = CustomerSegments.customer_id AND
                      COALESCE(CustomerOverrides.customercategory_id, Customers.customercategory_id) IN
                      (7, 14, 18) AND
                      Customers.member_group NOT IN ('5653107', 'TX1554') AND
                      CustomerSegments.def_price_list IN ('Catalog', 'Tier 1', 'Tier 2', 'Tier 3', 'Tier 4', 'VolumeDisc') AND
                      Customers.currency = 'USD'$singleAccountString
                    GROUP BY Customers.id
                  ) qualifiedCustomers ON (qualifiedCustomers.customer_id = Customers.id)
      GROUP BY CustomerRewardsStatus.customer_id) newTier
JOIN CustomerRewardsTiers currentTier ON (currentTier.customer_id = newTier.customer_id)
WHERE
  currentTier.tier > newTier.tier
  AND currentTier.latest_qualified_date < DATE(DATE_SUB(NOW(), INTERVAL 6 MONTH))
;
SQLEND;
        return $sql;
    }
}