<?php

namespace Hilco\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use DB;
use Auth;
use Elasticquent\ElasticquentTrait;
use Hilco\Facades\HilcoAssets;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Request;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Class WebPart
 * @package Hilco\Models
 * @property int $id
 * @property string $name
 * @property string $part_detail
 * @property string $safe_name
 * @property string $override_name
 * @property-read Part $part
 * @property-read WebPartFamily $family
 * @property-read WebPartAttributeValue[] $attributeValues
 * @method static Builder latest()
 * @property string $part_number
 * @property integer $part_id
 * @property integer $webpartfamily_id
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property-read \Hilco\Models\WebPartFamily $webPartFamily
 * @property-write mixed $web_part_attributes
 * @property-read mixed $part_name
 * @property-read mixed $family_name
 * @property-read mixed $context
 * @property-read mixed $default_context
 * @mixin \Eloquent
 * @property integer $webfamily_id
 * @method static WebPart firstOrNew(array $attributes)
 */
class WebPart extends WebModel
{
    protected $table = "WebParts";
    protected $fillable = ['min_quantity', 'quantity_step', 'quantity_presets', 'name', 'webAttributes', 'keywords', 'webfamily_id', 'part_id', 'part_number', 'is_visible', 'is_family_image', 'in_stock_label', 'out_of_stock_label', 'en_name', 'de_name', 'fr_name'];
    protected $casts = [
        ['is_visible' => 'boolean' , 'is_family_image' => 'boolean']
    ];
    protected $with = ['translations'];

    use AlgoliaEloquentTrait;

    public $algoliaSettings = [
        // Order here determines the order in which records are returned.
        // Attributes at the front of the list are displayed first.
        // e.g. Matches to Part Number will be returned before matches to DE Name.
        // The order here was requested by Germany on 2018-07-23.
        // The request was to have part number matches display before name matches.
        'searchableAttributes' => [
            'part_number',
            'nodash_part_number',
            'de_name',
            'en_name',
            'fr_name',
            'name',
            'is_substitute_for',
        ],
        'attributesForFaceting' => [
            'webSiloIds',
            'validPlants',
            'is_rx',
        ],
        'ranking' => [
            'asc(part_number)',
            'asc(nodash_part_number)',
            'typo',
            'geo',
            'words',
            'filters',
            'proximity',
            'attribute',
            'exact',
        ]
    ];

    public static $perEnvironment = true;

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

//        DB::enableQueryLog();
        static::has('webFamily.webCollections.webCategories.webGroups')->whereHas('part', function ($query) {
            $query->whereHas('inventoryItems', function ($query) {
                $query->isValid();
            });
        })
        ->where('is_visible', 1)
        ->with('part.inventoryItems', 'webFamily.webCollections.webCategories.webGroups')->chunk(100, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
            /** @var \AlgoliaSearch\Index $index */
//            dd(DB::getQueryLog());
//            DB::flushQueryLog();
            foreach ($indicesTmp as $index) {
                $records         = [];
                $recordsAsEntity = [];

                foreach ($models as $model) {
                    if ($modelHelper->indexOnly($model, $index->indexName)) {
                        $records[] = $model->getAlgoliaRecordDefault($index->indexName);

                        if ($onInsert && is_callable($onInsert)) {
                            $recordsAsEntity[] = $model;
                        }
                    }
//                    dd(DB::getQueryLog());
                }

                $index->addObjects($records);

                if ($onInsert && is_callable($onInsert)) {
                    call_user_func_array($onInsert, [$recordsAsEntity]);
                }
            }

        });

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }
    }

    public function indexOnly() {
        if (!array_get($this, 'is_visible', false)) return false;
        if (!array_get($this, 'webFamily.is_visible', false)) return false;
        if (!array_get($this, 'part.web_visible', false)) return false;
        if (!array_get($this, 'part.web_orderable', false)) return false;

        return true;
    }

    public function getAlgoliaRecord() {
        $record = array_only($this->toArray(), [
            'id',
            'part_id',
            'name',
            'part_number',
            'webFamily',
        ]);
        $record['nodash_part_number'] = str_replace('-', '', $this->part_number);

//        $paths = $this->hierarchyPaths;
//        $silos = [];
//        foreach ($paths as $path => $info) {
//            $silos[$info['webSilo']->id] = $info['webSilo']->id;
//        }

        $record['webSiloIds'] = [];
        foreach (WebSilo::visible()->get() as $webSilo) {
            if ($this->visibleInSilo($webSilo)) $record['webSiloIds'][] = $webSilo->id;
        }
        $record['webFamily']['slug'] = !is_null($this->webFamily) ? $this->webFamily->slug : null;
        $record['validPlants'] = !is_null($this->part) && !is_null($this->part->inventoryItems) ? $this->part->inventoryItems->pluck('plant') : [];
        $record['webCollectionIds'] = !is_null($this->webFamily) ? $this->webFamily->webCollections->pluck('id') : [];
        $record['is_rx'] = $this->part->is_rx;
        $record['en_name'] = $this->en_name;
        $record['de_name'] = $this->de_name;
        $record['fr_name'] = $this->fr_name;
        $subsFor = $this->part->substituteForParts->pluck('part_no');
        $record['is_substitute_for'] = $subsFor;
        return $record;
    }

    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

//    use ElasticquentTrait;

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords){
        $params=[
            'multi_match' => [
                'query'     => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields'    => ['_all']
            ]
        ];
        return $params;
    }

//	use KeyWordsFunctional;

	public static function clearIsFamilyImage($familyId, $notIn=[]){
		self::where('webfamily_id', $familyId)->where('is_family_image',1)->whereNotIn('id',$notIn)->update(['is_family_image'=>0]);
	}

    /**
     * @description index all the entries in an Eloquent model (if needed)
     */
    public static function elasticIndexingIfNeeded(){
//		$elasticClient = ClientBuilder::create()->setHosts(config('elasticquent.config.hosts'))->build();
//		$instance = new static;

//		$basicParams = $instance->getBasicEsParams();
//		$indexParams = ['index' => $basicParams['index']];
//
//		$result = $elasticClient->indices()->exists($indexParams);
//		if (!$result)

//		self::createIndex();
        $mappingExists = self::mappingExists();
        if (!$mappingExists) {
            self::createIndex();
            self::putMapping();
            self::addAllToIndex();
        }

//		if(!$result || !$mappingExists)
//		return (!$result || !$mappingExists);
	}
	
	public function part() {
		return $this->belongsTo(Part::class, 'part_id', 'id');
	}

	/**
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function webFamily() {
		return $this->belongsTo(WebFamily::class, 'webfamily_id', 'id');
	}

	public function scopeJoinDiscount($query){
		$customerId = Auth::user()->customer_id;
		return $query->select(['WebParts.*',DB::raw('SUM(CustomerDiscounts.disc_val) as disc_val')])->leftJoin('CustomerDiscounts', function($query)use($customerId){
			return $query->on('CustomerDiscounts.part_id','=','WebParts.id')
						 ->orOn('CustomerDiscounts.productfamily_id','=','WebParts.webfamily_id')
						 ->where('CustomerDiscounts.customer_id','=',$customerId);
		})->groupBy('id');
	}

	public function scopeIsFamilyImage($query){
		$query->where('is_family_image',1)->visible()->whereHas('part', function ($query) {
            return $query->whereHas('inventoryItems', function ($query) {
                return $query->isValid();
            });
        });
	}

	public function scopePriceList($query, $priceList, $currency, $quantityLevel = 1) {
		return $query
			->leftJoin('PriceLists', 'PriceLists.part_no', '=', 'WebParts.part_number')
			->where('price_list', $priceList)
			->where('currency', $currency)
			->where('quantity_level', '>=', $quantityLevel)
			->select('WebParts.*', 'PriceLists.price')
		;
	}

	public function webAttributes() {
		return $this->belongsToMany(WebAttribute::class, 'WebAttribute_WebPart', 'webpart_id', 'webattribute_id')
			->withPivot(['id','attribute_value']);
	}

	public function webAttributeValues() {
	    return $this->hasMany(WebAttribute_WebPart::class, 'webpart_id', 'id');
    }

    public function hasWebAttributeValue($webAttributeId) {
        return $this->webAttributeValues->where('webattribute_id', $webAttributeId)->count() > 0;
    }

	public function webAttributeValue($webAttributeId) {
	    $webAttributeValue = $this->webAttributeValues->where('webattribute_id', $webAttributeId)->first();
	    return $webAttributeValue->attribute_value;
//		if (!$this->relationLoaded('webAttributes')) $this->load('webAttributes');
//		$webAttributeValue = $this->webAttributes->find($webAttributeId);
//		if (!is_null($webAttributeValue)) return $webAttributeValue->pivot->attribute_value;
//		return null;
	}

	public function setWebAttributeValue($webAttributeId, $value) {
		$this->webAttributes()->sync([$webAttributeId => ['attribute_value' => $value]], false);
	}
	public function setWebAttributesAttribute($attributes) {
		$this->save();
		foreach ($attributes as $webAttributeId => $attributeValues) {
		    $this->webAttributes()->syncWithoutDetaching([$webAttributeId]);
		    $webAttributeValue = $this->webAttributeValues()->where('webattribute_id', $webAttributeId)->first();
            if (array_has($attributeValues, 'en')) $webAttributeValue->en_value = $attributeValues['en'];
            if (array_has($attributeValues, 'de')) $webAttributeValue->de_value = $attributeValues['de'];
            if (array_has($attributeValues, 'fr')) $webAttributeValue->fr_value = $attributeValues['fr'];
            $webAttributeValue->save();
        }
	}

    public function relatedParent() {
        return $this->morphToMany(WebFamily::class, 'related', 'WebFamily_Related', 'related_id', 'webfamily_id')->withPivot(['related_category']);
    }

	public function getPartNameAttribute() {
		$part = $this->part;
		if (is_null($part)) return null;
		return $part->part_desc;
	}
	
	public function getFamilyNameAttribute() {
		$family = $this->webFamily;
		if (is_null($family)) return null;
		return $family->name;
	}

	public function getListPriceAttribute() {
	    return $this->part->list_price;
    }

    public function getCustomerPriceAttribute() {
        return $this->part->customer_price;
    }

    public function getCustomerPrice($quantity = 1) {
        return $this->part->getCustomerPrice($quantity);
    }

    public function customerPrices() {
        return $this->part->customerPrices();
    }

    public function getCustomerDiscount() {
	    return $this->part->getCustomerDiscount();
    }

    public function getImageLink($width = null, $height = null, $avoidCache = false) {
        return $this->primaryImage($width ? $width : false, $height ? $height : false);
    }

//	public static function getImageUrl($partNumber, $width = null, $height = null, $avoidCache=false) {
//		$url = config('hilco.images.url');
//		if(!isset($url)) $url = config('hilco-b2b.images.url');
//		$url .= '/' . $partNumber;
//		if ($width) $url .= "/$width";
//		if ($height) $url .= "/$height";
//
//		return ($avoidCache)?\ImgHelper::avoidCache($url):$url;
//	}

    public function getImageUrl($width = false, $height = false) {
        return HilcoAssets::part($this, $width, $height);
    }

	public function scopePriceRange($query, $min, $max) {
		return $query->whereHas('part', function($part) use ($min, $max){
			$part->whereBetween('list_price', [$min, $max]);
		});
	}

	public function scopeManagedBy($query, $manager) {
		return $query->whereHas('part', function ($query) use ($manager) {
			return $query->where('pf_prod_mgr', $manager);
		});
	}

	public function scopeInFamily($query, $webFamilyId) {
		return $query->whereHas('webFamily', function ($query) use ($webFamilyId) {
			return $query->where('webfamily_id', $webFamilyId);
		});
	}

    public function scopeInCollection($query, $webCollectionId) {
        return $query->whereHas('webFamily', function ($query) use ($webCollectionId) {
            return $query->inCollection($webCollectionId);
        });
    }

    public function scopeInCategory($query, $webCategoryId) {
        return $query->whereHas('webFamily', function ($query) use ($webCategoryId) {
            return $query->inCategory($webCategoryId);
        });
    }

	public function scopeInGroup($query, $webGroupId) {
		return $query->whereHas('webFamily', function ($query) use ($webGroupId) {
			return $query->inGroup($webGroupId);
		});
	}

	public function scopeHasVisibleChildren(Builder $query, $activeWebSilo = false, $activeCountry = false, $activePlant = false) {
	    return $query
            ->whereHas('part', function ($query) use ($activeWebSilo, $activeCountry, $activePlant) {
                return $query->visibleToActiveUser($activeWebSilo, $activeCountry, $activePlant);
            })
        ;
    }

    public function scopeHasVisibleParents(Builder $query, $activeWebSilo = false) {
        return $query
            ->whereHas('webFamily', function ($query) use ($activeWebSilo) {
                return $query->hasVisibleParents($activeWebSilo);
            })
            ;
    }

    public function scopeIsVisible(Builder $query) {
        return $query->where('WebParts.is_visible', '1');
    }

	public function scopeWebVisible(Builder $query) {
	    return $query
            ->isVisible()
            ->hasVisibleParents()
            ->hasVisibleChildren()
        ;
    }

    public function scopeWebSiloApproved(Builder $query) {
        return $query
            ->whereHas('part', function ($partQuery) {
                return $partQuery->webSiloApproved();
            })
        ;
    }

	public function scopeVisible(Builder $query, $webSiloId = false) {
	    if ($webSiloId === false) {
	        $webSiloId = b2b()->activeWebSilo()->id;
        }
		return $query
			->where('WebParts.is_visible', '1')
			->whereHas('part', function ($query) use ($webSiloId) {

                if (!b2b()->activeWebSilo()->is_default) {
                    $query->whereHas('webSilos', function ($query) {
                        return $query->where('websilo_id', '=', b2b()->activeWebSilo()->id);
                    });
                }

                if (!b2b()->activeWebSilo()->allow_private_label) {
                    $query->where('private_lbl', '0');
                }

//				$query->whereHas('inventoryItems', function ($query) use ($webSiloId) {
//                    $activeCountry = b2b()->activeCountry();
//                    if ($activeCountry == 'US') $query->inPlant('PLAINVILLE');
//                    return $query->inSilo($webSiloId);
//				});
                
                return $query;
			})
		;
	}

    public function getIsRXAttribute() {
	    $part = $this->part;
	    if (is_null($part)) return null;

        return (in_array($part->product_family_no, Part::$rxProductFamilyNumbers));
    }

	public function visibleInSilo($webSilo) {
	    if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);
//	    $webSiloId = ($webSilo instanceof WebSilo) ? $webSilo->id : $webSilo;
        $inSilo = true;

        if ($webSilo->limit_parts) {
            $found = false;
            foreach (array_get($this->part, 'webSilos', []) as $partWebSilo) {
                if (array_get($partWebSilo, 'id') == $webSilo->id) {
                    $found = true;
                }
            }
            if (!$found) $inSilo = false;
        }

        if ($this->private_lbl && !$webSilo->allow_private_label) {
            $inSilo = false;
        }

//        foreach ($this->part->inventoryItems as $inventoryItem) {
//            foreach ($webSilo->plants as $plant) {
//                $flags = explode(',', $plant->pivot->web_flags);
//                $inPlant = $inventoryItem->plant_id == $plant->id;
//                $hasFlag = in_array($inventoryItem->web_flag, $flags);
//                $isNotObsolete = ($inventoryItem->part_status != 'Phase-out' || $inventoryItem->max_available_qty > 0);
//                $inSilo = $inPlant && $hasFlag && $isNotObsolete;
//            }
//
//        }
        return $inSilo;
    }

    public function getHasValidInventoryAttribute() {
        $status = 'none';
        $hasAll = true;
        if ($this->part) {
            foreach ($this->part->inventoryItems as $inventoryItem) {
                if ($inventoryItem->is_valid) {
                    $status = 'some';
                } else {
                    $hasAll = false;
                }
            }
        }

        if ($hasAll) $status = 'all';
        return $status;
    }

    public function getHierarchyPathsAttribute() {
        $isVisible = true;
        if (!$this->is_visible) $isVisible = trans('hilco::app.manually_hidden');
        if (is_null($this->part)) $isVisible = trans('hilco::app.not_in_point_man');
        if (is_null($this->webFamily)) $isVisible = trans('hilco::app.not_in_a_family');
        if ($this->has_valid_inventory == 'none') $isVisible = trans('hilco::app.no_valid_inventory');

        $paths = [];
        if (!is_null($this->webFamily)) {
            foreach ($this->webFamily->webCollections as $webCollection) {
                foreach ($webCollection->webCategories as $webCategory) {
                    foreach ($webCategory->webGroups as $webGroup) {
                        foreach ($webGroup->webHierarchies as $webHierarchy) {
                            if (!$this->webFamily->is_visible)
                                $isVisible = trans('hilco::app.parent_family_not_visible');
                            if (!$webCollection->is_visible)
                                $isVisible = trans('hilco::app.parent_collection_not_visible');
                            if (!$webCategory->is_visible)
                                $isVisible = trans('hilco::app.parent_category_not_visible');
                            if (!$webGroup->is_visible) $isVisible = trans('hilco::app.parent_group_not_visible');
                            $paths[] = [
                                'webHierarchy' => $webHierarchy,
                                'webGroup' => $webGroup,
                                'webCategory' => $webCategory,
                                'webCollection' => $webCollection,
                                'webFamily' => $this->webFamily,
                                'isVisible' => $isVisible,
                            ];
                        }
                    }
                }
            }
        }
        return $paths;
    }

    public function hierarchyPaths() {
        $paths = ['webPart' => $this];
        if ($this->webFamily) $paths = $this->webFamily->hierarchyPaths($paths);
        return $paths;
    }


    public function setWebfamilyIdAttribute($value) {
        if (!$value || $value < 1) $this->attributes['webfamily_id'] = null;
        else $this->attributes['webfamily_id'] = $value;
    }

    public function getInStockAttribute() {
        return $this->part->in_stock;
    }

    public function getInStockLabelAttribute() {
//        $modelValue = array_get($this->attributes, 'in_stock_label');
//        return strlen($modelValue) ? $modelValue : trans('app.in_stock');
        return trans('app.in_stock');
    }

    public function getOutOfStockLabelAttribute() {
//        $modelValue = array_get($this->attributes, 'out_of_stock_label');
//        return strlen($modelValue) ? $modelValue : trans('app.out_of_stock');
        return trans('app.out_of_stock');
    }

    public function getStockStatusAttribute() {
        return $this->part->stock_status;
    }

    public function getStockLabelAttribute() {
        $status = $this->stock_status;
        switch ($status) {
            case Part::IN_STOCK:
                return $this->in_stock_label;
            case Part::OUT_OF_STOCK:
                return $this->out_of_stock_label;
            case Part::DELAYED:
                return "Ships in 3-5 days";
        }
    }

    public function getPrimaryImage() {
	    $webAsset = $this->assetsByType('primary')->first();
	    if (is_null($webAsset)) $webAsset = $this->partAsset;

	    return $webAsset;
    }

    public function partAsset() {
	    return $this->hasOne(WebAsset::class, 'name', 'part_number')->where('asset_category', 'part')->orderBy('date_created', 'DESC');
    }

    public function getIsSiloPartAttribute() {
        foreach (array_get($this, 'part.webSilos', []) as $partWebSilo) {
            if ($partWebSilo->id == b2b()->activeWebSilo()->id) return true;
        }
        return false;
    }


    public function getRequireApprovalAttribute() {
        return $this->part->require_approval;
    }

    public function scopeSearchWithSubstitutes($query, $partNumber) {
	    return $query->where(function ($query) use ($partNumber) {
	        return $query
                ->where('part_number', $partNumber)
                ->orWhereHas('part', function ($query) use ($partNumber) {
                    return $query->whereHas('substituteForParts', function ($query) use ($partNumber) {
                        return $query->where('PartSubstitutions.part_no', $partNumber);
                    });
                })
            ;
        });
    }

    public function getQuantityPresetsArrayAttribute() {
        $presets = $this->quantity_presets;
        if (strlen($presets) < 1) return [];

        $array = explode(',', $presets);
        return array_combine($array, $array);
    }

    public function slugUrl() {
        $groupId = Request::get('g', false);
        $categoryId = Request::get('c', false);
        $collectionId = Request::get('l', false);
        $params = [];
        if ($groupId) $params['g'] = $groupId;
        if ($categoryId) $params['c'] = $categoryId;
        if ($collectionId) $params['l'] = $collectionId;
        return route('family.slug', array_merge([$this->webFamily->slug, $this->id], $params));
    }

    use HasAssets, RevisionableTrait, HasTranslations;

}
