<?php

namespace Hilco\Models;

    use Faker\Provider\Uuid;
    use Hilco\Facades\HilcoAssets;
    use Illuminate\Database\Eloquent\Relations\Relation;
    use Illuminate\Http\UploadedFile;
    use Illuminate\Support\Facades\Storage;
    use Symfony\Component\HttpFoundation\File\File;
    use Symfony\Component\HttpFoundation\FileBag;

class WebAsset extends WebModel {
    protected $table = 'WebAssets';
    protected $fillable = ['asset_key', 'asset_category', 'original_filename', 'mime_type', 'name', 'description'];
    protected $appends = ['full_key', 'url', 'urls'];

    protected static $morphMap = [
        'webPart' => WebPart::class,
        'webFamily' => WebFamily::class,
        'webCollection' => WebCollection::class,
        'webCategory' => WebCategory::class,
        'webGroup' => WebGroup::class,
    ];

    protected static $mimeTypeNames = [
        'image/jpg' => 'Image',
        'image/jpeg' => 'Image',
        'image/png' => 'Image',
        'image/bmp' => 'Image',
        'application/pdf' => 'PDF',
    ];

    protected static $assetCategories = [
        'part' => 'Part Images',
        'image' => 'Other Images',
        'banner' => 'Banners',
        'document' => 'Documents/Forms',
    ];

    protected static $assetableJoinTypes = [
        'primary' => 'Primary Image',
        'thumbnail' => 'Thumbnail Image',
        'additional' => 'Additional Image',
        'featured' => 'Featured Image',
        'banner' => 'Banner',
    ];

    public static function getMorphMap() {
        return self::$morphMap;
    }

    public static function getAssetableClass($morphMapKey) {
        return array_get(self::$morphMap, $morphMapKey);
    }

    public static function getAssetableModel($morphMapKey) {
        $className = self::getAssetableClass($morphMapKey);
        return ($className) ? new $className : null;
    }


    public static function getAssetCategory($category = false) {
        return $category !== false  ? array_get(self::$assetCategories, $category) : self::$assetCategories;
    }

    public static function getAssetableJoinType($type = false) {
        return $type !== false  ? array_get(self::$assetableJoinTypes, $type) : self::$assetableJoinTypes;
    }

    public static function getMimeType($type = false) {
        return $type !== false ? array_get(self::$mimeTypeNames, $type) : self::$mimeTypeNames;
    }

    public function webParts() {
        return $this->morphedByMany(WebPart::class, 'assetable', 'Assetables', 'webasset_id', 'assetable_id')->withPivot(['sequence', 'assetable_join_type']);
    }

    public function webFamilies() {
        return $this->morphedByMany(WebFamily::class, 'assetable', 'Assetables', 'webasset_id', 'assetable_id')->withPivot(['sequence', 'assetable_join_type']);
    }

    public function getFullKeyAttribute() {
        return $this->asset_category . '/' . $this->asset_key;
    }

    public function getUrlAttribute() {
        return WebAsset::getSizedUrl($this);
    }

    public function getDownloadUrlAttribute() {
        return WebAsset::urlHelper(['id' => $this->id, 'download' => true]);
    }

    public function getUrlsAttribute() {
        $urls['original'] = WebAsset::getSizedUrl($this);
        foreach (config('hilco.assets.sizes') as $size => $dimensions) {
            $urls[$size] = WebAsset::getSizedUrl($this, $dimensions['width'], $dimensions['height']);
        }
        return $urls;
    }

    public function getIconAttribute() {
        if (starts_with($this->mime_type, 'image')) return 'fa-file-image-o';
        if ($this->mime_type == 'application/pdf') return 'fa-file-pdf-o';
        return 'fa-file-o';
    }

    public function __get($key) {
        if (starts_with($key, 'url_')) {
            return self::getPredefinedSizeUrl($this, str_replace('url_', '', $key));
        }
        return parent::__get($key);
    }

    public function getFileAttribute() {
        Storage::disk('assets')->get($this->full_key);
    }

    public static function compileUrlForKey($fullKey, $width = false, $height = false) {
        $api = config('hilco.assets.url');
        $url = "$api/$fullKey";
        if ($width) $url .= "/$width";
        if ($height) $url .= "/$height";
        return $url;
    }

    public static function compileUrlForStub($stub) {
        $api = config('hilco.assets.url');
        $spacer = starts_with($stub, '/') ? '' : '/';
        return $api . $spacer . $stub;
    }

    public static function getPredefinedSizeUrl($webAsset, $size, $keyIfNotFound = false) {
        $size = config("hilco.assets.sizes.$size", [
            'width' => false,
            'height' => false,
        ]);
        return self::getSizedUrl($webAsset, $size['width'], $size['height'], $keyIfNotFound);
    }

    public static function getSizedUrl($webAsset, $width = false, $height = false, $keyIfNotFound = false) {
        if (!($webAsset instanceof WebAsset)) $webAsset = WebAsset::find($webAsset);
        if ($keyIfNotFound === false) $keyIfNotFound = config('hilco.assets.nopic');
        return self::compileUrlForKey(is_null($webAsset) ? $keyIfNotFound : $webAsset->full_key, $width, $height);
    }

    public static function getUrl($webAsset, $keyIfNotFound = false) {
        return self::getSizedUrl($webAsset, false, false, $keyIfNotFound);
    }

    public static function urlHelper($options) {
        if (!is_array($options)) {
            $options = ['prefix' => 'id', 'key' => $options];
        }
        $url = config('hilco.assets.url');
        if (array_has($options, 'id')) {
            $options['prefix'] = 'id';
            $options['key'] = $options['id'];
        }
        $url .= '/' . array_get($options, 'prefix', 'image');
        $url .= '/' . array_get($options, 'key');
        if (array_has($options, 'size')) {
            $size = config("hilco.assets.sizes.$options[size]", false);
            if ($size) {
                $options['width'] = $size['width'];
                $options['height'] = $size['height'];
            }
        }
        if (array_get($options, 'width')) $url .= '/' . array_get($options, 'width');

        if (array_get($options, 'download')) $url .= '?dl=1';

        return $url;
    }

    public function scopeCategory($query, $category) {
        return $query->where('asset_category', $category);
    }

//    public function scopeContentType($query, $type) {
//        return $query->where('content_type', $type);
//    }

    protected static function boot() {
        parent::boot();

        static::deleted(function ($webAsset) {
            Storage::disk('assets')->delete($webAsset->full_key);
        });
    }
}