<?php

namespace Hilco\Models;
use DB;
use Auth;
use Elasticquent\ElasticquentTrait;
use Hilco\Facades\HilcoAssets;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Class WebPart
 * @package Hilco\Models
 * @property int $id
 * @property string $name
 * @property string $part_detail
 * @property string $safe_name
 * @property string $override_name
 * @property-read Part $part
 * @property-read WebPartFamily $family
 * @property-read WebPartAttributeValue[] $attributeValues
 * @method static Builder latest()
 * @property string $part_number
 * @property integer $part_id
 * @property integer $webpartfamily_id
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property-read \Hilco\Models\WebPartFamily $webPartFamily
 * @property-read \Illuminate\Database\Eloquent\Collection|\Hilco\Models\Image[] $images
 * @property-read \Illuminate\Database\Eloquent\Collection|\Hilco\Models\WebPartAttribute[] $webPartAttributes
 * @property-write mixed $web_part_attributes
 * @property-read mixed $part_name
 * @property-read mixed $family_name
 * @property-read \Illuminate\Database\Eloquent\Collection|\Hilco\Models\WebContext_WebPart[] $contexts
 * @property-read mixed $context
 * @property-read mixed $default_context
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart whereId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart wherePartNumber($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart wherePartId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart whereWebpartfamilyId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart whereDateCreated($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart whereDateModified($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart whereDateUploaded($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart whereDeletedAt($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart priceRange($min, $max)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart managedBy($manager)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart inFamily($webPartFamilyId)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebModel joinContext($contextId = false)
 * @mixin \Eloquent
 * @property integer $webfamily_id
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart whereName($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebPart whereWebfamilyId($value)
 * @method static WebPart firstOrNew(array $attributes)
 */
class WebPart extends WebModel
{
    protected $table = "WebParts";
    protected $context_attributes = ['is_visible'];
    protected $context_table = 'WebContext_WebPart';
    protected $context_foreign_key = 'webpart_id';
    protected $fillable = ['name', 'webAttributes', 'webfamily_id', 'part_number', 'is_visible', 'is_family_image'];
    protected $casts = [
        ['is_visible' => 'boolean' , 'is_family_image' => 'boolean']
    ];

    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

    use ElasticquentTrait;

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords){
        $params=[
            'multi_match' => [
                'query'     => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields'    => ['_all']
            ]
        ];
        return $params;
    }
	
	public static function clearIsFamilyImage($familyId, $notIn=[]){
		self::where('webfamily_id', $familyId)->where('is_family_image',1)->whereNotIn('id',$notIn)->update(['is_family_image'=>0]);
	}

    /**
     * @description index all the entries in an Eloquent model (if needed)
     */
    public static function elasticIndexingIfNeeded(){
//		$elasticClient = ClientBuilder::create()->setHosts(config('elasticquent.config.hosts'))->build();
//		$instance = new static;

//		$basicParams = $instance->getBasicEsParams();
//		$indexParams = ['index' => $basicParams['index']];
//
//		$result = $elasticClient->indices()->exists($indexParams);
//		if (!$result)

//		self::createIndex();
        $mappingExists = self::mappingExists();
        if (!$mappingExists) {
            self::createIndex();
            self::putMapping();
            self::addAllToIndex();
        }

//		if(!$result || !$mappingExists)
//		return (!$result || !$mappingExists);
	}
	
	public function part() {
		return $this->belongsTo(Part::class, 'part_id', 'id');
	}

	/**
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function webFamily() {
		return $this->belongsTo(WebFamily::class, 'webfamily_id', 'id');
	}

	public function images() {
		return $this->hasMany(Image::class, 'part_number', 'part_number');
	}

	public function scopeJoinDiscount($query){
		$customerId = Auth::user()->customer_id;
		return $query->select(['WebParts.*',DB::raw('SUM(CustomerDiscounts.disc_val) as disc_val')])->leftJoin('CustomerDiscounts', function($query)use($customerId){
			return $query->on('CustomerDiscounts.part_id','=','WebParts.id')
						 ->orOn('CustomerDiscounts.productfamily_id','=','WebParts.webfamily_id')
						 ->where('CustomerDiscounts.customer_id','=',$customerId);
		})->groupBy('id');
	}

	public function scopeIsFamilyImage($query){
		$query->where('is_family_image',1)->has('images');
	}

	public function scopePriceList($query, $priceList, $currency, $quantityLevel = 1) {
		return $query
			->leftJoin('PriceLists', 'PriceLists.part_no', '=', 'WebParts.part_number')
			->where('price_list', $priceList)
			->where('currency', $currency)
			->where('quantity_level', '>=', $quantityLevel)
			->select('WebParts.*', 'PriceLists.price')
		;
	}

	public function webAttributes() {
		return $this->belongsToMany(WebAttribute::class, 'WebAttribute_WebPart', 'webpart_id', 'webattribute_id')
			->withPivot(['id','attribute_value']);
	}

	public function webAttributeValue($webAttributeId) {
		if (!$this->relationLoaded('webAttributes')) $this->load('webAttributes');
		$webAttributeValue = $this->webAttributes->find($webAttributeId);
		if (!is_null($webAttributeValue)) return $webAttributeValue->pivot->attribute_value;
		return null;
	}

	public function setWebAttributeValue($webAttributeId, $value) {
		$this->webAttributes()->sync([$webAttributeId => ['attribute_value' => $value]], false);
	}

	public function setWebAttributesAttribute($attributes) {
		$this->save();
		foreach ($attributes as $webAttributeId => $attributeValue) {
			$this->setWebAttributeValue($webAttributeId, $attributeValue);
		}
	}

	public function getPartNameAttribute() {
		$part = $this->part;
		if (is_null($part)) return null;
		return $part->part_desc;
	}
	
	public function getFamilyNameAttribute() {
		$family = $this->webFamily;
		if (is_null($family)) return null;
		return $family->name;
	}

    public function getImageLink($width = null, $height = null, $avoidCache = false) {
        return $this->primaryImage($width ? $width : false, $height ? $height : false);
    }

//	public static function getImageUrl($partNumber, $width = null, $height = null, $avoidCache=false) {
//		$url = config('hilco.images.url');
//		if(!isset($url)) $url = config('hilco-b2b.images.url');
//		$url .= '/' . $partNumber;
//		if ($width) $url .= "/$width";
//		if ($height) $url .= "/$height";
//
//		return ($avoidCache)?\ImgHelper::avoidCache($url):$url;
//	}

    public function getImageUrl($width = false, $height = false) {
        return HilcoAssets::part($this, $width, $height);
    }

	public function scopePriceRange($query, $min, $max) {
		return $query->whereHas('part', function($part) use ($min, $max){
			$part->whereBetween('list_price', [$min, $max]);
		});
	}

	public function scopeManagedBy($query, $manager) {
		return $query->whereHas('part', function ($query) use ($manager) {
			return $query->where('pf_prod_mgr', $manager);
		});
	}

	public function scopeInFamily($query, $webFamilyId) {
		return $query->whereHas('webFamily', function ($query) use ($webFamilyId) {
			return $query->where('webfamily_id', $webFamilyId);
		});
	}

	public function scopeInGroup($query, $webGroupId) {
		return $query->whereHas('webFamily', function ($query) use ($webGroupId) {
			return $query->inGroup($webGroupId);
		});
	}

	public function scopeVisible(Builder $query, $webSiloId = false) {
	    if ($webSiloId === false) {
	        $webSiloId = WebSilo::currentId();
        }
		return $query
			->where('WebParts.is_visible', '1')
			->whereHas('part', function ($query) use ($webSiloId) {
				return $query->whereHas('inventoryItems', function ($query) use ($webSiloId) {
					return $query->inSilo($webSiloId);
				});
			})
		;
	}

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webFamily->webCollections as $webCollection) {
            foreach ($webCollection->webCategories as $webCategory) {
                foreach ($webCategory->webGroups as $webGroup) {
                    foreach ($webGroup->webSilos as $webSilo) {
                        $paths[] = [
                            'webSilo' => $webSilo,
                            'webGroup' => $webGroup,
                            'webCategory' => $webCategory,
                            'webCollection' => $webCollection,
                            'webFamily' => $this->webFamily,
                        ];
                    }
                }
            }
        }
        return $paths;
    }


    public function setWebfamilyIdAttribute($value) {
        if (!$value || $value < 1) $this->attributes['webfamily_id'] = null;
        else $this->attributes['webfamily_id'] = $value;
    }

	use HasAssets, RevisionableTrait;

}
