<?php

namespace Hilco\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Input;
use Venturecraft\Revisionable\RevisionableTrait;
use Storage;

/**
 * Class WebFamily
 * @package Hilco\Models
 * @method static WebFamily firstOrNew(array $attributes)
 */
class WebFamily extends WebModel implements HasLandingPageInterface, HasWebSiloOverridesInterface
{
    protected $table = "WebFamilies";
    protected $fillable = ['name', 'slug', 'detail', 'webCollectionIds', 'webAttributeIds', 'is_visible', 'search_only', 'sort_priority', 'best_seller', 'label'];
    protected $perPage = 24;
    protected $casts = [
        ['is_visible' => 'boolean'],
    ];
    protected $with = ['webSiloOverride'];
    protected $hierarchyParent = WebCollection::class;

//	use KeyWordsFunctional;
    use AlgoliaEloquentTrait;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
        ],
        'attributesForFaceting' => [
            'webSiloIds',
            'is_rx',
        ]
    ];

    public static $perEnvironment = true;

    protected static $labelStyles = [
        'SOS' => 'label-info',
        'Hilco Original' => 'label-info',
        'New' => 'label-danger',
        'Sale' => 'label-warning',
        'Best Seller' => 'label-success',
        'Clearance' => 'label-warning',
    ];

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

        static
            ::has('webCollections.webCategories.webGroups')
            ->whereHas('webParts', function ($query) {
                $query->where('is_visible', 1)->has('part');
            })
            ->with('webParts.part')
            ->with('webCollections.webCategories.webGroups')
            ->chunk(100, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
                /** @var \AlgoliaSearch\Index $index */
                foreach ($indicesTmp as $index) {
                    $records         = [];
                    $recordsAsEntity = [];

                    foreach ($models as $model) {
                        if ($modelHelper->indexOnly($model, $index->indexName)) {
                            $records[] = $model->getAlgoliaRecordDefault($index->indexName);

                            if ($onInsert && is_callable($onInsert)) {
                                $recordsAsEntity[] = $model;
                            }
                        }
                    }

                    $index->addObjects($records);

                    if ($onInsert && is_callable($onInsert)) {
                        call_user_func_array($onInsert, [$recordsAsEntity]);
                    }
                }

            });

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }

    }

    public function getAlgoliaRecord() {
        $record = array_only($this->toArray(), [
            'id',
            'name',
            'slug',
            'keywords'
        ]);
        $record['webSiloIds'] = [];
        foreach (WebSilo::visible()->get() as $webSilo) {
            if ($this->visibleInSilo($webSilo)) $record['webSiloIds'][] = $webSilo->id;
        }

        $hasNonRX = false;
        foreach ($this->webParts as $webPart) {
            if ($webPart->part && !$webPart->part->is_rx) {
                $hasNonRX = true;
                break;
            }
        }
        $record['is_rx'] = !$hasNonRX;
        return $record;
    }

    public function algoliaTable() {
        return $this->hasMany(WebFamily_Algolia::class, 'webFamily_id', 'id');
    }

    public function getAlgoliaClassAttribute() {
        return WebFamily_Algolia::class;
    }

    public function getAlgoliaIDFieldAttribute() {
        return 'webFamily_id';
    }

    public function getPerPage()
    {
        $tmpPerPage =Input::get('perPage');
        return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
    }
    /**
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function webParts() {
        return $this->hasMany(WebPart::class, 'webfamily_id', 'id');
    }

    public function webFamily_Algolia()
    {
        return $this->hasMany(WebFamily_Algolia::class, 'webFamily_id', 'id');
    }

    public function webDetailTabs() {
        return $this->belongsToMany(WebDetailTab::class, 'WebDetailTab_WebFamily', 'webfamily_id', 'webdetailtab_id')->withPivot('is_visible');
    }
    public function visibleWebDetailTabs() {
        return $this->webDetailTabs()->wherePivot('is_visible',1);
    }

    public function relatedParent() {
        return $this->morphToMany(WebFamily::class, 'related', 'WebFamily_Related', 'related_id', 'webfamily_id')->withPivot(['related_category']);
    }

    public function relatedFamilies() {
        return $this->morphedByMany(WebFamily::class, 'related', 'WebFamily_Related', 'webfamily_id', 'related_id')->withPivot(['related_category']);
    }

    public function relatedParts() {
        return $this->morphedByMany(WebPart::class, 'related', 'WebFamily_Related', 'webfamily_id', 'related_id')->withPivot(['related_category']);
    }

    public function favoritedWebUsers() {
        return $this
            ->belongsToMany(WebUser::class, 'WebFamily_WebUser', 'webfamily_id', 'webuser_id')
            ->without('customer.segments', 'customer.discounts');
    }

    public function isFavorite() {
        return $this->favoritedWebUsers->where('id', auth()->id())->count() ? true : false;
    }

    public function getRelatedAttribute() {
        return $this->relatedFamilies->where('pivot.related_category', 'related');
    }

    public function getAccessoriesAttribute() {
        $result = $this->relatedFamilies->where('pivot.related_category', 'accessory');
        $result->merge($this->relatedParts->where('pivot.related_category', 'accessory'));
        return $result->sortBy('name');
    }

    public function getAlsoLikeAttribute() {
        $result = $this->relatedFamilies->where('pivot.related_category', 'also_like');
        $result->merge($this->relatedParts->where('pivot.related_category', 'also_like'));
        return $result->sortBy('name');
    }

    public function alsoLikeFamilies($limit = 4) {
        $currentWebCollection = $this->webCollections->first();
        $currentWebCategory = ($currentWebCollection) ? $currentWebCollection->webCategories->first() : false;
        $currentWebGroup = ($currentWebCategory) ? $currentWebCategory->webGroups->first() : false;

        $notIn = [$this->id];

        $alsoLike = $this->also_like;
        if (is_null($alsoLike)) $alsoLike = new Collection;
        $alCount = $alsoLike->count();
        if ($alCount) $alsoLike = $alsoLike->random(min($limit,$alCount));

        if ($alCount < 4) {
            $newItems = ($currentWebGroup)
                ? WebFamily::whereNotIn('id', $notIn)->inGroup($currentWebGroup)->webVisible()->inRandomOrder()->limit($limit-$alCount)->get()
                : WebFamily::whereNotIn('id', $notIn)->webVisible()->inRandomOrder()->limit($limit-$alCount)->get()
            ;
            $alsoLike = $alsoLike->merge($newItems);
        }

        return $alsoLike->shuffle();
    }


    /**
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function visibleWebParts() {
        return $this
            ->hasMany(WebPart::class, 'webfamily_id', 'id')
            ->webVisible()
            ->with(['part' => function ($query) {
                $query->with('productFamily.productCategory.productCategoryGroup');
                $query->with('customerPrices');
                $query->with('webSilos');
                $query->with(['inventoryItems' => function ($query) {
                    $query->inSilo();
                }]);
            }])
            ->with(['assets' => function ($query) {
                return $query->wherePivot('assetable_join_type', 'primary');
            }])
            ->with(['webAttributes'])
            ->with(['partAsset'])
//            ->with(['customerPrices'])
            ->orderBy('name');
    }

    /**
     * Stripped down version of the visibleWebParts query, currently intended for use in fetching of results for
     *  itemReOrder page.
     * @return mixed
     */
    public function visibleWebPartsCondensed(){
        return $this
            ->hasMany(WebPart::class, 'webfamily_id', 'id')
            ->webVisible()
            ->orderBy('name');
    }


    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

    /**
     * @return BelongsToMany
     */
    public function webCollections() {
        return $this->belongsToMany(WebCollection::class, 'WebFamily_WebCollection', 'webfamily_id', 'webcollection_id')->withPivot(['is_featured', 'display_order']);
    }

    /**
     * @return BelongsToMany
     */
    public function webAttributes() {
        return $this
            ->belongsToMany(WebAttribute::class, 'WebAttribute_WebFamily', 'webfamily_id', 'webattribute_id')
            ->withPivot(['is_visible', 'display_order', 'is_thumbnail', 'is_filterable', 'is_select'])
            ->orderBy('pivot_display_order');
    }

    public function getWebAttributeIdsAttribute() {
        return $this->webAttributes->pluck('id')->toArray();
    }

    public function getWebCollectionIdsAttribute() {
        return $this->webCollections->pluck('id')->toArray();
    }

    public function setWebAttributeIdsAttribute($ids) {
        $this->save();
        $this->createRevisionRecord('webAttributeIds', $this->webAttributes->pluck('id')->toJson(), json_encode($ids));
        $this->webAttributes()->sync($ids);
    }

    public function setWebCollectionIdsAttribute($ids) {
        $this->save();
        $this->createRevisionRecord('webCollectionIds', $this->webCollections->pluck('id')->toJson(), json_encode($ids));
        $this->webCollections()->syncWithoutDetaching($ids);
    }

//	public function getThumbnailsAttribute() {
//		$attributes = [];
//		foreach ($this->visibleWebParts as $webPart) {
//			foreach ($webPart->webAttributes as $partAttribute) {
//				if ($partAttribute->id == $this->thumbnail_attribute_id) {
//					$attributes[$partAttribute->pivot->attribute_value] = $webPart;
//				}
//			}
//		}
//		return $attributes;
//	}
//

    public static function deduplicatedList() {
        $webFamilies = WebFamily::with('webCollections.webCategories.webGroups.webHierarchies')->get()->sortBy('name')->keyBy('id');
        $output = [];
        foreach ($webFamilies as $id => $webFamily) {
            foreach ($webFamily->hierarchyPaths as $path) {
                $string = $webFamily->name;
                $string .= ' (' . $path['webCollection']->name;
                $string .= ' ->' . $path['webCategory']->name;
                $string .= ' -> ' . $path['webGroup']->name;
                $string .= ' -> ' . $path['webHierarchy']->name . ')';
                $output[$id] = $string;
                break;
            }
        }
        return collect($output);
    }

    public function getPriceRangeAttribute() {
        if (is_null($this->visibleWebParts)) return null;
        $min = PHP_INT_MAX;
        $max = 0;
        foreach ($this->visibleWebParts as $webPart) {
            $min = min($min, $webPart->customer_price);
            $max = max($max, $webPart->customer_price);
        }
        return [$min, $max];
    }

    public function getPriceAttribute() {
        try {
            if (is_null($this->visibleWebParts)) return null;
            return $this->visibleWebParts->first()->part->prices->first()->price;
        } catch (\Exception $e) {
            return null;
        }
    }

    public function getNameAttribute() {
        return (isset($webSiloOverride) && $webSiloOverride->is_visible == 1) ?
            $this->webSiloOverride->name :
            $this->getTranslation('name')
        ;
    }

    public function getDisplayNameForUACAttribute() {
        return $this->name;
    }

    public function getDetailAttribute () {
        return (isset($webSiloOverride) && $webSiloOverride->is_visible == 1) ?
            $this->webSiloOverride->detail :
            $this->getTranslation('detail')
        ;
    }

    public function getNameWithoutOverrideAttribute() {
        return $this->getTranslation('name');
    }

    public function getDetailWithoutOverrideAttribute() {
        return $this->getTranslation('detail');
    }

    public function visibleInSilo($webSilo) {
        if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);

        foreach ($this->webParts as $webPart) {
            if ($webPart->visibleInSilo($webSilo)) return true;
        }

        return false;
    }

    public function getCustomerPriceAttribute() {
        $part = $this->visibleWebParts->first();
        if (is_null($part)) return null;
        return $part->customer_price;
    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webCollections as $webCollection) {
            foreach ($webCollection->webCategories as $webCategory) {
                foreach ($webCategory->webGroups as $webGroup) {
                    foreach ($webGroup->webHierarchies as $webHierarchy) {
                        $paths[] = [
                            'webHierarchy' => $webHierarchy,
                            'webGroup' => $webGroup,
                            'webCategory' => $webCategory,
                            'webCollection' => $webCollection,
                        ];
                    }
                }
            }
        }
        return $paths;
    }

    public function scopeHasAttributes($query, $filters) {
        if(!isset($filters) || is_null($filters)) return $query;
        $attrs = [];
        foreach ($filters as $filter){
            if(!isset($attrs[$filter->webattribute_id])){
                $attrs[$filter->webattribute_id] = [];
            }
            $attrs[$filter->webattribute_id] = array_merge($attrs[$filter->webattribute_id],$filter->webAttributeFilterValues->pluck('filter_value')->toArray());
        }
        return $query->where(function ($query) use ($attrs) {
            foreach ($attrs as $id=>$values) {
                $query->whereHas('webAttributes', function ($query) use ($id, $values) {
                    return $query->where('webattribute_id', $id);
                });
            }
            return $query->whereHas('visibleWebParts', function ($query) use ($attrs) {
                foreach ($attrs as $id=>$values) {
                    $query = $query->whereHas('webAttributes', function ($query) use ($id, $values) {
                        return $query->where('webattribute_id', $id)->whereIn('attribute_value', $values);
                    });
                }
                return $query;
            });
        });
    }
    public function scopeManagedBy($query, $product_manager) {
        return $query->whereHas('webParts', function($query) use ($product_manager) {
            return $query->managedBy($product_manager);
        });
    }

    public function scopeInCollection($query, $webCollection) {
        $webCollectionId = ($webCollection instanceof WebCollection) ? $webCollection->id : $webCollection;
        return $query->whereHas('webCollections', function ($query) use ($webCollectionId) {
            return $query->where('webcollection_id', $webCollectionId);
        });
    }

    public function scopeInCategory($query, $webCategory) {
        $webCategoryId = ($webCategory instanceof WebCategory) ? $webCategory->id : $webCategory;
        return $query->whereHas('webCollections', function ($query) use ($webCategoryId) {
            return $query->inCategory($webCategoryId);
        });
    }

    public function scopeInGroup($query, $webGroup) {
        $webGroupId = ($webGroup instanceof WebGroup) ? $webGroup->id : $webGroup;
        return $query->whereHas('webCollections', function ($query) use ($webGroupId) {
            return $query->inGroup($webGroupId);
        });
    }

    public function scopeInSilo($query, $webSilo) {
        $webSiloId = ($webSilo instanceof WebSilo) ? $webSilo->id : $webSilo;
        return $query->whereHas('webCollections', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }

    public function scopeFilterCorrectly($query, $model) {
        if ($model instanceof WebGroup) {
            $webGroupId = $model->id;
            return $query->where('WebGroups.id', '=', $webGroupId);
        } else if ($model instanceof WebCategory) {
            $webCategoryId = $model->id;
            $defaultWebGroup = $model->webGroups->first();

            if (empty(session()->get('breadcrumb')) || empty(session()->get('breadcrumb.groupId'))) {
                $webGroupId = $defaultWebGroup->id;
            } else {
                $webGroupId = session()->get('breadcrumb.groupId');
            }

            return
                $query
                    ->where('WebCategories.id', '=', $webCategoryId)
                    ->where('WebGroups.id', '=', $webGroupId)
            ;
        } else if ($model instanceof WebCollection){
            $webCollectionId = $model->id;
            $defaultWebCategory = $model->webCategories->first();
            $defaultWebGroup = $defaultWebCategory->webGroups->first();

            if (empty(session()->get('breadcrumb')) || empty(session()->get('breadcrumb.categoryId'))) {
                $webCategoryId = $defaultWebCategory->id;
            } else {
                $webCategoryId = session()->get('breadcrumb.categoryId');
            }

            if (empty(session()->get('breadcrumb')) || empty(session()->get('breadcrumb.groupId'))) {
                $webGroupId = $defaultWebGroup->id;
            } else {
                $webGroupId = session()->get('breadcrumb.groupId');
            }

            return
                $query
                    ->where('WebCollections.id', '=', $webCollectionId)
                    ->where('WebCategories.id', '=', $webCategoryId)
                    ->where('WebGroups.id', '=', $webGroupId)
            ;
        } else if ($model instanceof WebSilo) {
            return $query;
        }
    }

    public function scopeIn($query, $model) {
        if ($model instanceof WebGroup) return $query->inGroup($model);
        else if ($model instanceof WebCategory) return $query->inCategory($model);
        else if ($model instanceof WebCollection) return $query->inCollection($model);
        else if ($model instanceof WebSilo) $query->webSiloApproved();
    }

    public static function in($model) {
        if ($model instanceof WebGroup) {return self::inGroup($model);}
        else if ($model instanceof WebCategory) return self::inCategory($model);
        else if ($model instanceof WebCollection) return self::inCollection($model);
        else if ($model instanceof WebSilo) return self::inSilo($model);
        else return self::newQuery();
    }
//
    public function scopeVisible(Builder $query) {
        return $query
            ->where('WebFamilies.is_visible', '1')
            ->whereHas('webParts', function ($query) {
                return $query->visible();
            })
            ->whereHas('webCollections', function ($query) {
                return $query->visible();
            })
            ;
    }

    public function scopeHasVisibleChildren(Builder $query, $activeWebSilo = false, $activeCountry = false, $activePlant = false) {
        return $query
            ->whereHas('webParts', function ($query) use ($activeWebSilo, $activeCountry, $activePlant) {
                return $query->hasVisibleChildren($activeWebSilo, $activeCountry, $activePlant);
            })
        ;
    }

    public function scopeHasVisibleParents(Builder $query, $activeWebSilo = false) {
        return $query
            ->whereHas('webCollections', function ($query) use ($activeWebSilo) {
                return $query->hasVisibleParents($activeWebSilo);
            })
        ;
    }

    public function scopeIsVisible(Builder $query) {
        return $query->where('WebFamilies.is_visible', '1');
    }

    public function scopeWebVisible(Builder $query) {
        return $query
//            ->where('WebFamilies.is_visible', '1')
//            ->whereHas('webParts', function ($webPartsQuery) {
//                return $webPartsQuery->webVisible();
//            })
            ->isVisible()
            ->hasVisibleParents()
            ->hasVisibleChildren()
        ;
    }

    public function scopeWebSiloApproved(Builder $query) {
        return $query->whereHas('webParts', function ($webPartsQuery) {
            return $webPartsQuery->webSiloApproved();
        });
        ;
    }

    public function scopeForDisplay(Builder $query, $slug) {
        return $query
            ->webVisible()
            ->with(['visibleWebParts' => function ($query) {
                $query->with(['assets', 'part' => function ($query) {
                    $query
                        ->with('customerPrices', 'inventoryItems', 'webSilos')
                        ->with('productFamily.productCategory.productCategoryGroup')
                    ;
                }]);
            }, 'assets'])
            ->with('webAttributes')
            ->with(['favoritedWebUsers' => function ($query) {
                $query->where('WebUsers.id', auth()->id());
            }])
            ->slug($slug)
            ;
    }

    public function scopeForBreadcrumbs($query) {
        $query->with([
            'webCollections' => function ($lQuery) {
                if (session('breadcrumb.collectionId')) {
                    $lQuery->where('WebCollections.id', '=', session('breadcrumb.collectionId'));
                }
                $lQuery->with(['webCategories' => function ($cQuery) {
                    if (session('breadcrumb.categoryId')) {
                        $cQuery->where('WebCategories.id', '=', session('breadcrumb.categoryId'));
                    }
                    $cQuery->with(['webGroups' => function ($gQuery) {
                        if (session('breadcrumb.groupId')) {
                            $gQuery->where('WebGroups.id', '=', session('breadcrumb.groupId'));
                        }
                    }]);
                }]);
            }
        ]);
        return $query;
    }

    public function scopeForLandingPage($query, HasLandingPageInterface $lpModel) {
        $query
            ->isVisible()
            ->hasVisibleParents()
            ->hasVisibleChildren()
            ->notForSearchOnly()
            ->in($lpModel)
            ->filterCorrectly($lpModel)
            ->join('WebFamily_WebCollection as WFWC', 'WFWC.webfamily_id', '=', 'WebFamilies.id')
            ->join('WebCollection_WebCategory as WLWC', 'WLWC.webcollection_id', '=', 'WFWC.webcollection_id')
            ->join('WebCategory_WebGroup as WCWG', 'WCWG.webcategory_id', '=', 'WLWC.webcategory_id')
            ->join('WebGroups', 'WebGroups.id', '=', 'WCWG.webgroup_id')
            ->join('WebCategories', 'WebCategories.id', '=', 'WCWG.webcategory_id')
            ->join('WebCollections', 'WebCollections.id', '=', 'WLWC.webcollection_id')
            ->where(function ($where) {
                return $where
                    ->whereNull('WebGroups.deleted_at')
                    ->orWhere('WebGroups.deleted_at', '=', DB::raw(0))
                    ;
            })
            ->where(function ($where) {
                return $where
                    ->whereNull('WebCategories.deleted_at')
                    ->orWhere('WebCategories.deleted_at', '=', DB::raw(0))
                    ;
            })
            ->where(function ($where) {
                return $where
                    ->whereNull('WebCollections.deleted_at')
                    ->orWhere('WebCollections.deleted_at', '=', DB::raw(0))
                    ;
            })
        ;

        $query
            ->orderBy('WebFamilies.sort_priority', 'desc')
            ->orderBy('WebGroups.name', 'asc')
            ->orderBy('WCWG.display_order', 'asc')
            ->orderBy('WebCategories.name', 'asc')
            ->orderBy('WLWC.display_order', 'asc')
            ->orderBy('WebCollections.name', 'asc')
            ->orderBy('WFWC.display_order', 'asc')
            ->orderBy('WebFamilies.name', 'asc')
            ->select(DB::raw('WebFamilies.*'))
            ->with(['visibleWebParts', 'assets'])
            ->with(['favoritedWebUsers' => function ($query) {
                $query->where('WebUsers.id', auth()->id());
            }])
        ;
        return $query;
    }

    public static function getImageUrl($partNumber, $width = null, $height = null) {
        $url = config('hilco.images.url');
        if(!isset($url)) $url = config('hilco-b2b.images.url');
        $url .= '/' . $partNumber;
        if ($width) $url .= "/$width";
        if ($height) $url .= "/$height";
        return $url;
    }


    public function scopeJoinWebPart(Builder $query){
        return $query->leftJoin('WebParts', 'WebParts.webfamily_id', '=', 'WebFamilies.id');
    }

    public function slugUrl() {
        return route('family.slug', [$this->slug]);
    }

    public function scopeForSearchOnly(Builder $query){
        return $query->where('search_only',1);
    }

    public function scopeNotForSearchOnly(Builder $query){
        return $query->where('search_only',0);
    }

    public function scopeBestSellers(Builder $query) {
        return $query->where('best_seller', 1);
    }

    public function getPrimaryImage() {
        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $part = $this->visibleWebParts->where('is_family_image', 1)->first();
        if (!is_null($part)) {
            $asset = $part->getPrimaryImage();
            if (!is_null($asset)) return $asset;
        }

        $part = $this->visibleWebParts->first();
        if (!is_null($part)) {
            $asset = $part->getPrimaryImage();
            if (!is_null($asset)) return $asset;
        }

        return null;
    }

    public function getIsSiloPartAttribute() {
        foreach ($this->visibleWebParts as $webPart) {
            if ($webPart->isSiloPart) return true;
        }
        return false;
    }

    public function getRequireApprovalAttribute() {
        $requireApproval = true;
        if($this->visibleWebParts()->count() == 0){
            return false;
        }
        foreach ($this->visibleWebParts as $webPart) {
            if (!$webPart->require_approval) $requireApproval = false;
        }
        return $requireApproval;
    }

    use HasAssets, HasSlug, RevisionableTrait, HasLandingPage, ManyToManyRevisionableTrait, HasWebSiloOverrides, HasCommitSequence, HasModelTranslations;

    public function getParentRelationName()
    {
        return 'webCollections';
    }

    public function getLabelStyleAttribute() {
        return array_get(self::$labelStyles, $this->label);
    }

    public function getIsWebVisibleAttribute() {
        return static::webVisible()->where('id', $this->id)->count() > 0;
    }

}
