<?php
namespace Hilco\XMLBuilder;

/**
 * Class RateShopUtility
 * Utility class, holds Process Weaver response as SimpleXMLElement object
 * @package Hilco\XMLBuilder
 * The 'simpleXML' class field should hold the Process Weaver RateResponse XML object,
 * which should resemble the following example:
 * <RateResponse>
 *     <ErrorMessage></ErrorMessage>
 *     <OrderNo></OrderNo>
 *     <RateShop xmlns="processRequest">
 *         <Rate>
 *             <Carrier>FedEx</Carrier>
 *             <Service>FEDEX_GROUND</Service>
 *             <CarrierServiceCode>FEDEX_GROUND</CarrierServiceCode>
 *             <PublishedRate>23.23</PublishedRate>
 *             <DiscountedRate>23.23</DiscountedRate>
 *             <BaseServiceCharge>21.97</BaseServiceCharge>
 *             <TransportationCharges>21.97</TransportationCharges>
 *             <ServiceOptionsCharges>1.26</ServiceOptionsCharges>
 *             <Fual>1.26</Fual>
 *             <Transitdays>3</Transitdays>
 *             <EstmatedTime>2020-10-21T12:00:00</EstmatedTime>
 *         </Rate>
 *         <Rate>
 *             ....
 *     </RateShop>
 * </RateResponse>
 */
use Hilco\XMLBuilder\XMLUtility;

class RateShopUtility extends XMLUtility {

    public function __construct($attributes) {
        parent::__construct($attributes);
    }

    /**
     * This is used by the old v4 getRates function.
     */
    public function getRates(){
        $rateValues = [];
        if($this->simpleXML == null){
            $this->simpleXML = simplexml_load_string($this->xml);
        }
        $simpleXML = $this->simpleXML;
        if ($simpleXML != null) {
            $Rates = $simpleXML->RateShop->Rate;
            $rateValues = XMLUtility::retrieveStackValues($Rates, [], [
                'carrier_name' => ['Carrier'],
                'api_code' => ['CarrierServiceCode'],
                'carrier_desc' => ['Service'],
                'rate' => ['PublishedRate'],
                'discounted_rate' => ['DiscountedRate'],
                'delivery_datetime' => ['EstmatedTime'] // Yes, it's mispelled like that at least as of this time (7/26/2019: 3:13 PM)
            ]);
        }

        return $rateValues;
    }


    /**
     * This is used by the new v5 getRates function
     * Return relevant shipping rate information from SimpleXMLElement object as array
     * @return array
     * e.g.,
     * [
     *     [
     *         "carrier" => "FedEx",
     *         "service" => "FEDEX_GROUND",
     *         "serviceCode" => "FEDEX_GROUND",
     *         "publishedRate" => "23.23",
     *         "discountedRate" => "23.23",
     *         "estimatedTime" => "2020-10-21T12:00:00",
     *     ],
     *     ...
     * ]
     */
    public function getPWResponseArray(){
        $rateValues = [];
        if($this->simpleXML == null){
            $this->simpleXML = simplexml_load_string($this->xml);
        }
        $simpleXML = $this->simpleXML;
        if ($simpleXML != null) {
            $Rates = $simpleXML->RateShop->Rate;
            $rateValues =
                XMLUtility::retrieveStackValues(
                    $Rates,
                    [],
                    [
                        'carrier'           => ['Carrier'],
                        'service'           => ['Service'],
                        'serviceCode'       => ['CarrierServiceCode'],
                        'publishedRate'     => ['PublishedRate'],
                        'discountedRate'    => ['DiscountedRate'],
                        'estimatedTime'     => ['EstmatedTime'] // Yes, it's mispelled like that at least as of this time (7/26/2019: 3:13 PM)
                    ]
                )
            ;
        }

        return $rateValues;
    }

}