<?php

namespace Hilco;

use Hilco\Models\Customer;
use Hilco\Models\Part;
use Hilco\Models\WebFamily;
use Hilco\Models\WebPart;
use Hilco\Models\WebSilo;
use Hilco\Models\WebUrl;
use Illuminate\Database\Eloquent\Model;

class B2BHelper {
    protected $activeCustomer;
    protected $activeWebSilo;
    protected $activeShippingAddress;

    public function __construct() {
        $activeShippingAddress = null;
        $this->loadActiveCustomer();
    }

    protected function loadActiveCustomer() {
        $this->activeCustomer = false;

        $aliasedCustomerId = session()->get('aliasedCustomerId', false);
        if ($aliasedCustomerId) {
            $this->activeCustomer = Customer::with('segments.address.plant', 'discounts', 'availableShippingAddresses')->find($aliasedCustomerId);
        } else {
            $user = auth()->user();
            if ($user) $this->activeCustomer = $user->customer;
        }

        $this->setActiveShippingAddress();

        $this->loadActiveWebSilo();
    }

    protected function loadActiveWebSilo() {
        $this->activeWebSilo = false;

        $activeWebSiloId = session()->get('activeWebSiloId', false);

        if (!$activeWebSiloId) {
            if (auth()->user()) {
                $defaultWebSilo = auth()->user()->defaultWebSilo;
                if ($defaultWebSilo) {
                    $activeWebSiloId = $defaultWebSilo->id;
                }
            }
            if (!$activeWebSiloId) {
                $query = WebSilo::with('plants', 'webLandingPage');

                if ($this->activeCustomer) {
                    $cust_cat_summary = $this->activeCustomer->customerCategory->cust_cat_summary;
                    $divisions = $this->activeCustomer->divisions->pluck('id');
                    $query->whereHas('divisionRules', function ($query) use ($divisions) {
                        $query->whereIn('Divisions.id', $divisions);
                    })->whereHas('customerCategorySummaryRules', function ($query) use ($cust_cat_summary) {
                        $query->where('cust_cat_summary', $cust_cat_summary);
                    });
                } else {
                    $webUrl = WebUrl::current()->first();
                    if (!is_null($webUrl)) {
                        $query->whereHas('webUrl', function ($query) use ($webUrl) {
                            $query->where('id', '=', $webUrl->id);
                        });
                    }
                }
                $result = $query->first();
                if (!is_null($result)) $activeWebSiloId = $result->id;
            }

            if (!$activeWebSiloId) $activeWebSiloId = WebSilo::first()->id;
        }
        $this->activeWebSilo = WebSilo::with('plants', 'webLandingPage')->find($activeWebSiloId);
        session()->put('activeWebSiloId', $activeWebSiloId);

        return $this->activeWebSilo;
    }

    public function aliasAs($customerId = false) {
        if ($customerId) {
            session()->put('aliasedCustomerId', $customerId);
        } else {
            session()->forget('aliasedCustomerId');
            session()->forget('activeWebSiloId');
            $this->loadActiveWebSilo();
        }
        $this->loadActiveCustomer();
    }

    public function isAliased() {
        return (session()->get('aliasedCustomerId', false));

    }

    public function stopAliasing() {
        $this->aliasAs(false);
    }

    public function setActiveShippingAddress($customerShippingAddress = false) {
        if ($customerShippingAddress === false) {
            if ($this->activeCustomer) {
                $customerShippingAddress = $this->activeCustomer->default_shipping_address;
            }
        }
        $this->activeShippingAddress = $customerShippingAddress;
        return $this->activeShippingAddress;
    }

    public function activeCustomer() {
        return $this->activeCustomer;
    }

    public function activeShippingAddress() {
        return $this->activeShippingAddress;
    }

    public function activeCountry() {
        if ($this->activeShippingAddress) return $this->activeShippingAddress()->country;

        return null;
    }

    public function activePriceList() {
        $activePriceList = 'Catalog';
        if ($this->activeCustomer) {
            $customerPriceList = $this->activeCustomer->default_price_list;
            if ($customerPriceList) $activePriceList = $customerPriceList;
        }

        return $activePriceList;
    }

    public function activeCurrency() {
        $activeCurrency = 'USD';

        if ($this->activeCustomer) {
            $activeCurrency = $this->activeCustomer->currency;
        }

        return $activeCurrency;
    }

    public function activeWebSilo() {
        return $this->activeWebSilo;
    }

    public function currencySymbol($currencyCode = false) {
        if ($currencyCode === false) $currencyCode = $this->activeCurrency();
        if ($currencyCode == 'GBP') return '&pound;';

        return '$';
    }

    public function isOrderable(Model $model) {
        if ($model instanceof WebFamily) return $this->isWebFamilyOrderable($model);
        else if ($model instanceof WebPart) return $this->isWebPartOrderable($model);
        else if ($model instanceof Part) return $this->isPartOrderable($model);

        return false;
    }

    public function isWebFamilyOrderable(WebFamily $webFamily) {
        $isOrderable = true;
        foreach ($webFamily->visibleWebParts as $webPart) {
            if (!$this->isWebPartOrderable($webPart)) $isOrderable = false;
        }
        return $isOrderable;
    }

    public function isWebPartOrderable(WebPart $webPart) {
        $part = $webPart->part;
        if (is_null($part)) return false;
        return $this->isPartOrderable($part);
    }

    public function isPartOrderable(Part $part) {
        if ($this->activeCountry() != 'US') {
            if ($part->isRX) return false;
        }
        return true;
    }
}