<?php

namespace Hilco\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Venturecraft\Revisionable\RevisionableTrait;
use Illuminate\Support\Facades\Input;

class WebCollection extends WebModel implements HasLandingPageInterface, HasWebSiloOverridesInterface {
    protected $table = "WebCollections";
    protected $fillable = ['name', 'slug', 'webCategoryIds', 'is_visible', 'detail', 'sort_priority'];
    protected $casts = [
        ['is_visible' => 'boolean']
    ];
    protected $with = ['webSiloOverride'];
    protected $perPage = 24;

//    use KeyWordsFunctional;
    use AlgoliaEloquentTrait;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
        ],
        'attributesForFaceting' => [
            'webSiloIds',
            'is_rx',
        ]
    ];

    public static $perEnvironment = true;

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

        static
            ::has('webCategories.webGroups')
            ->whereHas('webFamilies', function ($query) {
                $query->whereHas('webParts', function ($query) {
                    $query->where('is_visible', 1)->has('part');
                });
            })
            ->with('webCategories.webGroups')
            ->with('webFamilies.webParts.part')
            ->chunk(100, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
            /** @var \AlgoliaSearch\Index $index */
            foreach ($indicesTmp as $index) {
                $records         = [];
                $recordsAsEntity = [];

                foreach ($models as $model) {
                    if ($modelHelper->indexOnly($model, $index->indexName)) {
                        $records[] = $model->getAlgoliaRecordDefault($index->indexName);

                        if ($onInsert && is_callable($onInsert)) {
                            $recordsAsEntity[] = $model;
                        }
                    }
                }

                $index->addObjects($records);

                if ($onInsert && is_callable($onInsert)) {
                    call_user_func_array($onInsert, [$recordsAsEntity]);
                }
            }

        });

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }

    }

    public function getAlgoliaRecord() {
        $record = array_only($this->toArray(), [
            'id',
            'name',
            'slug',
        ]);

        $record['webSiloIds'] = [];
        foreach (WebSilo::visible()->get() as $webSilo) {
            if ($this->visibleInSilo($webSilo)) $record['webSiloIds'][] = $webSilo->id;
        }

        $hasNonRX = false;
        foreach($this->webFamilies as $webFamily) {
            if ($webFamily->webParts) {
                foreach ($webFamily->webParts as $webPart) {
                    if ($webPart->part && !$webPart->part->is_rx) {
                        $hasNonRX = true;
                        break 2;
                    }
                }
            }
        }
        $record['is_rx'] = !$hasNonRX;
        return $record;
    }

    public function algoliaTable() {
        return $this->hasMany(WebCollection_Algolia::class, 'webCollection_id', 'id');
    }

    public function getAlgoliaClassAttribute() {
        return WebCollection_Algolia::class;
    }

    public function getAlgoliaIDFieldAttribute() {
        return 'webCollection_id';
    }

    public function getIsAlgoliaVisibleAttribute() {
        return static::algoliaVisible()->where('id', $this->id)->count() > 0;
    }

    public function scopeAlgoliaVisible(Builder $query) {
        return
            $query
                ->isVisible()
                ->hasAlgoliaVisibleParents()
                ->hasAlgoliaVisibleChildren()
            ;
    }

    public function scopeHasAlgoliaVisibleChildren(Builder $query) {
        return $query
            ->whereHas('webFamilies', function ($query) {
                return $query->hasAlgoliaVisibleChildren();
            })
        ;
    }

    public function scopeHasAlgoliaVisibleParents(Builder $query) {
        return $query
            ->whereHas('webCategories', function ($query) {
                return $query->hasAlgoliaVisibleParents();
            })
        ;
    }

    public function getSolrRecord($webSilos) {
        $record = [
            'id' => $this->id,
            'name' => $this->name,
            'slug' => $this->slug,
        ];

        foreach ($record as $key => $item) {
            $record[$key."_searchable"] = preg_replace("/[^A-Za-z0-9]/", '', $item);
        }

        $record['webSiloIds'] = [];
        if ($webSilos === null) {
            $webSilos = WebSilo::visible()->get();
        }

        foreach ($webSilos as $webSilo) {
            if ($this->visibleInSilo($webSilo)) $record['webSiloIds'][] = $webSilo->id;
        }

        $hasNonRX = false;
        foreach($this->webFamilies as $webFamily) {
            if ($webFamily->webParts) {
                foreach ($webFamily->webParts as $webPart) {
                    if ($webPart->part && !$webPart->part->is_rx) {
                        $hasNonRX = true;
                        break 2;
                    }
                }
            }
        }
        $record['is_rx'] = !$hasNonRX;

        $record['b2b_web_visible'] = $this->is_solr_visible;

        return $record;
    }

    public function solrTable()
    {
        return $this->hasMany(WebCollection_Solr::class, 'webCollection_id', 'id');
    }

    public function getSolrClassAttribute() {
        return WebCollection_Solr::class;
    }

    public function getSolrIDFieldAttribute() {
        return 'webCollection_id';
    }

    public function getIsSolrVisibleAttribute() {
        return static::solrVisible()->where('id', $this->id)->count() > 0;
    }

    public function scopeSolrVisible(Builder $query) {
        return
            $query
                ->isVisible()
                ->hasSolrVisibleParents()
                ->hasSolrVisibleChildren()
            ;
    }

    public function scopeHasSolrVisibleChildren(Builder $query) {
        return $query
            ->whereHas('webFamilies', function ($query) {
                return $query->hasSolrVisibleChildren();
            })
            ;
    }

    public function scopeHasSolrVisibleParents(Builder $query) {
        return $query
            ->whereHas('webCategories', function ($query) {
                return $query->hasSolrVisibleParents();
            })
            ;
    }

    public function getPerPage()
    {
        $tmpPerPage =Input::get('perPage');
        return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
    }

    public function getDisplayNameForUACAttribute() {
        return $this->attributes['name'];
    }

    public function getNameAttribute() {
        if (config('hilco.ignoreActiveWebSilo')) {
            return $this->getTranslation('name', 'en', $this->attributes['name']);
        } else {
            return (isset($this->webSiloOverride) && $this->webSiloOverride->is_visible == 1) ?
                $this->webSiloOverride->name :
                $this->getTranslation('name');
        }
    }

    public function getNameForEditAttribute() {
        return $this->getTranslation('name', 'en', $this->attributes['name']);
    }

    public function getDetailAttribute() {
        if (config('hilco.ignoreActiveWebSilo')) {
            return $this->getTranslation('detail', 'en', $this->attributes['detail']);
        } else {
            return (isset($this->webSiloOverride) && $this->webSiloOverride->is_visible == 1) ?
                $this->webSiloOverride->detail :
                $this->getTranslation('detail')
                ;
        }
    }

    public function webCategories() {
        $returnMe = $this->belongsToMany(WebCategory::class, 'WebCollection_WebCategory', 'webcollection_id', 'webcategory_id')->withPivot('display_order');
        return $returnMe;
    }

    public function webCollectionTranslations(){
        return $this->hasMany(Translations_WebCollection::class, 'webcollection_id', 'id');
    }

    /**
     * @return BelongsToMany
     */
    public function webFamilies() {
        return $this->belongsToMany(WebFamily::class, 'WebFamily_WebCollection', 'webcollection_id', 'webfamily_id')->withPivot(['is_featured', 'display_order']);
    }

    public function webCollection_Algolia()
    {
        return $this->hasMany(WebCollection_Algolia::class, 'webCollection_id', 'id');
    }

    public function visibleWebFamilies() {
        return $this->belongsToMany(WebFamily::class, 'WebFamily_WebCollection', 'webcollection_id', 'webfamily_id')
            ->withPivot(['is_featured', 'display_order'])
            ->webVisible()
            ->displayOrder()
        ;
    }

    public function visibleInSilo($webSilo) {
        if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);

        foreach ($this->webFamilies as $webFamily) {
            if ($webFamily->visibleInSilo($webSilo)) return true;
        }

        return false;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

    public function getFeaturedChildrenAttribute() {
        return $this->visibleWebFamilies()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        $children = $this->visibleWebFamilies();
        if (request()->get('ap', false) == 1) $children->webSiloApproved();

        return $children->get();
    }

    public function scopeInCategory($query, $webCategoryId) {
        return $query->whereHas('webCategories', function ($query) use ($webCategoryId) {
            $query->where('webcategory_id', $webCategoryId);
        });
    }

    public function scopeHavingFamily($query, $webFamilyId) {
        return $query->whereHas('webFamilies', function ($query) use ($webFamilyId) {
            $query->where('webfamily_id', $webFamilyId);
        });
    }

    public function setWebCategoryIdsAttribute($ids) {
        $this->save();
        $this->createRevisionRecord('webCategoryIds', $this->webCategories->pluck('id')->toJson(), json_encode($ids));
        $this->webCategories()->sync($ids);
    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webCategories as $webCategory) {
            foreach ($webCategory->webGroups as $webGroup) {
                foreach ($webGroup->webHierarchies as $webHierarchy) {
                    $paths[] = [
                        'webHierarchy' => $webHierarchy,
                        'webGroup' => $webGroup,
                        'webCategory' => $webCategory,
                    ];
                }
            }
        }
        return $paths;
    }

    public static function deduplicatedList() {
        $webCollections = WebCollection::with('webCategories.webGroups.webHierarchies')->without('webSiloOverride')->get()->sortBy('name')->keyBy('id');

        $output = [];
        foreach ($webCollections as $id => $webCollection) {
            foreach ($webCollection->hierarchyPaths as $path) {
                $string = $webCollection->name;
                $string .= ' (' . $path['webCategory']->name;
                $string .= ' -> ' . $path['webGroup']->name;
                $string .= ' -> ' . $path['webHierarchy']->name . ')';
                $output[$id] = $string;
                break;
            }
        }
        return $output;
    }

    public function scopeInCollection($query, $webCollection) {
        return $query->whereHas('webCollections', function ($query) use ($webCollection) {
            $query->where('webcollection_id', $webCollection->id);
        });
    }

    public function scopeInGroup($query, $webGroupId) {
        return $query->whereHas('webCategories', function ($query) use ($webGroupId) {
            return $query->inGroup($webGroupId);
        });
    }

    public function scopeInSilo($query, $webSiloId) {
        return $query->whereHas('webCategories', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }    

    public function getAssetPrefix() {
        return "webCollections/{$this->id}";
    }

    public function scopeVisible(Builder $query) {
        $query
            ->where('WebCollections.is_visible', '1')
            ->whereHas('webCategories', function ($query) {
                return $query->visible();
            })
        ;

        return $query;
    }

    public function scopeHasVisibleChildren(Builder $query, $activeWebSilo = false, $activeCountry = false, $activePlant = false) {
        return $query
            ->whereHas('webFamilies', function ($query) use ($activeWebSilo, $activeCountry, $activePlant) {
                return $query->isVisible()->hasVisibleChildren($activeWebSilo, $activeCountry, $activePlant);
            })
        ;
    }

    public function scopeHasVisibleParents(Builder $query, $activeWebSilo = false) {
        return $query
            ->whereHas('webCategories', function ($query) use ($activeWebSilo) {
                return $query->hasVisibleParents($activeWebSilo);
            })
        ;
    }

    public function scopeIsVisible(Builder $query) {
        return $query
            ->where('is_visible', '1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->where('WebSiloOverrides.is_visible', '=', 1);
                    })
                ;
            })
        ;
    }

    public function scopeWebVisible(Builder $query) {
        return $query
            ->where('is_visible', '1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->where('WebSiloOverrides.is_visible', '=', 1);
                    })
                ;
            })
            ->whereHas('webFamilies', function ($webFamiliesQuery) {
                return $webFamiliesQuery->webVisible();
            })
        ;
    }

    public function scopeWebSiloApproved(Builder $query) {
        return $query
            ->whereHas('webFamilies', function ($webFamiliesQuery) {
                return $webFamiliesQuery->webSiloApproved();
            })
            ;
    }

    public function slugUrl() {
        return route('collection.slug', [$this->slug,
            'g'=> session()->get('breadcrumb.groupId') ? session()->get('breadcrumb.groupId') : $this->webCategories->first()->webGroups->first()->id,
            'c' => session()->get('breadcrumb.categoryId') ? session()->get('breadcrumb.categoryId') : $this->webCategories->first()->id,
            'l' => $this->id]);
    }

    public function getParentRelationName() {
        return 'webCategories';
    }

    public function getPrimaryImage() {
        $asset = array_get($this, 'webSiloOverride.primaryImage', null);
        if (!is_null($asset)) return $asset;

        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webFamily = $this->visibleWebFamilies->first();
        if ($webFamily) return $webFamily->getPrimaryImage();
    }

    public function getIsWebVisibleAttribute() {
        return static::webVisible()->where('id', $this->id)->count() > 0;
    }


    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait, ManyToManyRevisionableTrait, HasWebSiloOverrides, HasModelTranslations;
}
