<?php

namespace Hilco\Models;

use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Facades\DB;

/**
 * Class PromotionCustomerTrigger
 * @package Hilco\Models
 */
class PromotionCustomerTrigger extends PromotionTriggerDetails {
    protected $table = 'PromotionCustomerTriggers';
    protected $fillable = ['target_all_customers', 'one_time_use', 'commit_sequence'];

    /**
     * Overrides Model::boot(), used to define a specific 'deleted' state
     */
    public static function boot() {
        parent::boot();

        // need to make sure all related PromotionCustomerTrigger_Customer rows are also deleted
        static::deleted(function (PromotionCustomerTrigger $promotionCustomerTrigger) {
            $promotionCustomerTriggerCustomers =
                PromotionCustomerTrigger_Customer
                    ::where('promotioncustomertrigger_id', '=', $promotionCustomerTrigger->id)
                    ->get()
            ;
            foreach ($promotionCustomerTriggerCustomers as $promotionCustomerTriggerCustomer) {
                $promotionCustomerTriggerCustomer->delete();
            }
        });
    }

    /**
     * Get relationship to list of customers defined in PromotionCustomerTrigger_Customer
     * @return BelongsToMany
     */
    public function customers() {
        return $this->belongsToMany(Customer::class, 'PromotionCustomerTrigger_Customer', 'promotioncustomertrigger_id', 'customer_id')
                    ->wherePivot('deleted_at', '=', DB::raw(0))
                    ->withPivot(['excluded', 'temporary_email', 'redeemed'])
                    ->with(['webUsers', 'contacts'])
            ;
    }

    /**
     * Get relationship to all related PromotionCustomerTrigger_Customer rows
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function promotionCustomerTriggerCustomers() {
        return $this->hasMany(PromotionCustomerTrigger_Customer::class, 'promotioncustomertrigger_id', 'id');
    }

    /**
     * Logic to determine if this promotion trigger is satisfied
     * @param Customer $soldToCustomer
     * @param array $webCartItems
     * @param array $distinctPartInfo
     * @return bool|mixed
     */
    public function isTriggered(Customer $soldToCustomer, $webCartItems, $distinctPartInfo = []) {
        $promotionCustomerTriggerCustomer =
            PromotionCustomerTrigger_Customer
                ::triggerCustomer($this->id, $soldToCustomer->id)
                ->first()
        ;
        $promotionCustomerRedemption =
            PromotionCustomerRedemption
                ::byPromotionCustomer($this->trigger->promotion->id, $soldToCustomer->id)
                ->first()
        ;

        if($this->one_time_use) {
            if ($this->target_all_customers) {
                return is_null($promotionCustomerRedemption);
            } else {
                return isset($promotionCustomerTriggerCustomer) && is_null($promotionCustomerRedemption);
            }
        } else {
            return ($this->target_all_customers == 1 || isset($promotionCustomerTriggerCustomer));
        }
    }

    public function postTriggerOperation($soldToCustomer) {
        $this->customers()->updateExistingPivot($soldToCustomer->id, array('redeemed' => 1));
    }

    public function isTargetingCustomer($customer) {
        if ($this->target_all_customers) {
            return true;
        } else {
            $targetedCustomer = $this->promotionCustomerTriggerCustomers()->where('customer_id', $customer->id)->first();
            return isset($targetedCustomer);
        }
    }
}
