<?php

namespace Hilco\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Facades\DB;


/**
 * Hilco\Models\Customer
 *
 * @property integer $id
 * @property string $cust_no
 * @property string $cust_name
 * @property string $aff_cust_no
 * @property integer $aff_customer_id
 * @property string $aff_cust_name
 * @property string $addr_1
 * @property string $addr_2
 * @property string $addr_3
 * @property string $addr_4
 * @property string $addr_5
 * @property string $city
 * @property string $state
 * @property string $zip_postal
 * @property string $country
 * @property string $bill_addr_1
 * @property string $bill_addr_2
 * @property string $bill_addr_3
 * @property string $bill_addr_4
 * @property string $bill_addr_5
 * @property string $bill_city
 * @property string $bill_state
 * @property string $bill_zip_postal
 * @property string $bill_country
 * @property string $category
 * @property integer $customercategory_id
 * @property string $incept_dt
 * @property string $currency
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property-read \Illuminate\Database\Eloquent\Collection|\Hilco\Models\Division[] $divisions
 * @property-read \Illuminate\Database\Eloquent\Collection|\Hilco\Models\CustomerPhone[] $phones
 * @property-read \Illuminate\Database\Eloquent\Collection|\Hilco\Models\CustomerEmail[] $emails
 * @property-read \Hilco\Models\CustomerCategory $customer_category
 * @property-read \Illuminate\Database\Eloquent\Collection|\Hilco\Models\CustomerContact[] $contacts
 * @property-read \Illuminate\Database\Eloquent\Collection|\Hilco\Models\CustomerDiscount[] $discounts
 * @property-read \Illuminate\Database\Eloquent\Collection|\Hilco\Models\Order[] $orders
 * @property-read mixed $id_string
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereCustNo($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereCustName($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereAffCustNo($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereAffCustomerId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereAffCustName($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereAddr1($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereAddr2($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereAddr3($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereAddr4($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereAddr5($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereCity($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereState($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereZipPostal($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereCountry($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereCategory($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereCustomercategoryId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereInceptDt($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereCurrency($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereDateCreated($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereDateModified($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereDateUploaded($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\Customer whereDeletedAt($value)
 * @mixin \Eloquent
 */
class Customer extends Model
{
	protected $table = 'Customers';
	protected $fillable = ['name', 'country', 'state', 'city', 'zip_postal', 'addr_1', 'addr_2', 'addr_3','currency','cust_name'];

	public static $currencies = [
		'USD'=>'USD',
		'NZD'=>'NZD',
		'JPY'=>'JPY',
		'GBP'=>'GBP',
		'CNY'=>'CNY',
		'CAD'=>'CAD',
		'AUD'=>'AUD'
	];
	const UPDATED_AT = 'date_modified';

    use \Hilco\SoftDeletes, AlgoliaEloquentTrait;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'cust_no',
            'cust_name',
        ],
    ];

    public static $perEnvironment = true;
    public static $objectIdKey = 'cust_no';

    public function getAlgoliaRecord() {
        $record = array_only($this->toArray(), [
            'id',
            'cust_name',
            'cust_no'
        ]);
        return $record;
    }

	public function webUser()
	{
		return $this->hasOne(WebUser::class, 'customer_id');
	}

	public function webUsers() {
        return $this->hasMany(WebUser::class, 'customer_id');
    }

	public function shippingAddresses() {
		return $this->hasMany(CustomerShippingAddress::class, 'cust_no', 'cust_no');
	}

	public function availableShippingAddresses() {
        return $this->hasMany(CustomerShippingAddress::class, 'cust_no', 'cust_no')->hasShipToProfile();
    }

	public function contacts() {
	    return $this->hasMany(Contact::class, 'customer_id', 'id');
    }

    public function customerEmails() {
        return $this->hasMany(CustomerEmail::class, 'customer_id', 'id');
    }

    public function jobsonDatas() {
        return $this->belongsToMany(JobsonData::class, 'Customer_JobsonData', 'customer_id', 'jobsondata_id');
    }

    public function customer_Algolia()
    {
        return $this->hasMany(Customer_Algolia::class, 'customer_id', 'id');
    }

	public static $priority = [
		'A'	=> [
			'from' => 2000,
			'to'   => 9999999
		],
		'B'	=> [
			'from' => 1000,
			'to'   => 1999
		],
		'C'	=> [
			'from' => 400,
			'to'   => 999
		],
		'D'	=> [
			'from' => 0,
			'to'   => 399
		]
	];

    public function getActiveSegmentAttribute() {
        return $this->segments->where('active', 1)->first();
    }

    public function getDefaultShippingAddressAttribute() {
        $segment = $this->active_segment;
        return is_null($segment) ? null : is_null($segment->address) ? $segment->shipping_addresses->first() : $segment->address;
    }

	public function activeSegments()
	{
		return $this->segments()->where('CustomerSegments.active', '=', 1);
	}

	public function pharmacyInfo()
	{
		return $this->hasOne(CustomerPharmacyInfos::class, 'customer_id');
	}

	public function segments() {
		return $this->hasMany(CustomerSegment::class, 'customer_id', 'id');
	}

	public function activeDivisions()
	{
		return $this
            ->belongsToMany(Division::class, 'CustomerSegments', 'customer_id', 'division_id')
            ->where('CustomerSegments.active', '=', 1)
            ->where('CustomerSegments.deleted_at', '=', DB::raw(0));
	}

	public function divisions()
	{
		return $this->belongsToMany(Division::class, 'CustomerSegments', 'customer_id', 'division_id');
	}

	public function phones()
	{
		return $this->hasMany(CustomerPhone::class);
	}

	public function emails()
	{
		return $this->hasMany(CustomerEmail::class);
	}

//	public function customer_category() {
//	    return $this->customerCategory;
//    }

	public function customerCategory()
	{
		return $this->belongsTo(CustomerCategory::class, 'customercategory_id');
	}


	public function discounts()
	{
		return $this->hasMany(CustomerDiscount::class);
	}

	public function customerStateTax(){
		return $this->belongsTo(StateTax::class,'state','state')
			->where(function($query){
				$query->where(function($query){
					return $query->where('effective_date','<=', date('Y-m-d',time()))
						->where('expiration_date','>=', date('Y-m-d',time()));
				})
					->orWhere('expiration_date', '000-00-00');
			});
	}

	public function customerVATTax(){

	}

	public function orders()
	{
		return $this->hasMany(Order::class, 'soldto_customer_id');
	}

	public function salesStats()
	{
		return $this->hasMany(CustomerSalesStats::class, 'customer_id');
	}

	public function stateTax() {
		return StateTax::where("state", $this->state)->inEffect()->first();
	}

	public function scopeWithEmails($query) {
		return $query->addSelect('email_raw', 'good_email', 'email_note', 'email_news', 'email_sales', 'email_ship')
			->join('CustomerEmails', 'Customers.id', '=', 'CustomerEmails.customer_id')
			;
	}

    /**
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
	public function webCustomerSalesSummaries() {
	    return $this->hasMany(WebCustomerSalesSummary::class, 'customer_id', 'id');
    }

    public function webCustomerLimits() {
        return $this->hasMany(WebCustomerLimit::class, 'customer_id', 'id');
    }

    public function recentOrders($count = 5) {
        return $this
            ->orders()
            ->submitted()
            ->with('orderLineSchedules.shipment')
            ->orderBy('order_date', 'desc')
            ->take($count)
            ->get();
    }

    public function topPurchasedItems($count = 5) {
        $ids = WebFamily::webVisible()
            ->join('WebParts', 'WebParts.webfamily_id', '=', 'WebFamilies.id')
            ->join('SalesOrderLines', function($join) {
                $join
                    ->on('SalesOrderLines.part_id', '=', 'WebParts.part_id');
            })
            ->join('SalesOrders', function($join) {
                $join
                    ->on('SalesOrders.id', '=', 'SalesOrderLines.salesorder_id');
            })
            ->leftJoin('SalesOrderLineSchedules', function($join) {
                $join
                    ->on('SalesOrderLineSchedules.salesorder_id', '=', 'SalesOrders.id')
                    ->on('SalesOrderLineSchedules.salesorderline_id', '=', 'SalesOrderLines.id');
            })
            ->where('SalesOrders.deleted_at', '=', DB::raw('0'))
            ->where('SalesOrderLines.deleted_at', '=', DB::raw('0'))
            ->where(function ($where) {
                $where
                    ->whereNull('SalesOrderLineSchedules.deleted_at')
                    ->orWhere('SalesOrderLineSchedules.deleted_at', '=', DB::raw('0'))
                ;
            })
            ->where('SalesOrders.soldto_customer_id', '=', $this->id)
            ->whereNotIn('SalesOrders.order_status', ['CANCELLED', 'DISCARDED'])
            ->groupBy('WebFamilies.id')
            ->orderBy(DB::raw('COUNT(DISTINCT SalesOrders.id)'), 'DESC')
            ->take($count)
            ->pluck('WebFamilies.id')
        ;

        return WebFamily::with('assets', 'visibleWebParts')->findMany($ids);
    }

    public function affiliateCustomer() {
	    return $this->hasOne(Customer::class, 'id', 'aff_customer_id');
    }

    public function affiliateChildren() {
        return $this->hasMany(Customer::class, 'aff_customer_id', 'id');
    }

    public function affiliatedCustomers() {
	    return $this->hasMany(Customer::class, 'aff_customer_id', 'aff_customer_id');
    }

    /**
     * @return BelongsToMany
     */
    public function webSilo() {
	    return $this
            ->belongsToMany(WebSilo::class, 'Customer_WebSilo', 'customer_id', 'websilo_id')
//            ->withPivot(['can_order_hilco_parts', 'can_order_silo_parts', 'require_approval_hilco_parts', 'require_approval_silo_parts'])
            ->withPivot(['spending_limit', 'spending_period', 'minimum_order'])
            ->with('webLandingPageWebSiloJoins')
        ;
    }

    public function getWebSiloAttribute() {
	    $webSilo = $this->getRelationValue('webSilo')->first();
	    if ($webSilo) return $webSilo;
	    return WebSilo::default();
    }

    public function getHasAffiliateAttribute() {
	    return !is_null($this->aff_customer_id);
    }

    public function croakiesAdventuresStatus() {
        return $this->hasMany(CroakiesAdventuresStatus::class, 'customer_id', 'aff_customer_id');
    }

    public function croakiesAdventuresTier() {
        return $this->hasOne(CroakiesAdventuresTiers::class, 'customer_id', 'id')
                    ->orderBy('first_qualified_date', 'desc')
                    ->orderBy('tier', 'desc')
                    ->take(1);
    }

    public function getCroakiesOrdered() {
        $statuses = $this->croakiesAdventuresStatus;
        $croakiesOrdered = 0;

        $lastDateModified = Carbon::now()->toDateString();
        foreach($statuses as $status) {
            $croakiesOrdered = $croakiesOrdered + $status->quantities;
            $lastDateModified = $status->date_modified;
        }

        $affCustomers = $this->affiliatedCustomers;
        $affCustomerIds = [];
        $index = 0;
        foreach ($affCustomers as $affCustomer) {
            $affCustomerIds[$index] = $affCustomer->id;
            $index = $index + 1;
        }

        $croakiesOrders = Order::whereIn('billto_customer_id', $affCustomerIds)
                                ->where('date_created', '>=', $lastDateModified)
                                ->whereIn('order_status', ['PENDING', 'ADDRESS_PENDING', 'PENDING_APPROVAL', 'CLOSED', 'EXPORTING', 'SENT_TO_POINTMAN'])
                                ->where('deleted_at', '=', '0000-00-00 00:00:00')
                                ->get();

        foreach ($croakiesOrders as $order) {
            $mainLines = $order->partLines;
            foreach ($mainLines as $mainLine) {
                $partLine = $mainLine->partLine;
                $quantity = $partLine->quantity;
                $partValue = 1;

                $prepackPart = CroakiesAdventuresPrepackPart::where('part_no', '=', $partLine->part->part_no)->first();
                if ($prepackPart) {
                    $partValue = $prepackPart->no_units;
                }

                $croakiesOrdered = $croakiesOrdered + ($quantity * $partValue);
            }
        }

        return $croakiesOrdered;
    }

    public function rewardsStatus() {
	    return $this->hasMany(CustomerRewardsStatus::class, 'customer_id', 'aff_customer_id');
    }

    public function rewardsTier() {
	    return $this->hasOne(CustomerRewardsTier::class, 'customer_id', 'aff_customer_id')->orderBy('first_qualified_date', 'DESC')->orderBy('tier', 'DESC')->take(1);
    }

    public function rewardsRegistration() {
	    return $this->hasOne(CustomerRewardsRegistration::class, 'customer_id', 'aff_customer_id');
    }

    public function getBillToCustomerAttribute() {
	    $activeSegment = $this->activeSegment;
	    if (is_null($activeSegment)) return null;
	    return $activeSegment->billToCustomer;
    }

    public function getBillsDirectAttribute() {
	    $activeSegment = $this->activeSegment;

	    return is_null($activeSegment) ? false : $this->activeSegment->billsDirect;
    }

    public function getRewardsTierLevelAttribute() {
        $tier = $this->rewardsTier;
        if (!is_null($tier)) {
            return $tier->tier;
        }
        return 0;
    }

    public function customerOverride() {
	    return $this->hasOne(CustomerOverride::class, 'customer_id', 'aff_customer_id');
    }

    public function getOverrideCustomerCategoryAttribute() {
        $relation = null;
	    $override = $this->customerOverride;
	    if (is_null($override)) {
            $relation = $this->customerCategory;
        } else {
            $relation = $override->customerCategory;
        }
        if(is_null($relation)){
	        return $this->customerCategory;
        }
        return $relation;
    }

    public function getProfileAttribute() {
        $profile = null;
        $override = $this->customerOverride;
        if (!is_null($override)) {
            $profile = $this->customerOverride->profile;
        }
        return $profile;
    }

    public function getDefaultPriceListAttribute() {
	    $activeSegment = $this->activeSegment;
	    if ($activeSegment) {
	        return $activeSegment->def_price_list;
        }

        return null;
    }
}
