<?php

namespace Hilco\Models;

use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use Carbon\Carbon;
use DB;
use Auth;
use Hilco\Facades\HilcoAssets;
use HilcoAPI\BODIngesters\TestModels\LegacyParts;
use HilcoB2B\M3Request;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\App;
use Log;
use Venturecraft\Revisionable\RevisionableTrait;

/**
 * Class WebPart
 * @package Hilco\Models
 * @property int $id
 * @property string $name
 * @property string $part_detail
 * @property string $safe_name
 * @property string $override_name
 * @property-read Part $part
 * @property-read WebPartFamily $family
 * @property-read WebPartAttributeValue[] $attributeValues
 * @method static Builder latest()
 * @property string $part_number
 * @property integer $part_id
 * @property integer $webpartfamily_id
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property-read \Hilco\Models\WebPartFamily $webPartFamily
 * @property-write mixed $web_part_attributes
 * @property-read mixed $part_name
 * @property-read mixed $family_name
 * @property-read mixed $context
 * @property-read mixed $default_context
 * @mixin \Eloquent
 * @property integer $webfamily_id
 * @method static WebPart firstOrNew(array $attributes)
 */
class WebPart extends WebModel
{
    protected $table = "WebParts";
    protected $fillable = [
        'min_quantity', 'quantity_step', 'quantity_presets',
        'name', 'webAttributes', 'webfamily_id', 'part_id', 'part_number',
        'keywords', 'is_visible', 'is_family_image',
        'in_stock_label','low_stock_label', 'out_of_stock_label'
    ];
    protected $casts = [
        ['is_visible' => 'boolean' , 'is_family_image' => 'boolean']
    ];

// ***************************************ALGOLIA METHODS***************************************************************
    use AlgoliaEloquentTrait;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
            'part_number',
            'nodash_part_number',
            'legacy_part_no',
            'nodash_legacy_part_no'
        ],
        'attributesForFaceting' => [
            'webSiloIds',
            'validPlants',
            'is_rx',
            'assortments',
        ]
    ];

    public static $perEnvironment = true;

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

//        DB::enableQueryLog();
        static::has('webFamily.webCollections.webCategories.webGroups')->whereHas('part', function ($query) {
            $query->whereHas('inventoryItems', function ($query) {
                $query->isValid();
            });
        })
        ->where('is_visible', 1)
        ->with('part.inventoryItems', 'webFamily.webCollections.webCategories.webGroups')->chunk(100, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
            /** @var \AlgoliaSearch\Index $index */
//            dd(DB::getQueryLog());
//            DB::flushQueryLog();
            foreach ($indicesTmp as $index) {
                $records         = [];
                $recordsAsEntity = [];

                foreach ($models as $model) {
                    if ($modelHelper->indexOnly($model, $index->indexName)) {
                        $records[] = $model->getAlgoliaRecordDefault($index->indexName);

                        if ($onInsert && is_callable($onInsert)) {
                            $recordsAsEntity[] = $model;
                        }
                    }
//                    dd(DB::getQueryLog());
                }

                $index->addObjects($records);

                if ($onInsert && is_callable($onInsert)) {
                    call_user_func_array($onInsert, [$recordsAsEntity]);
                }
            }

        });

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }
    }

    public function indexOnly() {
        if (!array_get($this, 'is_visible', false)) return false;
        if (!array_get($this, 'webFamily.is_visible', false)) return false;

        return true;
    }

    public function getAlgoliaRecord() {
        $record = array_only($this->toArray(), [
            'id',
            'part_id',
            'name',
            'part_number',
            'webFamily',
            'legacy_part_no',
        ]);
        $record['nodash_part_number'] = str_replace('-', '', $this->part_number);
        $record['legacy_part_no'] = $this->part->legacy_part_no;
        $record['nodash_legacy_part_no'] = str_replace('-', '', $this->part->legacy_part_no);

        $record['webSiloIds'] = [];
        foreach (WebSilo::visible()->get() as $webSilo) {
            if ($this->visibleInSilo($webSilo)) $record['webSiloIds'][] = $webSilo->id;
        }
        $record['webFamily']['slug'] = !is_null($this->webFamily) ? $this->webFamily->slug : null;
        $record['validPlants'] = !is_null($this->part) && !is_null($this->part->inventoryItems) ? $this->part->inventoryItems->pluck('plant') : [];
        $record['webCollectionIds'] = !is_null($this->webFamily) ? $this->webFamily->webCollections->pluck('id') : [];
        $record['is_rx'] = $this->part->is_rx;

        $record['assortments'] = [];
        foreach ($this->part->assortments as $assortment) {
            $record['assortments'][] = $assortment->assortment_name;
        }
        if (count($record['assortments']) == 0) {
            $record['assortments'][] = 'NO_ASSORTMENT';
        }

        return $record;
    }

    public function algoliaTable() {
        return $this->hasMany(WebPart_Algolia::class, 'webPart_id', 'id');
    }

    public function getAlgoliaClassAttribute() {
        return WebPart_Algolia::class;
    }

    public function getAlgoliaIDFieldAttribute() {
        return 'webPart_id';
    }

    public function getIsAlgoliaVisibleAttribute() {
        return static::algoliaVisible()->where('id', $this->id)->count() > 0;
    }

    public function scopeAlgoliaVisible(Builder $query) {
        return
            $query
                ->isVisible()
                ->hasAlgoliaVisibleParents()
                ->hasAlgoliaVisibleChildren()
        ;
    }

    public function scopeHasAlgoliaVisibleParents(Builder $query) {
        return
            $query
                ->whereHas('webFamily', function ($query) {
                    return $query->hasAlgoliaVisibleParents();
                })
        ;
    }

    public function scopeHasAlgoliaVisibleChildren(Builder $query) {
        return $query
            ->whereHas('part', function ($query) {
                return $query->hasAlgoliaVisibleChildren();
            })
        ;
    }

    public function webPart_Algolia()
    {
        return $this->hasMany(WebPart_Algolia::class, 'webPart_id', 'id');
    }

    public function getSolrRecord($webSilos) {
        $record = array_only($this->toArray(), [
            'id',
            'part_id',
            'name',
            'part_number',
            'webFamily',
            'legacy_part_no',
        ]);
        $record['nodash_part_number'] = str_replace('-', '', $this->part_number);
        $record['legacy_part_no'] = $this->part->legacy_part_no;
        $record['nodash_legacy_part_no'] = str_replace('-', '', $this->part->legacy_part_no);
        foreach ($record as $key => $item) {
            $record[$key."_searchable"] = preg_replace("/[^A-Za-z0-9]/", '', $item);
        }

        $record['webSiloIds'] = [];
        if ($webSilos === null) {
            $webSilos = WebSilo::visible()->get();
        }
        foreach ($webSilos as $webSilo) {
            if ($this->visibleInSilo($webSilo)) $record['webSiloIds'][] = $webSilo->id;
        }

        $record['webFamily_slug'] = !is_null($this->webFamily) ? $this->webFamily->slug : null;
        $record['webCollectionIds'] = !is_null($this->webFamily) ? $this->webFamily->webCollections->pluck('id')->toArray() : [];
        $record['is_rx'] = $this->part->is_rx;
        $record['validPlants'] = !is_null($this->part->inventoryItems) ? $this->part->inventoryItems->pluck('plant')->toArray() : [];
        $record['b2b_web_visible'] = $this->is_solr_visible;

        $record['assortments'] = [];
        foreach ($this->part->assortments as $assortment) {
            $record['assortments'][] = $assortment->assortment_name;
        }
        if (count($record['assortments']) == 0) {
            $record['assortments'][] = 'NO_ASSORTMENT';
        }

        return $record;
    }

    public function solrTable()
    {
        return $this->hasMany(WebPart_Solr::class, 'webPart_id', 'id');
    }

    public function getSolrClassAttribute() {
        return WebPart_Solr::class;
    }

    public function getSolrIDFieldAttribute() {
        return 'webPart_id';
    }

    public function getIsSolrVisibleAttribute() {
        return static::solrVisible()->where('id', $this->id)->count() > 0;
    }

    public function scopeSolrVisible(Builder $query) {
        return
            $query
                ->isVisible()
                ->hasSolrVisibleParents()
                ->hasSolrVisibleChildren()
            ;
    }

    public function scopeHasSolrVisibleParents(Builder $query) {
        return
            $query
                ->whereHas('webFamily', function ($query) {
                    return $query->hasSolrVisibleParents();
                })
            ;
    }

    public function scopeHasSolrVisibleChildren(Builder $query) {
        return $query
            ->whereHas('part', function ($query) {
                return $query->hasSolrVisibleChildren();
            })
            ;
    }
// *********************************************************************************************************************

// ***************************************ELASTICSEARCH METHODS (PROBABLY UNUSED)***************************************
    protected $indexSettings = [
        'analysis' => [
            'char_filter' => [
                'replace' => [
                    'type' => 'mapping',
                    'mappings' => [
                        '&=> and '
                    ],
                ],
            ],
            'filter' => [
                'word_delimiter' => [
                    'type' => 'word_delimiter',
                    'split_on_numerics' => false,
                    'split_on_case_change' => false,
                    'generate_word_parts' => true,
                    'generate_number_parts' => true,
                    'catenate_all' => true,
                    'preserve_original' => true,
                    'catenate_numbers' => true,
                ]
            ],
            'analyzer' => [
                'default' => [
                    'type' => 'custom',
                    'char_filter' => [
                        'html_strip',
                        'replace',
                    ],
                    'tokenizer' => 'whitespace',
                    'filter' => [
                        'lowercase',
                        'word_delimiter',
                    ],
                ],
            ],
        ],
    ];

    protected $mappingProperties = [
        'title' => [
            'type' => 'string',
            'analyzer' => 'standard'
        ]
    ];

//    use ElasticquentTrait;

    public static $elasticFuzziness = 1;

    /**
     * @description return elastic query with fuzziness
     */
    public static function buildElasticQuery($keyWords){
        $params=[
            'multi_match' => [
                'query'     => $keyWords,
                'fuzziness' => self::$elasticFuzziness,
                'fields'    => ['_all']
            ]
        ];
        return $params;
    }

//	use KeyWordsFunctional;

	public static function clearIsFamilyImage($familyId, $notIn=[]){
		self::where('webfamily_id', $familyId)->where('is_family_image',1)->whereNotIn('id',$notIn)->update(['is_family_image'=>0]);
	}

    /**
     * @description index all the entries in an Eloquent model (if needed)
     */
    public static function elasticIndexingIfNeeded(){
        $mappingExists = self::mappingExists();
        if (!$mappingExists) {
            self::createIndex();
            self::putMapping();
            self::addAllToIndex();
        }
	}
// *********************************************************************************************************************

// ***************************************RELATIONSHIPS*****************************************************************
	public function part() {
		return $this->belongsTo(Part::class, 'part_id', 'id');
	}

	/**
	 * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
	 */
	public function webFamily() {
		return $this->belongsTo(WebFamily::class, 'webfamily_id', 'id');
	}

    public function legacyPart(){
	    return $this->hasOne(LegacyPart::class, 'part_id', 'id');
    }

    public function webAttributes() {
        return $this->belongsToMany(WebAttribute::class, 'WebAttribute_WebPart', 'webpart_id', 'webattribute_id')
            ->withPivot(['id','attribute_value']);
    }

    public function relatedParent() {
        return $this->morphToMany(WebFamily::class, 'related', 'WebFamily_Related', 'related_id', 'webfamily_id')->withPivot(['related_category']);
    }

    public function customerPrices() {
        return $this->part->customerPrices();
    }

    public function hierarchyPaths() {
        $paths = ['webPart' => $this];
        if ($this->webFamily) $paths = $this->webFamily->hierarchyPaths($paths);
        return $paths;
    }

    public function partAsset() {
        return $this->hasOne(WebAsset::class, 'name', 'part_number')->where('asset_category', 'part')->orderBy('date_created', 'DESC');
    }
// *********************************************************************************************************************

// ***************************************GETTERS/SETTERS***************************************************************
    public function webAttributeValue($webAttributeId) {
        if (!$this->relationLoaded('webAttributes')) $this->load('webAttributes');
        $webAttributeValue = $this->webAttributes->find($webAttributeId);
        if (!is_null($webAttributeValue)) return $webAttributeValue->pivot->attribute_value;
        return null;
    }

    public function setWebAttributeValue($webAttributeId, $value) {
        $this->webAttributes()->sync([$webAttributeId => ['attribute_value' => $value]], false);
    }

    public function setWebAttributesAttribute($attributes) {
        $this->save();
        foreach ($attributes as $webAttributeId => $attributeValue) {
            $this->setWebAttributeValue($webAttributeId, $attributeValue);
        }
    }

    public function getPartNameAttribute() {
        $part = $this->part;
        if (is_null($part)) return null;
        return $part->part_desc;
    }

    public function getFamilyNameAttribute() {
        $family = $this->webFamily;
        if (is_null($family)) return null;
        return $family->name;
    }

    public function getNameAttribute($name)
    {
        $activeLanguage = b2b()->activeLanguage();
        if($activeLanguage != "en"){
            $translated = Translations_WebPart::where('language', '=', $activeLanguage)
                ->where('part_no', '=', $this->part_number)
                ->where('deleted_at', '=', '0')
                ->first();
            if(isset($translated)) {
                return $translated->name;
            }
        }
        return $name;
    }

    public function getDisplayNameForUACAttribute() {
        return $this->name;
    }

    public function getListPriceAttribute() {
        return $this->part->list_price;
    }

    public function getCustomerPriceAttribute() {
        return $this->part->customer_price;
    }

    public function getCustomerPrice($quantity = 1) {
        return $this->part->getCustomerPrice($quantity);
    }

    public function getCustomerDiscount() {
        return $this->part->getCustomerDiscount();
    }

    public function getImageLink($width = null, $height = null, $avoidCache = false) {
        return $this->primaryImage($width ? $width : false, $height ? $height : false);
    }

    public function getImageUrl($width = false, $height = false) {
        return HilcoAssets::part($this, $width, $height);
    }

    public function getIsRXAttribute() {
        $part = $this->part;
        if (is_null($part)) return null;
        return (!is_null($part->pharmacyFamilyExclusion) || !is_null($part->pharmacyPartExclusion));
    }

    public function getHasValidInventoryAttribute() {
        $status = 'none';
        $hasAll = true;
        if ($this->part) {
            foreach ($this->part->inventoryItems as $inventoryItem) {
                if ($inventoryItem->is_valid) {
                    $status = 'some';
                } else {
                    $hasAll = false;
                }
            }
        }

        if ($hasAll) $status = 'all';
        return $status;
    }

    public function getHierarchyPathsAttribute() {
        $isVisible = true;
        if (!$this->is_visible) $isVisible = trans('hilco::app.manually_hidden');
        if (is_null($this->part)) $isVisible = trans('hilco::app.not_in_point_man');
        if (is_null($this->webFamily)) $isVisible = trans('hilco::app.not_in_a_family');
        if ($this->has_valid_inventory == 'none') $isVisible = trans('hilco::app.no_valid_inventory');

        $paths = [];
        if (!is_null($this->webFamily)) {
            foreach ($this->webFamily->webCollections as $webCollection) {
                foreach ($webCollection->webCategories as $webCategory) {
                    foreach ($webCategory->webGroups as $webGroup) {
                        foreach ($webGroup->webHierarchies as $webHierarchy) {
                            if (!$this->webFamily->is_visible)
                                $isVisible = trans('hilco::app.parent_family_not_visible');
                            if (!$webCollection->is_visible)
                                $isVisible = trans('hilco::app.parent_collection_not_visible');
                            if (!$webCategory->is_visible)
                                $isVisible = trans('hilco::app.parent_category_not_visible');
                            if (!$webGroup->is_visible) $isVisible = trans('hilco::app.parent_group_not_visible');
                            $paths[] = [
                                'webHierarchy' => $webHierarchy,
                                'webGroup' => $webGroup,
                                'webCategory' => $webCategory,
                                'webCollection' => $webCollection,
                                'webFamily' => $this->webFamily,
                                'isVisible' => $isVisible,
                            ];
                        }
                    }
                }
            }
        }
        return $paths;
    }

    public function setWebfamilyIdAttribute($value) {
        if (!$value || $value < 1) $this->attributes['webfamily_id'] = null;
        else $this->attributes['webfamily_id'] = $value;
    }

    public function getInStockAttribute() {
        return $this->part->in_stock;
    }

    public function getInStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'in_stock_label');
        return strlen($modelValue) ? $modelValue : config('hilco.inStockLabels.0', 'In Stock');
    }

    public function getLowStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'low_stock_label');
        return strlen($modelValue) ? $modelValue : config('hilco.lowStockLabels.0', 'Low Stock');
    }

    public function getOutOfStockLabelAttribute() {
        $modelValue = array_get($this->attributes, 'out_of_stock_label');
        return strlen($modelValue) ? $modelValue : config('hilco.outOfStockLabels.0', 'Out of stock');
    }

    public function getStockStatusAttribute() {
        $atpData = [$this->part_number];
        $atps = [];
        try {
            $atps = WebPart::fetchAllItemsStockInfo($atpData);
        }catch(Exception $e){
            Log::error("Exception caught while fetching stock info: " . $e->getMessage());
            return Part::UNKNOWN;
        }

        return $atps[0]['status'];
    }

    public function getPartTaxability($linePrice){
        $client = null;
        $guzzle = new M3Request();
        if(empty($guzzle->client)){
            Log::error('Exception Caught while performing ' . __FUNCTION__ . ': Guzzle Http Client does not exist. Base URI may be missing!');
            return Part::UNKNOWN;
        }else{
            $client = $guzzle->client;
        }

        $shippingAddress = b2b()->activeShippingAddress();
        $queryAttributes = [
            'requestFrom' => 'b2b',
            'shippingAddress1' => $shippingAddress->addr_1,
            'shippingAddress2' => $shippingAddress->addr_2,
            'shippingAddress3' => $shippingAddress->addr_3,
            'shippingAddress4' => $shippingAddress->addr_4,
            'shippingAddress5' => $shippingAddress->addr_5,
            'city' => $shippingAddress->city,
            'mainDivision' => $shippingAddress->state,
            'postalCode' => $shippingAddress->postal_cd,
            'country' => $shippingAddress->country,
            'part_number' => $this->part->part_no,
            'productClass' => $this->part->product_family_no,
            'unit_price' => $linePrice
        ];

        $taxResults = json_decode($client->get('v4/' . __FUNCTION__, ['query' => http_build_query($queryAttributes), 'headers' => ['Accept' => 'application/json']])->getBody()->getContents(), true);
        $taxInfo = [];
        foreach($taxResults['details']['data']['lineItems'][0]['taxes'][0] as $key => $taxResult){
            if(in_array($key, ['taxable', 'taxResult', 'calculatedTax'])){
                $taxInfo[$key] = $taxResult;
            }
        }

        return $taxInfo;
    }

    public function getStockLabel($status){
        switch ($status) {
            case Part::IN_STOCK:
                return $this->in_stock_label;
            case Part::LOW_STOCK:
                return $this->low_stock_label;
            case Part::OUT_OF_STOCK:
                return $this->out_of_stock_label;
            case Part::DELAYED:
                return "Ships in 3-5 days";
            case Part::UNKNOWN:
                return ucfirst(Part::UNKNOWN);
        }
    }

    public function getStockLabelAttribute() {
        $status = $this->stock_status;
        switch ($status) {
            case Part::IN_STOCK:
                return $this->in_stock_label;
            case Part::OUT_OF_STOCK:
                return $this->out_of_stock_label;
            case Part::DELAYED:
                return "Ships in 3-5 days";
            case Part::UNKNOWN:
                return ucfirst(Part::UNKNOWN);
        }
    }

    public function getPrimaryImage() {
        $webAsset = $this->assetsByType('primary')->first();
        if (is_null($webAsset)) $webAsset = $this->partAsset;

        return $webAsset;
    }

    public function getIsSiloPartAttribute() {
        foreach (array_get($this, 'part.webSilos', []) as $partWebSilo) {
            if ($partWebSilo->id == b2b()->activeWebSilo()->id) return true;
        }
        return false;
    }

    public function getRequireApprovalAttribute() {
//        foreach (array_get($this, 'part.webSilos', []) as $partWebSilo) {
//            if ($partWebSilo->id == b2b()->activeWebSilo()->id) {
//                if (array_get($partWebSilo, 'pivot.require_approval', false)) return true;
//            }
//        }
//        return false;
        return $this->part->require_approval;
    }

    public function getIsWebVisibleAttribute() {
        return static::webVisible()->where('id', $this->id)->count() > 0;
    }
// *********************************************************************************************************************

// ***************************************SCOPES************************************************************************
    public function scopeJoinDiscount($query){
        $customerId = Auth::user()->customer_id;
        return $query->select(['WebParts.*',DB::raw('SUM(CustomerDiscounts.disc_val) as disc_val')])->leftJoin('CustomerDiscounts', function($query)use($customerId){
            return $query->on('CustomerDiscounts.part_id','=','WebParts.id')
                ->orOn('CustomerDiscounts.productfamily_id','=','WebParts.webfamily_id')
                ->where('CustomerDiscounts.customer_id','=',$customerId);
        })->groupBy('id');
    }

	public function scopeIsFamilyImage($query){
		$query->where('is_family_image',1)->visible()->whereHas('part', function ($query) {
            return $query->whereHas('inventoryItems', function ($query) {
                return $query->isValid();
            });
        });
	}

	public function scopePriceList($query, $priceList, $currency, $quantityLevel = 1) {
		return $query
			->leftJoin('PriceLists', 'PriceLists.part_no', '=', 'WebParts.part_number')
			->where('price_list', $priceList)
			->where('currency', $currency)
			->where('quantity_level', '>=', $quantityLevel)
			->select('WebParts.*', 'PriceLists.price')
		;
	}

	public function scopePriceRange($query, $min, $max) {
		return $query->whereHas('part', function($part) use ($min, $max){
			$part->whereBetween('list_price', [$min, $max]);
		});
	}

	public function scopeManagedBy($query, $manager) {
		return $query->whereHas('part', function ($query) use ($manager) {
			return $query->where('pf_prod_mgr', $manager);
		});
	}

	public function scopeInFamily($query, $webFamilyId) {
		return $query->whereHas('webFamily', function ($query) use ($webFamilyId) {
			return $query->where('webfamily_id', $webFamilyId);
		});
	}

    public function scopeInCollection($query, $webCollectionId) {
        return $query->whereHas('webFamily', function ($query) use ($webCollectionId) {
            return $query->inCollection($webCollectionId);
        });
    }

    public function scopeInCategory($query, $webCategoryId) {
        return $query->whereHas('webFamily', function ($query) use ($webCategoryId) {
            return $query->inCategory($webCategoryId);
        });
    }

	public function scopeInGroup($query, $webGroupId) {
		return $query->whereHas('webFamily', function ($query) use ($webGroupId) {
			return $query->inGroup($webGroupId);
		});
	}

	public function scopeHasVisibleChildren(Builder $query, $activeWebSilo = false, $activeCountry = false, $activePlant = false) {
	    return $query
            ->whereHas('part', function ($query) use ($activeWebSilo, $activeCountry, $activePlant) {
                return $query->visibleToActiveUser($activeWebSilo, $activeCountry, $activePlant);
            })
        ;
    }

    public function scopeHasVisibleParents(Builder $query, $activeWebSilo = false) {
        return $query
            ->whereHas('webFamily', function ($query) use ($activeWebSilo) {
                return $query->hasVisibleParents($activeWebSilo);
            })
            ;
    }

    public function scopeIsVisible(Builder $query) {
        return $query->where('WebParts.is_visible', '1');
    }

	public function scopeWebVisible(Builder $query) {
	    return $query
            ->isVisible()
            ->hasVisibleParents()
            ->hasVisibleChildren()
        ;
    }

    public function scopeWebSiloApproved(Builder $query) {
        return $query
            ->whereHas('part', function ($partQuery) {
                return $partQuery->webSiloApproved();
            })
        ;
    }

	public function scopeVisible(Builder $query, $webSiloId = false) {
	    if ($webSiloId === false) {
	        $webSiloId = b2b()->activeWebSilo()->id;
        }
		return $query
			->where('WebParts.is_visible', '1')
			->whereHas('part', function ($query) use ($webSiloId) {

                if (!b2b()->activeWebSilo()->is_default) {
                    $query->whereHas('webSilos', function ($query) {
                        return $query->where('websilo_id', '=', b2b()->activeWebSilo()->id);
                    });
                }

                if (!b2b()->activeWebSilo()->allow_private_label) {
//                    $query->doesntHave('assortments');
                    $query->where('private_lbl', '0');
                }

//				return $query->whereHas('inventoryItems', function ($query) use ($webSiloId) {
//                    $activeCountry = b2b()->activeCountry();
//                    if ($activeCountry == 'US') $query->inPlant('PLAINVILLE');
//                    return $query->inSilo($webSiloId);
//				});
			})
		;
	}
// *********************************************************************************************************************


	public function visibleInSilo($webSilo, $processDownChain = true, $processUpChain = true) {
        $inSilo = true;
        $isFamilyVisible = !$processUpChain;

	    if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);

        if($processDownChain){
            if (isset($webSilo) && $webSilo->limit_parts) {
                $found = false;
                foreach (array_get($this->part, 'webSilos', []) as $partWebSilo) {
                    if (array_get($partWebSilo, 'id') == $webSilo->id) {
                        $found = true;
                    }
                }
                if (!$found) {
                    $inSilo = false;
                }
            }

            if (isset($webSilo) && !$webSilo->allow_private_label &&
                isset($this->part) && /*$this->part->assortments()->count()*/ $this->part->private_lbl > 0) {

                $inSilo = false;
            }
        }

        if($processUpChain){
            if ($this->webFamily->visibleInSilo($webSilo, false, true)){
                $isFamilyVisible = true;
            }
        }

        return ($inSilo && $isFamilyVisible);
    }

    public static function fetchAllItemsStockInfo($partNumbers, $warehouse = null){
        $atpInfo = [];
        if (auth()->user()) {
            $guzzleClient = null;
            $guzzleM3Request = new M3Request();
            if (empty($guzzleM3Request->client)) {
                Log::error(
                    'Exception Caught while performing ' . __FUNCTION__ .
                    ': Guzzle Http Client does not exist. Base URI may be missing!')
                ;
                return Part::UNKNOWN;
            } else {
                $guzzleClient = $guzzleM3Request->client;
            }

            $atpRequestData = [];
            foreach($partNumbers as $partNumber) {
                $atpRequestData[$partNumber] = [
                    'item_number' => $partNumber,
                    'warehouse' => !is_null($warehouse) ? $warehouse : b2b()->activeCustomer()->activeSegment->def_ship_from
                ];
            }

            // Make getATP calls in chunks of 10
            $chunks = array_chunk($atpRequestData, 10, true);
            $response['atps'] = [];
            foreach ($chunks as $chunk) {
                //not doing array_merge here because we need to preserve our keys
                $response['atps'] =
                    $response['atps'] +
                    json_decode(
                        $guzzleClient
                            ->post(
                                'v4/getItemWarehouseBalance',
                                [
                                    'form_params'   => [
                                        'requestFrom'           => 'b2b',
                                        'itemsATPRequestInfo'   => $chunk
                                    ],
                                    'headers'       => [
                                        'Accept' => 'application/json'
                                    ]
                                ]
                            )
                            ->getBody()
                            ->getContents(),
                        true)['atps']
                ;
            }

            // Process all responses
            //$atpId = partNumber
            foreach ($response['atps'] as $atpId => $atp) {
                if (count($atp['details']) == 1) {
                    Log::error('Exception Caught while performing ' . __FUNCTION__ . ': ' . $atp['details']);
                    $atpInfo[$atpId] = [
                        'status' => Part::UNKNOWN,
                        'stock' => 0,
                        'stockable' => false
                    ];
                } else if (
                    array_has($atp['details'], 'MIRecord') &&
                    array_has($atp['details']['MIRecord'][0], 'NameValue')) {

                    $results = $atp['details']['MIRecord'][0]['NameValue'];
                    foreach ($results as $result) {
                        if ($result['Name'] == 'AV01') {
                            $numStock = trim($result['Value'], ' ');
                            $atpInfo[$atpId]['stock'] = !empty($numStock) ? round($numStock, 0) : 0;
                            $atpInfo[$atpId]['stockable'] = true;

                            $part = Part::wherePartNo($atpId)->first();
                            $inventoryItem = $part->inventoryItems()->inPlant(b2b()->activePlant()->plant)->first();

                            if(!empty($inventoryItem)) {
                                if ($inventoryItem->part_stat == 'DoNotReorder') {
                                    $atpInfo[$atpId]['status'] = Part::LOW_STOCK;
                                } else if ($atpInfo[$atpId]['stock'] > 0) {
                                    $atpInfo[$atpId]['status'] = Part::IN_STOCK;
                                } else {
                                    $atpInfo[$atpId]['status'] = Part::OUT_OF_STOCK;
                                }
                            }else{
                                $atpInfo[$atpId]['status'] = Part::UNKNOWN;
                            }
                        }
                    }

                } else {
                    Log::error('Exception Caught while performing ' . __FUNCTION__ . ': ' . $atp['details']['Message']);
                    $atpInfo[$atpId] = [
                        'status' => Part::UNKNOWN,
                        'stock' => 0
                    ];

                    if (strpos($atp['details']['Message'], "does not exist in the warehouse") !== false) {
                        $atpInfo[$atpId]['stockable'] = false;
                    }
                }
            }

        }

        return $atpInfo;
    }

    /*
     * Made this function only to find out that there was a simpler way to solve the problem with was designed to solve.
     * Necessary to ensure quick order and multi line quick order enforces Web Silo restrictions, but probably going to
     *  want to rotate that functionality for scope visibile functions.
     */
    public function isWebSiloAccessible($webSilo = null){
        if(is_null($webSilo)){
            $webSilo = b2b()->activeWebSilo();
        }
        if(!($webSilo->limit_parts)){
            return true;
        }
        $webSiloParts = $webSilo->getParts();
        $id = $this->part->id;
        $validWebPart = $webSiloParts->filter(function($item) use ($id){
                return $item->part_id === $id;
            })->first();
        if(!is_null($validWebPart)){
            return true;
        }
        return false;
    }

    public function getQuantityPresetsArrayAttribute() {
	    $presets = $this->quantity_presets;
	    if (strlen($presets) < 1) return [];

	    $array = explode(',', $presets);
	    return array_combine($array, $array);
    }

    public function slugUrl() {
        return route('family.slug', [$this->webFamily->slug, $this->id]);
    }


    use HasAssets, RevisionableTrait, HasCommitSequence, HasModelTranslations;
}
