<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 3/28/2017
 * Time: 12:09 PM
 */

namespace Hilco\Shipments;


use Carbon\Carbon;
use Hilco\Models\Carrier;
use Hilco\Models\CustomerShippingAddress;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use RocketShipIt\Rate as RocketShipItRate;
use RocketShipIt\TimeInTransit;

class ShipmentManager
{
    protected $carriers;

    public function __construct() {
        $this->carriers = new Collection;
    }

    public function getWebCartRates(CustomerShippingAddress $customerShippingAddress) {
        $cartComposer = App::make('cartSingleton');
        $cartStatus = $cartComposer->getCartStatus();
        $activeWebSilo = b2b()->activeWebSilo;
        $rateType = array_get($activeWebSilo, 'shipment_rate_type', 'api');
        $rates = new Collection;

        if ($rateType == 'api'){
            $rates = $this->getWebCartApiRates($cartStatus, $customerShippingAddress);
        } else {
            $siloRates = array_get($this->activeWebSilo, 'shipmentFlatRates', false);
            krsort($siloRates);
            $cartValue = 0;
            if ($rateType == 'flatByWeight') $cartValue = array_get($cartStatus, 'totalWeight', 0);
            if ($rateType == 'flatByPrice') $cartValue = array_get($cartStatus, 'discountedTotal', 0);
            foreach ($siloRates as $threshold => $siloRate) {
                if ($cartValue >= $threshold) {
                    $rates->push(new Rate(null, null, $siloRate, $siloRate));
                    break;
                }
            }
        }

        return $rates;
    }

    protected function getWebCartApiRates($cartStatus, CustomerShippingAddress $customerShippingAddress) {
        $defaultPlant = $customerShippingAddress->plant;
        $fromZipCode = array_get($defaultPlant, 'plantAddress.zip');
        $packages = [];

        foreach (array_get($cartStatus, 'items', []) as $cartItem) {
            $part = array_get($cartItem, 'part');
            if ($part->inStockForPlant($defaultPlant)) {
                array_set($packages, 'inStock.weight', $part->weight_lbs * array_get($cartItem, 'quantity', 0));
                array_set($packages, 'inStock.volume', $part->vol_ci * array_get($cartItem, 'quantity', 0));
            } else {
                array_set($packages, 'backordered.weight', $part->weight_lbs * array_get($cartItem, 'quantity', 0));
                array_set($packages, 'backordered.volume', $part->vol_ci * array_get($cartItem, 'quantity', 0));
            }
        }

        foreach ($packages as $package) {

        }
    }

    public function getApiRates(Collection $carriers, $package) {
        $this->carriers = $carriers;
        $rates = new Collection;
        $apiCarriers = $carriers->keyBy('api_carrier');
        $apiRates = [];
        foreach ($apiCarriers as $carrier) {
            $apiCarrier = $carrier->api_carrier;
            switch ($apiCarrier) {
                case 'UPS':
                    $apiRates[$apiCarrier] = $this->getRateForUpsCarrier($package, $carrier->residential);
                    break;
                case 'USPS':
                    $apiRates[$apiCarrier] = $this->getRateForUspsCarrier($package, $carrier->residential);
                    break;
            }
        }

        foreach ($apiRates as $apiCarrier => $apiCarrierRates) {
            foreach ($apiCarrierRates as $carrierCode => $apiRate) {
                $rate = new Rate();
                $rate->package = $package;
                $rate->carrier = $this->carriers->keyBy('carrier_code')->get($carrierCode);
                $rate->base_rate = (float)array_get($apiRate, 'rate', 0);
                $rate->discounted_rate = (float)$rate->base_rate * (1 - ($rate->carrier->discount/100));
                $rate->delivery_datetime = Carbon::parse(array_get($apiRate, 'delivery_date') . ' ' . array_get($apiRate, 'delivery_time'));
                $rate->guaranteed = array_get($apiRate, 'guaranteed', false);
                $rates->put($carrierCode, $rate);
            }
        }

        return $rates;
    }

    protected function callRocketShipIt($apiCarrier, $toCountry, $weight, $volume, $fromZipCode, $toZipCode, $isResidential) {
        $rate = new RocketShipItRate($apiCarrier);
        $rate->setParameter('shipCountry', 'US');
        $rate->setParameter('toCountry', $toCountry);

        $rate->setParameter('weightUnit', 'LBS');
        $rate->setParameter('weight', max(1, $weight));

        $rate->setParameter('lengthUnit', 'IN');
        $rate->setParameter('volume', $volume);

        $rate->setParameter('currency', 'USD');
        $rate->setParameter('shipCode', $fromZipCode);
        $rate->setParameter('toCode', $toZipCode);

        $rate->setParameter('residentialAddressIndicator', $isResidential);

        $rateResponse = $rate->getAllRates();

        $transit = new TimeInTransit($apiCarrier);

        $transit->setParameter('shipCountry', 'US');
        $transit->setParameter('shipCode', $fromZipCode);

        $transit->setParameter('toCountry', $toCountry);
        $transit->setParameter('toCode', $toZipCode);

        $transit->setParameter('weightUnit', 'LBS');
        $transit->setParameter('weight', $weight);
        $transit->setParameter('saturdayDelivery', 'YES');

        $transit->setParameter('lengthUnit', 'IN');
        $transit->setParameter('volume', $volume);

        $transit->setParameter('pickupDate', date('Y-m-d'));

        $transitResponse = $transit->getTimeInTransit();
        return [
            'rates' => $rateResponse,
            'transit' => $transitResponse,
        ];
    }

    protected function getRateForUpsCarrier(Package $package, $isResidential) {
        $apiInfo = $this->callRocketShipIt('UPS', $package->country, $package->weight, $package->volume, $package->fromZipCode, $package->zipCode, $isResidential);

        $rateInfo = [];
        foreach (array_get($apiInfo, 'rates.RatingServiceSelectionResponse.RatedShipment', []) as $ratedShipment) {
            $apiCode = array_get($ratedShipment, 'Service.Code');
            $carriers = $this->carriers->where('api_code', $apiCode);
            foreach ($carriers as $carrier) {
                $carrierCode = $carrier->carrier_code;
                array_set($rateInfo, "$carrierCode.rate", array_get($ratedShipment, 'TotalCharges.MonetaryValue'));
                array_set($rateInfo, "$carrierCode.api_code", $apiCode);
            }
        }

        foreach (array_get($apiInfo, 'transit.TimeInTransitResponse.TransitResponse.ServiceSummary', []) as $transitService) {
            $transitCode = array_get($transitService, 'Service.Code');
            $carriers = $this->carriers->where('transit_code', $transitCode);
            foreach ($carriers as $carrier) {
                $carrierCode = $carrier->carrier_code;
                array_set($rateInfo, "$carrierCode.transit_code", $transitCode);
                array_set($rateInfo, "$carrierCode.delivery_date", date('Y-m-d', strtotime(array_get($transitService, 'EstimatedArrival.Date'))));
                array_set($rateInfo, "$carrierCode.delivery_time", date('H:i', strtotime(array_get($transitService, 'EstimatedArrival.Time'))));
                array_set($rateInfo, "$carrierCode.guaranteed", (array_get($transitService, 'Guaranteed.Code', 'N') == 'Y'));
            }
        }

        return $rateInfo;
    }

    protected function getRateForUspsCarrier(Package $package, $isResidential) {
        $apiInfo = $this->callRocketShipIt('USPS', 'US', $package->weight, $package->volume, $package->fromZipCode, $package->zipCode, $isResidential);

        $rateInfo = [];
        foreach (array_get($apiInfo, 'rates.RateV4Response.Package.Postage', []) as $ratedShipment) {
            $apiCode = array_get($ratedShipment, 'CLASSID');
            $carriers = $this->carriers->where('api_code', $apiCode);
            foreach ($carriers as $carrier) {
                $carrierCode = $carrier->carrier_code;
                array_set($rateInfo, "$carrierCode.rate", array_get($ratedShipment, 'Rate'));
                array_set($rateInfo, "$carrierCode.api_code", $apiCode);
            }
        }

        foreach (array_get($apiInfo, 'transit.ExpressMailCommitmentResponse', []) as $transitCode => $transitService) {
            $carriers = $this->carriers->where('transit_code', $transitCode);
            foreach ($carriers as $carrier) {
                $carrierCode = $carrier->carrier_code;
                if ($transitCode == 'ExpressMailCommitmentResponse') {
                    $commitment = array_get($transitService, 'Commitment.0', array_get($transitService, 'Commitment', []));
                    array_set($rateInfo, "$carrierCode.delivery_date", date('Y-m-d', strtotime("+ " . str_replace("-", " ", array_get($commitment, 'CommitmentName')))));
                    array_set($rateInfo, "$carrierCode.delivery_time", date('H:i', strtotime(array_get($commitment, 'CommitmentTime'))));
                    array_set($rateInfo, "$carrierCode.guaranteed", true);
                } else {
                    array_set($rateInfo, "$carrierCode.delivery_date", date('H:i', strtotime("+ " . array_get($transitService, 'Days') . ' weekdays')));
                    array_set($rateInfo, "$carrierCode.guaranteed", false);
                }
            }
        }

        return $rateInfo;
    }
}