<?php

namespace Hilco\Models;
use AlgoliaSearch\Laravel\AlgoliaEloquentTrait;
use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Venturecraft\Revisionable\RevisionableTrait;
use Illuminate\Support\Facades\Input;

class WebCollection extends WebModel implements HasLandingPageInterface, HasWebSiloOverridesInterface {
    protected $table = "WebCollections";
    protected $fillable = ['name', 'slug', 'webCategoryIds', 'is_visible', 'detail', 'sort_priority'];
    protected $casts = [
        ['is_visible' => 'boolean']
    ];
    protected $with = ['webSiloOverride', 'webCollectionTranslations'];
    protected $perPage = 24;

//    use KeyWordsFunctional;
    use AlgoliaEloquentTrait;

    public $algoliaSettings = [
        'searchableAttributes' => [
            'name',
        ],
        'attributesForFaceting' => [
            'webSiloIds',
            'is_rx',
        ]
    ];

    public static $perEnvironment = true;

    public function _reindex($safe = true, $setSettings = true, $mergeOldSettings = false, \Closure $onInsert = null)
    {
        /** @var \AlgoliaSearch\Laravel\ModelHelper $modelHelper */
        $modelHelper = App::make('\AlgoliaSearch\Laravel\ModelHelper');

        $indices = $modelHelper->getIndices($this);
        $indicesTmp = $safe ? $modelHelper->getIndicesTmp($this) : $indices;

        if ($setSettings === true) {
            $setToTmpIndices = ($safe === true);
            $this->_setSettings($setToTmpIndices, $mergeOldSettings);
        }

        static
            ::has('webCategories.webGroups')
            ->whereHas('webFamilies', function ($query) {
                $query->whereHas('webParts', function ($query) {
                    $query->where('is_visible', 1)->has('part');
                });
            })
            ->with('webCategories.webGroups')
            ->with('webFamilies.webParts.part')
            ->chunk(100, function ($models) use ($indicesTmp, $modelHelper, $onInsert) {
            /** @var \AlgoliaSearch\Index $index */
            foreach ($indicesTmp as $index) {
                $records         = [];
                $recordsAsEntity = [];

                foreach ($models as $model) {
                    if ($modelHelper->indexOnly($model, $index->indexName)) {
                        $records[] = $model->getAlgoliaRecordDefault($index->indexName);

                        if ($onInsert && is_callable($onInsert)) {
                            $recordsAsEntity[] = $model;
                        }
                    }
                }

                $index->addObjects($records);

                if ($onInsert && is_callable($onInsert)) {
                    call_user_func_array($onInsert, [$recordsAsEntity]);
                }
            }

        });

        if ($safe) {
            for ($i = 0; $i < count($indices); $i++) {
                $modelHelper->algolia->moveIndex($indicesTmp[$i]->indexName, $indices[$i]->indexName);
            }

            $this->_setSettings(false); // To a setSettings to set the slave on the master
        }

    }

    public function getAlgoliaRecord() {
        $record = array_only($this->toArray(), [
            'id',
            'name',
            'slug',
        ]);

        $record['webSiloIds'] = [];
        foreach (WebSilo::visible()->get() as $webSilo) {
            if ($this->visibleInSilo($webSilo)) $record['webSiloIds'][] = $webSilo->id;
        }

        $hasNonRX = false;
        foreach($this->webFamilies as $webFamily) {
            if ($webFamily->webParts) {
                foreach ($webFamily->webParts as $webPart) {
                    if ($webPart->part && !$webPart->part->is_rx) {
                        $hasNonRX = true;
                        break 2;
                    }
                }
            }
        }
        $record['is_rx'] = !$hasNonRX;
        return $record;
    }

    public function getPerPage()
    {
        $tmpPerPage =Input::get('perPage');
        return (isset($tmpPerPage) && $tmpPerPage > 0)?$tmpPerPage:$this->perPage;
    }

    public function getNameAttribute($name)
    {
        $activeLanguage = b2b()->activeLanguage();
        if($activeLanguage != "en"){
            $translation = Translations_WebCollection::getTranslationByWebCollectionId($this->id);
            if(!is_null($translation)){
                return $translation->name;
            }
        }
        return $name;
    }

    public function getDetailAttribute($detail)
    {
        $activeLanguage = b2b()->activeLanguage();
        if($activeLanguage != "en"){
            $translation = Translations_WebCollection::getTranslationByWebCollectionId($this->id);
            if(!is_null($translation)){
                return $translation->description;
            }
        }
        return $detail;
    }

    public function webCategories() {
        $returnMe = $this->belongsToMany(WebCategory::class, 'WebCollection_WebCategory', 'webcollection_id', 'webcategory_id')->withPivot('display_order');
        return $returnMe;
    }

    public function webCollectionTranslations(){
        return $this->hasMany(Translations_WebCollection::class, 'webcollection_id', 'id');
    }

    /**
     * @return BelongsToMany
     */
    public function webFamilies() {
        return $this->belongsToMany(WebFamily::class, 'WebFamily_WebCollection', 'webcollection_id', 'webfamily_id')->withPivot(['is_featured', 'display_order']);
    }

    public function webCollection_Algolia()
    {
        return $this->hasMany(WebCollection_Algolia::class, 'webCollection_id', 'id');
    }

    public function visibleWebFamilies() {
        return $this->belongsToMany(WebFamily::class, 'WebFamily_WebCollection', 'webcollection_id', 'webfamily_id')
            ->withPivot(['is_featured', 'display_order'])
            ->webVisible()
            ->displayOrder()
        ;
    }

    public function visibleInSilo($webSilo) {
        if (!$webSilo instanceof WebSilo) $webSilo = WebSilo::find($webSilo);

        foreach ($this->webFamilies as $webFamily) {
            if ($webFamily->visibleInSilo($webSilo)) return true;
        }

        return false;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

    public function getFeaturedChildrenAttribute() {
        return $this->visibleWebFamilies()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        $children = $this->visibleWebFamilies();
        if (request()->get('ap', false) == 1) $children->webSiloApproved();

        return $children->get();
    }

    public function scopeInCategory($query, $webCategoryId) {
        return $query->whereHas('webCategories', function ($query) use ($webCategoryId) {
            $query->where('webcategory_id', $webCategoryId);
        });
    }

    public function scopeHavingFamily($query, $webFamilyId) {
        return $query->whereHas('webFamilies', function ($query) use ($webFamilyId) {
            $query->where('webfamily_id', $webFamilyId);
        });
    }

    public function setWebCategoryIdsAttribute($ids) {
        $this->save();
        $this->createRevisionRecord('webCategoryIds', $this->webCategories->pluck('id')->toJson(), json_encode($ids));
        $this->webCategories()->sync($ids);
    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webCategories as $webCategory) {
            foreach ($webCategory->webGroups as $webGroup) {
                foreach ($webGroup->webHierarchies as $webHierarchy) {
                    $paths[] = [
                        'webHierarchy' => $webHierarchy,
                        'webGroup' => $webGroup,
                        'webCategory' => $webCategory,
                    ];
                }
            }
        }
        return $paths;
    }

    public static function deduplicatedList() {
        $webCollections = WebCollection::with('webCategories.webGroups.webHierarchies')->get()->sortBy('name')->keyBy('id');
//        $names = $webCollections->groupBy('name');
        $output = [];
        foreach ($webCollections as $id => $webCollection) {
            foreach ($webCollection->hierarchyPaths as $path) {
                $string = $webCollection->name;
                $string .= ' (' . $path['webCategory']->name;
                $string .= ' -> ' . $path['webGroup']->name;
                $string .= ' -> ' . $path['webHierarchy']->name . ')';
                $output[$id] = $string;
                break;
            }
        }
        return $output;
    }

    public function scopeInCollection($query, $webCollection) {
        return $query->whereHas('webCollections', function ($query) use ($webCollection) {
            $query->where('webcollection_id', $webCollection->id);
        });
    }

    public function scopeInGroup($query, $webGroupId) {
        return $query->whereHas('webCategories', function ($query) use ($webGroupId) {
            return $query->inGroup($webGroupId);
        });
    }

    public function scopeInSilo($query, $webSiloId) {
        return $query->whereHas('webCategories', function ($query) use ($webSiloId) {
            return $query->inSilo($webSiloId);
        });
    }    

    public function getAssetPrefix() {
        return "webCollections/{$this->id}";
    }

    public function scopeVisible(Builder $query) {
        $query
            ->where('WebCollections.is_visible', '1')
            ->whereHas('webCategories', function ($query) {
                return $query->visible();
            })
        ;

        return $query;
    }

    public function scopeHasVisibleChildren(Builder $query, $activeWebSilo = false, $activeCountry = false, $activePlant = false) {
        return $query
            ->whereHas('webFamilies', function ($query) use ($activeWebSilo, $activeCountry, $activePlant) {
                return $query->hasVisibleChildren($activeWebSilo, $activeCountry, $activePlant);
            })
        ;
    }

    public function scopeHasVisibleParents(Builder $query, $activeWebSilo = false) {
        return $query
            ->whereHas('webCategories', function ($query) use ($activeWebSilo) {
                return $query->hasVisibleParents($activeWebSilo);
            })
        ;
    }

    public function scopeIsVisible(Builder $query) {
        return $query
            ->where('is_visible', '1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->where('WebSiloOverrides.is_visible', '=', 1);
                    })
                ;
            })
        ;
    }

    public function scopeWebVisible(Builder $query) {
        return $query
            ->where('is_visible', '1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->where('WebSiloOverrides.is_visible', '=', 1);
                    })
                ;
            })
            ->whereHas('webFamilies', function ($webFamiliesQuery) {
                return $webFamiliesQuery->webVisible();
            })
        ;
    }

    public function scopeWebSiloApproved(Builder $query) {
        return $query
            ->whereHas('webFamilies', function ($webFamiliesQuery) {
                return $webFamiliesQuery->webSiloApproved();
            })
            ;
    }

    public function slugUrl() {
        return route('collection.slug', [$this->slug]);
    }

    public function getParentRelationName() {
        return 'webCategories';
    }

    public function getPrimaryImage() {
        $asset = array_get($this, 'webSiloOverride.primaryImage', null);
        if (!is_null($asset)) return $asset;

        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webFamily = $this->visibleWebFamilies->first();
        if ($webFamily) return $webFamily->getPrimaryImage();
    }

    public function getIsWebVisibleAttribute() {
        return static::webVisible()->where('id', $this->id)->count() > 0;
    }


    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait, ManyToManyRevisionableTrait, HasWebSiloOverrides;
}
