<?php

namespace Hilco\Models;

use Illuminate\Auth\Authenticatable;
use Illuminate\Auth\Passwords\CanResetPassword;
use Illuminate\Contracts\Auth\Access\Authorizable as AuthorizableContract;
use Illuminate\Contracts\Auth\Authenticatable as AuthenticatableContract;
use Illuminate\Contracts\Auth\CanResetPassword as CanResetPasswordContract;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Foundation\Auth\Access\Authorizable;
use Venturecraft\Revisionable\RevisionableTrait;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * Hilco\Models\WebUser
 *
 * @property integer $id
 * @property string $email
 * @property string $password
 * @property string $name
 * @property integer $customer_id
 * @property string $remember_token
 * @property string $authy_status
 * @property string $authy_id
 * @property string $date_created
 * @property string $date_modified
 * @property string $date_uploaded
 * @property string $deleted_at
 * @property string $last_login
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereEmail($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser wherePassword($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereName($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereCustomerId($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereRememberToken($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereDateCreated($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereDateModified($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereDateUploaded($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereDeletedAt($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereLastLogin($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereAuthyStatus($value)
 * @method static \Illuminate\Database\Query\Builder|\Hilco\Models\WebUser whereAuthyId($value)
 * @method static WebUser firstOrNew($attributes)
 * @mixin \Eloquent
 * @property-read \Hilco\Models\Customer $customer
 */
class WebUser extends WebModel implements AuthenticatableContract, AuthorizableContract, CanResetPasswordContract {
	use Authenticatable, Authorizable, CanResetPassword;

	protected $table = "WebUsers";
	protected $fillable = ['email', 'password', 'username', 'name', 'customer_id', 'is_enabled', 'webRoles', 'webRoleIds', 'webPermissionIds', 'needs_password_reset', 'default_websilo_id', 'parent_webuser_id', 'commit_sequence'];
	protected $hidden = ['password', 'remember_token'];
    protected $with = ['customer.segments', 'customer.discounts'];

    public function scopeUsername($query, $username) {
        return $query->where('username', $username);
    }

	public function customer() {
		return $this->belongsTo(Customer::class, 'customer_id', 'id');
	}

	public function payment_profile()
	{
		return $this->hasOne(UserPaymentProfile::class, 'user_id', 'id');
	}

    public function webAuthEvents() {
	    return $this->hasMany(WebAuthEvent::class, 'email', 'email');
    }

	public function webRoles()
    {
        return $this->belongsToMany(WebRole::class, 'WebRole_WebUser', 'webuser_id', 'webrole_id');
    }

    public function webPermissions() {
        return $this->belongsToMany(WebPermission::class, 'WebPermission_WebUser', 'webuser_id', 'webpermission_id');
    }

    /**
     * @return BelongsToMany
     */
    public function webSilo() {
        return $this
            ->belongsToMany(WebSilo::class, 'WebSilo_WebUser', 'webuser_id', 'websilo_id')
            ->withPivot(['spending_limit', 'spending_period', 'minimum_order'])
            ;
    }

    public function webSilos() {
        return $this
            ->belongsToMany(WebSilo::class, 'WebSilo_WebUser', 'webuser_id', 'websilo_id')
            ->withPivot(['spending_limit', 'spending_period', 'minimum_order'])
            ;
    }

    public function getWebSiloAttribute() {
        $webSilo = $this->getRelationValue('webSilo')->first();
        if ($webSilo) return $webSilo;
        return WebSilo::default();
    }

    public function defaultWebSilo() {
        return $this->belongsTo(WebSilo::class, 'default_websilo_id', 'id');
    }

    public function parentWebUser() {
        return $this->belongsTo(WebUser::class, 'parent_webuser_id', 'id');
    }

    public function favoriteFamilies() {
        return $this
            ->belongsToMany(WebFamily::class, 'WebFamily_WebUser', 'webuser_id', 'webfamily_id')
            ->webVisible()
            ->with('visibleWebParts')
        ;
    }

	public function can($permissionSlug, $arguments = []) {
        return $this->checkPermission($permissionSlug);
    }

    public function hasRole($roleSlug) {
        return $this->webRoles->where('slug', $roleSlug)->count() > 0;
    }

    public function hasDirectPermission($permissions) {
        if ($permissions instanceof WebPermission) $permissions = $permissions->slug;
        if (!is_array($permissions)) $permissions = [$permissions];
        return $this->webPermissions->whereIn('slug', $permissions)->count() > 0;
    }

    public function hasRolePermission($permissions) {
        if ($permissions instanceof WebPermission) $permissions = $permissions->slug;
        if (!is_array($permissions)) $permissions = [$permissions];
        return $this->webRoles->pluck('webPermissions')->flatten()->whereIn('slug', $permissions)->count() > 0;
    }

	protected function checkPermission($permissionSlug) {
        return $this->effective_permissions->where('slug', $permissionSlug)->count() > 0;
    }

	public function getEffectivePermissionsAttribute() {
	    $rolePemissions = $this->webRoles->pluck('webPermissions')->flatten()->keyBy('id');
        $userPermissions = $this->webPermissions->keyBy('id');
        return $rolePemissions->merge($userPermissions);
	}

	public function userPhone() {
		$this->hasOne(CustomerPhone::class, 'customer_id', 'customer_id');
	}


    public function hasActiveCustomer() {
        return !is_null($this->active_customer);
    }

    public function setWebRolesAttribute($webRoleIds) {
        $this->save();
        $this->webRoles()->sync($webRoleIds);
    }

    public function setWebRoleIdsAttribute($ids) {
        return $this->setWebRolesAttribute($ids);
    }

    public function getWebRoleIdsAttribute() {
        return $this->webRoles->pluck('id')->toArray();
    }

    public function setWebPermissionIdsAttribute($ids) {
        $this->save();
        $this->webPermissions()->sync($ids);
    }

    public function webFeedbackResponses() {
	    return $this->hasMany(WebFeedbackResponse::class, 'webuser_id', 'id');
    }

    public function getTaxes($shippingAddress) {
        $taxes = [];
        $customer = b2b()->activeCustomer();
        if (is_null($customer)) return 0;

        $segment = $customer->activeSegment;
        if (is_null($segment)) return 0;

        $additionalTaxCustomerSegments = $segment->additionalTaxCustomerSegment;
        if ($additionalTaxCustomerSegments) {
            foreach ($additionalTaxCustomerSegments as $additionalTaxCustomerSegment) {
                $additionalTaxes = $additionalTaxCustomerSegment->additionalTax;
                if ($additionalTaxes) {
                    foreach ($additionalTaxes as $additionalTax) {
                        $taxes[] = $additionalTax->tax_percentage;
                    }
                }
            }
        }

        $tax = StateTax::inEffect()->whereState($shippingAddress->state)->first();
        if ($tax) {
            $taxes[] = $tax->tax_percentage;
        }

        return $taxes;
    }

    public function getHasActiveCustomerSegmentAttribute() {
	    $customer = $this->customer;
	    if (is_null($customer)) return false;

	    return !is_null($customer->activeSegment);
    }

    public function isUKM3User() {
        $customer = $this->customer;

        $webSilo = (isset($customer) ? $customer->webSilo : null);
        $customerSegment = (isset($customer) ? $customer->active_segment : null);
        $division = (isset($customerSegment) ? $customerSegment->division : null);

        if (isset($webSilo) && ($webSilo->slug == 'leightons-opticians' || $webSilo->slug == 'specsavers')) {
            return true;
        }

        if (isset($division) && $division == Division::UK_DIVISION) {
            return true;
        }

        return false;
    }

    public function getPrimarySalesRep() {
        $activeCustomerSegment = $this->customer->getActiveSegmentAttribute();

        if ($activeCustomerSegment != null && $activeCustomerSegment->primarySalesRep != null) {
            return $activeCustomerSegment->primarySalesRep->pointman_name;
        } else {
            return '';
        }

    }

    public function getLastSalesCallDate() {
        $activeCustomerSegment = $this->customer->getActiveSegmentAttribute();
        if ($activeCustomerSegment != null) {
            $latestCall = $activeCustomerSegment->getLatestCallAmongAffiliates();
            if ($latestCall != null) {
                $date = $latestCall->start_time;
                return date('Y-m-d', strtotime($date));
            }
        }

        return '';
    }

    public function getLicenseExpiration() {
        if ($this->customer->pharmacyInfo != null) {
            if ($this->customer->pharmacyInfo->license_expiration_date != '0000-00-00') {
                return $this->customer->pharmacyInfo->license_expiration_date;
            }
        } else {
            return '';
        }
    }

    public function getRegion() {
        return $this->customer->country;
    }

    public function getBuyingGroupName() {
        $activeCustomerSegment = $this->customer->getActiveSegmentAttribute();

        if ($activeCustomerSegment != null) {
            return $activeCustomerSegment->bill_cust_name;
        }

        return '';
    }

    public function getMicrosite() {
        $webSilo = $this->getWebSiloAttribute();

        return $webSilo->name;
    }

    public function get4PDesignation() {
        return '';
    }

    public function getRewardsTier() {
        return rewards()->tierName();
    }

    public function getTotalSpend() {
        return rewards()->totalSpending();

    }

    public function getCategoryGroupSpendingArray() {
        return rewards()->productCategoryGroups($this->customer);
    }

    public function getSpendNeeded() {
        return rewards()->calculateSpendingToUpgrade();
    }

    public function getCategoriesNeeded() {
        return rewards()->calculateCategoriesToUpgrade();
    }

    public function getCustomerCategory() {
        $customer = $this->customer;
        $customerCategory = $customer->overrideCustomerCategory ? $customer->overrideCustomerCategory : $customer->customerCategory;

        return $customerCategory->cust_category;
    }

    use RevisionableTrait, HasCommitSequence, BypassWith;
}
