<?php

namespace Hilco;

use Debugbar;
use Hilco\Models\ApplicationTranslation;
use Hilco\Models\ApplicationTranslationKey;
use Hilco\Models\AvailableLanguage;
use Hilco\Models\AustraliaFreightPostCode;
use Hilco\Models\BannedPartsCustomer;
use Hilco\Models\BannedPartsState;
use Hilco\Models\Carrier;
use Hilco\Models\Customer;
use Hilco\Models\CustomerCreditProfile;
use Hilco\Models\DeliveryMethodTerm;
use Hilco\Models\LegacyCarrierCodeMapping;
use Hilco\Models\Order;
use Hilco\Models\Part;
use Hilco\Models\Plant;
use Hilco\Models\Promotion;
use Hilco\Models\PromotionCodeTrigger;
use Hilco\Models\RewardsProductCategoryGroupDiscount;
use Hilco\Models\Translations_WebCollection;
use Hilco\Models\Translations_WebGroup;
use Hilco\Models\VATTax;
use Hilco\Models\WebAsset;
use Hilco\Models\WebCart;
use Hilco\Models\WebCategory;
use Hilco\Models\WebCollection;
use Hilco\Models\WebFamily;
use Hilco\Models\WebGroup;
use Hilco\Models\WebPart;
use Hilco\Models\WebRole;
use Hilco\Models\WebSilo;
use Hilco\Models\WebUrl;
use Hilco\RateShopHelper\RateShop_v2;
use Hilco\Shipments\Rate;
use HilcoB2B\M3Request;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Hilco\Facades\RateShop;
use GuzzleHttp\Client;
use Hilco\GuzzleWrappers\APIGuzzleWrapper;
use Hilco\XMLBuilder\RateShopUtility;


class B2BHelper
{
    protected $activeCustomer;
    protected $activeWebSilo;
    protected $activeShippingAddress;
    protected $activeBillingCustomer;
    protected $activeCarrier;
    protected $activeRate;
    protected $activePlant;
    protected $activePlants;
    protected $activeLocale;
    protected $activeLanguage;
    protected $activePromoCode;
    protected $rewardsProductCategoryGroupDiscounts;
    protected $activeWebGroupsForLayout;
    protected $activeWebTranslations;

    public function __construct()
    {
        $this->activeShippingAddress = null;
        $this->activeBillingCustomer = null;
        $this->activeCarrier = null;
        $this->activeRate = null;
        $this->activePlant = null;
        $this->activePlants = null;
        $this->activePromoCode = null;
        $this->activeWebGroupsForLayout = null;
        $this->activeWebTranslations = null;

        $this->loadActiveCustomer();
        if ($this->activeCustomer) {
            $this->loadActiveBillingCustomer();
            $this->loadActiveShippingAddress();
            $this->loadActiveCarrier();
            $this->loadActiveRate();
        }

        $this->loadActiveLocale();
        $this->loadActiveLanguage();
        $this->loadActiveWebSilo();
        $this->loadActivePlants();
        $this->loadRewardsProductCategoryGroupDiscounts();
        if ($this->activeLanguage != "en") {
            $this->loadActiveWebTranslations();
        }
    }

//  ******************************** LOAD ********************************
    protected function loadActiveCustomer()
    {
        $this->activeCustomer = false;

        $aliasedCustomerId = session()->get('b2b.aliasedCustomerId', false);
        if ($aliasedCustomerId) {
            $this->activeCustomer = Customer::with('segments.address.plant', 'discounts', 'availableShippingAddresses', 'webSilo')->find($aliasedCustomerId);
        } else {
            $user = auth()->user();
            if ($user) $this->activeCustomer = $user->customer;
        }
    }

    protected function loadActiveBillingCustomer()
    {
        $this->activeBillingCustomer = false;

        $activeBillingCustomerId = session()->get('b2b.activeBillingCustomerId', false);
        if ($activeBillingCustomerId) {
            $this->activeBillingCustomer = Customer::with('segments.address.plant', 'discounts', 'availableShippingAddresses')->find($activeBillingCustomerId);
        } else {
            $this->activeBillingCustomer = array_get($this->activeCustomer, 'billToCustomer', false);
        }
    }

    protected function loadActiveShippingAddress()
    {
        $this->activeShippingAddress = $this->activeCustomer->default_shipping_address;

        $activeShippingAddressId = session()->get('b2b.activeShippingAddressId', false);
        if ($activeShippingAddressId) {
            foreach ($this->activeCustomer->availableShippingAddresses as $address) {
                if ($address->id == $activeShippingAddressId) {
                    $this->activeShippingAddress = $address;
                    break;
                }
            }
        }
    }

    protected function loadActiveCarrier()
    {
        $this->activeCarrier = array_get($this->activeShippingAddress, 'defaultCarrier', false);

        $activeCarrierId = session()->get('b2b.activeCarrierId', false);
        if ($activeCarrierId) {
            $this->activeCarrier = Carrier::find($activeCarrierId);
        }
    }

    protected function loadActiveRate()
    {
        $this->activeRate = session()->get('b2b.activeRate', false);
    }

    protected function loadActiveWebSilo()
    {
        $this->activeWebSilo = false;
        $user = auth()->user();

        $activeWebSiloId = !is_null($user) ? session()->get('b2b.activeWebSiloId', false) : false;

        if (!$activeWebSiloId) {
            if ($user) {
                if ($this->activeCustomer) {
                    $defaultWebSilo = $this->activeCustomer->webSilo;
                    if ($defaultWebSilo && $defaultWebSilo->is_visible) {
                        $activeWebSiloId = $defaultWebSilo->id;
                    }
                }

                if (!$activeWebSiloId) {
                    $userSilo = $user->defaultWebSilo;
                    if ($userSilo) {
                        $activeWebSiloId = $userSilo->id;
                    }
                }
            }
            if (!$activeWebSiloId) {
                $query = WebSilo::with('webLandingPageWebSiloJoins');

                if ($this->activeCustomer) {
                    $cust_cat_summary = $this->activeCustomer->customerCategory->cust_cat_summary;
                    $divisions = $this->activeCustomer->divisions->pluck('id');
                    $query->whereHas('divisionRules', function ($query) use ($divisions) {
                        $query->whereIn('Divisions.id', $divisions);
                    })->whereHas('customerCategorySummaryRules', function ($query) use ($cust_cat_summary) {
                        $query->where('cust_cat_summary', $cust_cat_summary);
                    });
                } else {
                    $webUrl = WebUrl::current()->first();
                    if (!is_null($webUrl)) {
                        $query->whereHas('webUrls', function ($query) use ($webUrl) {
                            $query->where('id', '=', $webUrl->id);
                        });
                    }
                }
                $result = $query->first();
                if (!is_null($result)) $activeWebSiloId = $result->id;
            }

            if (!$activeWebSiloId) $activeWebSiloId = WebSilo::first()->id;
        }
        $this->activeWebSilo = WebSilo::with('webLandingPageWebSiloJoins')->find($activeWebSiloId);
        session()->put('b2b.activeWebSiloId', $activeWebSiloId);

        return $this->activeWebSilo;
    }

    public function loadActiveLanguage()
    {
        $this->activeLanguage = false;

        $activeLanguage = session()->get('b2b.activeLanguage', false);
        if ($activeLanguage) {
            $this->activeLanguage = $activeLanguage;
        } else {
            switch ($this->activeCountry()) {
                case 'GB':
                    $this->activeLanguage = 'en_GB';
                    break;
                default:
                    $this->activeLanguage = config('app.locale', 'en');
                    break;
            }
        }
        if (!in_array($this->activeLanguage, ['en', 'fr', 'ts', 'foo'])) $this->activeLanguage = 'en';
        $oldLanguage = App::getLocale();

        if ($oldLanguage != $this->activeLanguage) {
            Debugbar::addMessage("Setting language from $oldLanguage to {$this->activeLanguage}");
            App::setLocale($this->activeLanguage);
        }
    }

    public function loadActiveWebTranslations()
    {
        $this->activeWebTranslations = false;
        // Fetch the WebGroups.

        // For some reason I kept getting "unable to process request" issues when I was initially calling down the web silo.
        $webGroupTranslations = Translations_WebGroup::where('language', '=', $this->activeLanguage)->get();
        $webCollectionTranslations = Translations_WebCollection::where('language', '=', $this->activeLanguage)->get();
        $activeWebTranslations = ['webCollectionTranslations' => $webCollectionTranslations];
        $this->activeWebTranslations = $activeWebTranslations;
    }

    public function loadActiveLocale()
    {
        $this->activeLocale = false;

        $activeLocale = session()->get('b2b.activeLocale', false);
        if ($activeLocale) {
            $this->activeLocale = $activeLocale;
        } else {
            switch ($this->activeCountry()) {
                case 'GB':
                    $this->activeLocale = 'en_GB';
                    break;
                default:
                    $this->activeLocale = $this->defaultLocale();
                    break;
            }
        }
        Debugbar::addMessage("Setting locale to {$this->activeLocale}");
    }

    public function loadActivePlants() {
        $this->activePlants = false;

        $activePlantsIds = session()->get('b2b.activePlantsIds', false);

        if ($activePlantsIds) {
            $this->activePlants = Plant::findMany($activePlantsIds);
        } else {
            $activePlantCodes = [];

            if ($this->activeWebSilo()->is_outdoor_site) {
                $activePlantCodes = [Plant::PLAINVILLE_CODE, Plant::BOZEMAN_CODE];
            } else {
                $activeSegment = $this->activeSegment();
                $customerPlantCode = array_get($activeSegment, 'plant.plant', array_get(Plant::defaultPlant(), 'plant'));
                $activePlantCodes[] = $customerPlantCode;
                // Extra check to make sure Plainville is added to the list because Montreal is technically shut down
                // NOTE: I don't think there will be customers with default plant "Montreal" in M3 -- ntaylor 20200221
//                if ($customerPlantName == 'MONTREAL') {
//                    $activePlantCodes[] = 'PLAINVILLE';
//                }

                // Extra check because the Kaiser microsite technically has some stock in Dallas
                // use 'strpos() !== false' because strpos() returns offset str position if found
                // (so 0 is technically valid) and false if not found at all
                if (strpos($this->activeWebSilo()->slug, 'kaiser') !== false) {
                    $activePlantCodes[] = Plant::DALLAS_CODE;
                }
            }
            $this->activePlants = Plant::whereIn('plant', $activePlantCodes)->get();
        }

        $activePlantsIds = [];
        foreach ($this->activePlants as $plant) {
            $activePlantsIds[] = $plant->id;
        }
        session()->put('b2b.activePlantsIds', $activePlantsIds);
    }

    public function loadRewardsProductCategoryGroupDiscounts()
    {
        $rewardsProductCategoryGroupDiscounts = session()->get('b2b.rewardsProductCategoryGroupDiscounts', false);
        $rewardsProductCategoryGroupDiscounts = RewardsProductCategoryGroupDiscount::all();
        $this->rewardsProductCategoryGroupDiscounts = $rewardsProductCategoryGroupDiscounts;
    }
//  ******************************** LOAD ********************************


//  ******************************** SETTERS ********************************
    public function setActiveShippingAddress($activeShippingAddressId = false)
    {
        if ($activeShippingAddressId) {
            session()->put('b2b.activeShippingAddressId', $activeShippingAddressId);
        } else {
            session()->forget('b2b.activeShippingAddressId');
        }
        $this->loadActiveShippingAddress();
    }

    public function setActiveBillingCustomer($billingCustomerId = false)
    {
        if ($billingCustomerId) {
            session()->put('b2b.activeBillingCustomerId', $billingCustomerId);
        } else {
            session()->forget('b2b.activeBillingCustomerId');
        }
        $this->loadActiveBillingCustomer();
    }

    public function setActiveCarrier($activeCarrierId = false)
    {
        if ($activeCarrierId) {
            session()->put('b2b.activeCarrierId', $activeCarrierId);
        } else {
            session()->forget('b2b.activeCarrierId');
        }
        $this->loadActiveCarrier();
    }

    public function setActiveRate($activeRate = false)
    {
        if ($activeRate) {
            session()->put('b2b.activeRate', $activeRate);
        } else {
            session()->forget('b2b.activeRate');
        }
        $this->activeRate = $activeRate;
    }

    public function setActiveLanguage($language)
    {
        session()->put('b2b.activeLanguage', $language);
        $this->loadActiveLanguage();
    }

    public function setActiveLocale($locale)
    {
        session()->put('b2b.activeLocale', $locale);
        $this->loadActiveLocale();
    }

    public function setActivePromoCode($promoCode)
    {
        session()->put('b2b.activePromoCode', $promoCode);
        $this->activePromoCode = $promoCode;
    }

    public function setRewardsProductCategoryGroupDiscounts($rewardsProductCategoryGroupDiscounts)
    {
        session()->put('b2b.rewardsProductCategoryGroupDiscounts', $rewardsProductCategoryGroupDiscounts);
        $this->rewardsProductCategoryGroupDiscounts = $rewardsProductCategoryGroupDiscounts;
    }
//  ******************************** SETTERS ********************************


//  ******************************** GETTERS ********************************
    public function activeCustomer()
    {
        return $this->activeCustomer;
    }

    public function activeBillingCustomer()
    {
        return $this->activeBillingCustomer;
    }

    public function activeShippingAddress()
    {
        return $this->activeShippingAddress;
    }

    public function activeCarrier()
    {
        return $this->activeCarrier;
    }

    /**
     * @return Rate
     */
    public function activeRate()
    {
        return $this->activeRate;
    }

    public function activeCountry()
    {
        if ($this->activeShippingAddress) return $this->activeShippingAddress()->country;

        return null;
    }

    public function activePriceList()
    {
        $activePriceList = 'Catalog';
        if ($this->activeCustomer) {
            $customerPriceList = $this->activeCustomer->default_price_list;
            if ($customerPriceList) $activePriceList = $customerPriceList;
        }

        return $activePriceList;
    }

    public function activeCurrency()
    {
        $activeCurrency = 'USD';

        if ($this->activeCustomer) {
            $activeCurrency = $this->activeCustomer->currency;
        }

        return $activeCurrency;
    }

    public function activeWebHierarchy()
    {
        return $this->activeWebSilo()->webHierarchy;
    }

    /** @return WebSilo */
    public function activeWebSilo()
    {
        return $this->activeWebSilo;
    }

    public function activeLocale()
    {
        return $this->activeLocale;
    }

    public function activeLanguage()
    {
        return $this->activeLanguage;
    }

    public function activeSegment() {
        $activeCustomer = $this->activeCustomer();

        return array_get($activeCustomer, 'activeSegment');
    }

    public function activePlant() {
        $activePlant = null;
        $activeSegment = $this->activeSegment();

        $activePlant = array_get($activeSegment, 'plant', Plant::defaultPlant());

        return $activePlant;
    }

    public function activePlants() {
        return $this->activePlants;
    }

    public function activePlantNames() {
        $plantNames = [];
        foreach ($this->activePlants as $activePlant) {
            $plantNames[] = $activePlant->plant;
        }
        return $plantNames;
    }

    public function activePromoCode()
    {
        return $this->activePromoCode;
    }

    public function rewardsProductCategoryGroupDiscounts()
    {
        return $this->rewardsProductCategoryGroupDiscounts;
    }

    public function activeWebTranslations()
    {
        return $this->activeWebTranslations;
    }

//  ******************************** GETTERS ********************************


    public function aliasAs($customerId = false)
    {
        session()->forget('b2b');

        if ($customerId) {
            session()->put('b2b.aliasedCustomerId', $customerId);
        }

        $this->loadActiveCustomer();
    }

    public function isAliased()
    {
        return (session()->get('b2b.aliasedCustomerId', false));

    }

    public function stopAliasing()
    {
        $this->aliasAs(false);
    }

    public function currencySymbol($currencyCode = false)
    {
        if ($currencyCode === false) $currencyCode = $this->activeCurrency();
        if ($currencyCode == 'GBP') return '£';

        return '$';
    }

    public function formatPrice($price, $decimals = 2, $symbol = false)
    {
        if ($symbol === false) $symbol = $this->currencySymbol();
        if ($decimals == -1) {
            if ($price == round($price)) $decimals = 0;
            else $decimals = 2;
        }
        return is_string($price) ? $price : $symbol . number_format($price, $decimals);
    }

    public function dateFormatString($division = false)
    {
        if ($division === false) $division = $this->activeCustomer()->activeSegment->division;

        $format = 'm/d/Y';
        switch ($division) {
            case 'HILCO-UK':
                $format = 'd/m/Y';
                break;
            case 'HILCO-US':
            case 'WILSON-US':
                $format = 'm/d/Y';
        }

        return $format;
    }

    public function isOrderable(Model $model)
    {
        if ($model instanceof WebFamily) return $this->isWebFamilyOrderable($model);
        else if ($model instanceof WebPart) return $this->isWebPartOrderable($model);
        else if ($model instanceof Part) return $this->isPartOrderable($model);

        return false;
    }

    public function isWebFamilyOrderable(WebFamily $webFamily)
    {
        $isOrderable = true;
        foreach ($webFamily->visibleWebParts as $webPart) {
            if (!$this->isWebPartOrderable($webPart)) $isOrderable = false;
        }
        return $isOrderable;
    }

    public function isWebPartOrderable(WebPart $webPart)
    {
        $part = $webPart->part;
        if (is_null($part)) return false;
        return $this->isPartOrderable($part);
    }

    public function isPartOrderable(Part $part)
    {
        if ($this->activeCountry() != 'US' && $this->activeCountry() != 'GB') {
            if ($part->isRX) return false;
        }
        return true;
    }

    public function allowCreditCardOrders()
    {
        return (bool)
            !$this->activeRate()->isError() &&
            $this->orderCreditEligible();
    }

    public function orderCreditEligible(){
        return (bool)
            $this->activeWebSilo->allow_credit_cards &&
            $this->isBillingDirect();
    }

    public function allowDirectBilling()
    {
        if (!$this->isBillingDirect() && !$this->allowBillingChange()) {
            return false;
        }

        return (bool)$this->activeWebSilo->allow_direct_billing;
    }

    public function allowGroupBilling()
    {
        if ($this->isBillingDirect() && !$this->allowBillingChange()) {
            return false;
        }
        return (bool)$this->activeWebSilo->allow_group_billing;
    }

    public function allowBillingChange()
    {
        if ($this->activeCountry() == 'GB') return false;
        else return true;
    }

    public function requireApprovalSiloParts()
    {
        return (bool)array_get($this->activeCustomer, 'webSilo.pivot.require_approval_silo_parts', false);
    }

    public function requireApprovalHilcoParts()
    {
        return (bool)array_get($this->activeCustomer, 'webSilo.pivot.require_approval_hilco_parts', false);
    }

    public function allowPOOrders()
    {
        $terms = array_get($this->activeBillingCustomer, 'activeSegment.def_pmt_terms');
        return !in_array($terms, ['CC', 'CR CARD', 'CR CARD+BAL']);
    }

    public function isPONumberRequired()
    {
        if ($this->activeBillingCustomer->activeSegment->division == 'HILCO-US') return false;

        return true;
    }

    public function creditProfiles($industry, $billTo = null) {
        $creditProfiles = [];
        try {
            $client = new Client(['base_uri' => config('cenpos.base_uri')]);
            $profiles = json_decode($client->post('api/GetToken/', ['form_params' => ['verifyingpost' => $this->generateVerifyingPostParam($industry, 'getToken', null, null, null, $billTo)],
                'headers' => ['Content-Type' => 'application/x-www-form-urlencoded']])->getBody()->getContents(), true);
            foreach($profiles['Tokens'] as $token){
                $creditProfiles[] = [
                    'id' => $token['TokenId'],
                    'text' => $token['CardNumber'] . ' - ' . $token['CardExpirationDate'],
                    'last4' => $token['CardNumber'],
                    'expDate' => substr_replace($token['CardExpirationDate'], '/', 2, 0),
                    'cardType' => $token['CardType'],
                ];
            }
            $creditProfiles[] = ['id' => 0, 'text' => 'Use new Card'];
        } catch (\Exception $e) {}

        return $creditProfiles;
    }

    public function generateVerifyingPostParam($industry, $operation = null, $orderNumber = null, $authAmount = null, $token = null, $billTo = null){
        $client = new Client(['base_uri' => config('cenpos.base_uri')]);
        $params = [
            'secretkey' => config('cenpos.secret_key'),
            'merchant' => config('cenpos.merchant_id'),
            'industry' => $industry
        ];
        if($operation){
            if($operation != 'getToken' && $operation != 'voidTrx'){
                $params['tokenid'] = $token;
            }

            if($operation == 'useToken' || $operation == 'useCrypto'){
                $custNo = $this->activeBillingCustomer ? $this->activeBillingCustomer->cust_no : array_get($billTo, 'custNo');
                $currencyCode = Customer::whereCustNo($custNo)->first()->currency;
                $params['currencycode'] = $currencyCode;
            }

            switch ($operation){
                case 'getToken':
                    $params['customercode'] = $this->activeBillingCustomer ? $this->activeBillingCustomer->cust_no : array_get($billTo, 'custNo');
                    break;
                case 'useToken':
                    $params['amount'] = $authAmount;
                    $params['invoicenumber'] = $orderNumber;
                    $params['city'] = array_get($billTo, 'city');
                    $params['state'] = array_get($billTo, 'state');
                    $params['address'] = array_get($billTo, 'address');
                    $params['zipcode'] = array_get($billTo, 'zipcode');
                    $params['type'] = 'Auth';
                    $params['customercode'] = $this->activeBillingCustomer ? $this->activeBillingCustomer->cust_no : array_get($billTo, 'custNo');
                    break;
                case 'deleteToken':
                    break;
                case 'useCrypto':
                    $params['amount'] = $authAmount;
                    $params['invoicenumber'] = $orderNumber;
                    $params['city'] = array_get($billTo, 'city');
                    $params['state'] = array_get($billTo, 'state');
                    $params['address'] = array_get($billTo, 'address');
                    $params['zipcode'] = array_get($billTo, 'zipcode');
                    $params['type'] = 'Auth';
                    $params['customercode'] = $this->activeBillingCustomer ? $this->activeBillingCustomer->cust_no : array_get($billTo, 'custNo');
                    break;
                case 'convertCrypto':
                    $params['city'] = array_get($billTo, 'city');
                    $params['state'] = array_get($billTo, 'state');
                    $params['address'] = array_get($billTo, 'address');
                    $params['zipcode'] = array_get($billTo, 'zipcode');
                    $params['customercode'] = $this->activeBillingCustomer ? $this->activeBillingCustomer->cust_no : array_get($billTo, 'custNo');
                    break;
                case 'voidTrx':
                    $params['referencenumber'] = $token;
                    break;
                default:
                    break;
            }
        }

        return array_get(json_decode($client->post('', ['query' => ['app' => 'genericcontroller', 'action' => 'siteVerify'], 'form_params' => $params,
            'headers' => ['Content-Type' => 'application/x-www-form-urlencoded']])->getBody()->getContents(), true), 'Data');
    }

    public function isRewardsEnabled()
    {
        if (!config('rewards.rewardsEnabled')) return false;
        return (bool)$this->activeWebSilo->allow_rewards;
    }

    public function isBillingDirect()
    {
        return $this->activeCustomer->id == $this->activeBillingCustomer->id;
    }

    public function isCustomerTaxable()
    {
        return array_get($this->activeBillingCustomer, 'activeSegment.cust_txbl', false);
    }

    public function availableShippingAddresses()
    {
        $defaultAddress = $this->activeCustomer->default_shipping_address;
        $addresses = collect([$defaultAddress]);
        if ($this->activeWebSilo()->allow_drop_shipping) {
            if(!empty($this->activeCustomer->active_segment->def_ship_from)) {
                $availableAddresses = $this->activeCustomer->availableShippingAddresses;
                if (!empty($availableAddresses->toArray())) {
                    $addresses = $availableAddresses->sort(function ($a, $b) use ($defaultAddress) {
                        if ($a->id == $defaultAddress->id) return -1;
                        if ($b->id == $defaultAddress->id) return 1;

                        return strcmp($a->cust_name, $b->cust_name);
                    });
                }
            }
        }

        return $addresses;
    }

    public function isValidPromoCode($promoCode)
    {
        // does promo code even belong to an existing codeTrigger
        $codeTriggers = PromotionCodeTrigger::where('code', '=', $promoCode)->where('deleted_at', '=', '0000-00-00 00:00:00')->get();
        if ($codeTriggers == null) {
            return false;
        }
        /**
         * Original Case: We pull up a list of codetriggers, and just pull from the first.
         * Now, we need to conisder ALL of the CodeTriggers that have this code.
         * To do this successfully, we need to evaluate any and all promotions assosciated with this code.
         * If at least one of them are valid, let it go through.
         * Otherwise, return invalid.
         */

        foreach ($codeTriggers as $codeTrigger) {
            $promotion = $codeTrigger->promotion()->first();
            // Is this an active Promo?
            if (!$this->existsInActivePromos($promotion)) {
                continue;
            }
            $promotionIsValid = true;
            foreach ($promotion->triggers as $trigger) {
                if ($trigger->trigger_type != 'codeTrigger') {
                    // Is this triggered?
                    if (!$trigger->details->isTriggered($this->activeCustomer, WebCart::getWebCartItems(auth()->user()))) {
                        // Then mark it off. This promo isn't acceptable,
                        // Unless of course it has an Almost Qualifying Field.
                        if ($trigger->details->hasAlmostQualifyingField()) {
                            continue;
                        }
                        $promotionIsValid = false;
                    }
                }
            }
            if ($promotionIsValid) {
                return true;
            }

        }

        return false;
    }

    // TODO: Come up with a better name for this, probably.
    // Checks for promotions that are code triggered, but still have un-triggered triggers (ugh)
    // This requires a seperate display for the checkout controller to indicate this when the code is input.
    // We could possible expand this to display the kind of promotion it is and the specifics if needed.
    public function nonTriggeredCodeTriggeredPromotions($customerTriggeredPromotions, $triggeredPromotions){
        $nonTriggeredCodeTriggeredPromotions = [];
        $billingCustomer = b2b()->activeBillingCustomer;
        $webCartItems = WebCart::getWebCartItems(auth()->user());
        
        foreach($customerTriggeredPromotions as $customerTriggeredPromotion){
            $codeTrigger = $customerTriggeredPromotion->codeTriggers()->first();
            if(is_null($codeTrigger)){
                continue;
            }
            foreach($triggeredPromotions as $triggeredPromotion){
                if($customerTriggeredPromotion->id === $triggeredPromotion->id){
                    continue 2;
                }
            }
            // Compatible for the event that we ever let them have multiple promo codes. Probably.
            if($codeTrigger->isTriggered($billingCustomer, $webCartItems)){
                $nonTriggeredCodeTriggeredPromotions[$codeTrigger->code] = $customerTriggeredPromotion;
            }

        }
        return $nonTriggeredCodeTriggeredPromotions;

    }

    private function existsInActivePromos($promotion) {
        foreach (Promotion::getActivePromotions() as $activePromotion) {
            if ($activePromotion->id === $promotion->id) return true;
        }
        return false;
    }

//  ******************************** RATE SHOP ********************************

    /**
     * @return Collection|Rate[]
     */
    public function rateShop()
    {
        if ($this->activeWebSilo()->allow_rate_shopping && !b2b()->doesCartHaveBannedProductRestrictions()) {
            $oldRates = $this->oldRateShop();
        }

        $siloCodesString = $this->activeWebSilo()->allowed_carrier_codes;
        $siloCodes = [];
        if (strlen($siloCodesString)) {
            $siloCodes = explode(',', $siloCodesString);
            foreach ($oldRates as $index => $oldRate) {
                if (!in_array(array_get($oldRate, 'hilco_carrier_code'), $siloCodes)) {
                    unset($oldRates[$index]);
                }
            }
        }
//
//
//        $carrierCodes = [];
//        foreach ($oldRates as $oldRate) {
//            $code = array_get($oldRate, 'hilco_carrier_code', false);
//            if ($code) {
//                if (!count($siloCodes) || in_array($code, $siloCodes)) {
//                    $carrierCodes[$code] = $code;
//                }
//            }
//        }
//        $carriers = Carrier::codes($carrierCodes)->get()->keyBy('carrier_code');

        $methodCodes = [];
        foreach($oldRates as $oldRate){
            $code = array_get($oldRate, 'delivery_method', false);
            if($code){
                if (!count($siloCodes) || in_array($code, $siloCodes)) {
                    $methodCodes[$code] = $code;
                }
            }
        }
        $methodTerms = DeliveryMethodTerm::codes($methodCodes)->get()->keyBy('delivery_method_code');

        $activeRate = $this->activeRate();
        $activeRateCode = ($activeRate) ? $activeRate->code() : false;
        $wasRateSelected = false;

        $rates = new Collection();
        foreach ($oldRates as $oldRate) {
            $code = array_get($oldRate, 'delivery_method', false);
            if ($code === false) continue;
            $method = array_get($methodTerms, $code);
            $rate = array_get($oldRate, 'rate', 0);
            $isDefault = array_get($oldRate, 'is_default', false);
            $discount = $rate - array_get($oldRate, 'discounted_rate', 0);
            $tax = array_get($oldRate, 'tax_amount', 0);

            $rate = new Rate($method, $rate, $isDefault, $discount, $tax);
            $rate->setText(array_get($oldRate, 'carrier_desc'));

            if ($code == $activeRateCode) {
                $rate->selected = true;
                $wasRateSelected = true;
            }
            $rates->put($code, $rate);
        }

        $rates = $rates->sort(function (Rate $a, Rate $b) {
            $aSpecial = $a->isDefault() || $a->isDiscounted();
            $bSpecial = $b->isDefault() || $b->isDiscounted();

            if ($aSpecial && !$bSpecial) return -1; //a is special but b isn't, a goes first
            if (!$aSpecial && $bSpecial) return 1; //b is special but a isn't, b goes first

            //both or neither are special
            if ($a->rate() > $b->rate()) return 1;
            if ($a->rate() < $b->rate()) return -1;

            return 0;
        });

        if ($rates->count() == 0) {
            $rates->put('-1', new Rate(null, -1, true));
        }

        if ($activeRateCode === false || $wasRateSelected === false) {
            $firstRate = $rates->shift();
            $firstRate->selected = true;
            $this->setActiveRate($firstRate);
            $rates->prepend($firstRate, $firstRate->code());
        }

        return $rates;
    }

    protected function rateShopGroupCode()
    {
        $shippingAddress = $this->activeShippingAddress;
        $countryString = array_get($shippingAddress, 'country');
        return $countryString != null && $countryString == "CA" ? 2 : 3;
    }

    protected function getFromCode($shippingAddressObj, $groupCode)
    {
        if ($groupCode === 2) {
            return array_get($shippingAddressObj, 'plant.plantAddress.canada_zip');
        } else {
            return array_get($shippingAddressObj, 'plant.plantAddress.zip');
        }
    }

    public function getMultiWarehouseShipFrom($cartItems){
        $segment = b2b()->activeCustomer()->activeSegment;
        $atpRequestData = [];
        foreach ($cartItems as $cartItem) {
            $part = $cartItem['part'];
            $atpRequestData[] = $part->part_no;
        }

        $vegasResponse = WebPart::fetchAllItemsStockInfo($atpRequestData, 'U31');

        $allInStock = true;
        $allStockable = true;
        foreach($vegasResponse as $atp) {
            if($atp['stock'] == 0){
                $allInStock = false;
            }

            if(!$atp['stockable']){
                $allStockable = false;
            }
        }

        if($allInStock){
            return [
                'plant' => $segment->plant->plant,
                'id' => $segment->plant->id
            ];
        }else{
            $allInStock = true;
            $plant = Plant::where('plant', '=', 'U11')->where('deleted_at', '=', '0000-00-00 00:00:00')->first();

            $plainvilleResponse = WebPart::fetchAllItemsStockInfo($atpRequestData, $plant->plant);
            foreach($plainvilleResponse as $atp) {
                if($atp['stock'] == 0){
                    $allInStock = false;
                }

                if(!$allInStock){
                    break;
                }
            }

            if($allInStock || (!$allInStock && !$allStockable)){
                return [
                    'plant' => $plant->plant,
                    'id' => $plant->id
                ];
            }
        }

        return [
            'plant' => $segment->plant->plant,
            'id' => $segment->plant->id
        ];
    }

    protected function oldRateShop()
    {
        $customerSegment = $this->activeCustomer()->activeSegment;
        $billingSegment = $this->activeBillingCustomer->activeSegment;
        $isCusTax = isset($billingSegment) && $billingSegment->cust_txbl == 1;


        $shippingAddress = $this->activeShippingAddress;
        $toCountry = array_get($shippingAddress, 'country');
        $groupCode = $this->rateShopGroupCode();
        $fromCode = $customerSegment->plant->plantAddress->zip;
        $fromCountry = $customerSegment->plant->plantAddress->country;
        $defaultDeliveryMethod = array_get($shippingAddress, 'def_delivery_method');

        $cartStatus = App::make('cartSingleton')->getCartStatus();

        $cartItems = $cartStatus['items'];
        $excludedItems = [];
        $rewardsItems = [];
        $cartTotal = 0;
        $weights = 0;
        $excludedWeights = 0;
        $rewardsWeights = 0;
        $volume = 0;
        $excludedVolume = 0;
        $rewardsVolume = 0;
        $hideAir = false;
        foreach ($cartItems as $cartItem) {
            $quantity = $cartItem['quantity'];
            $part = $cartItem['part'];
            if ($part->isRewardsExcluded()) {
                $excludedItems[$cartItem['id']] = $cartItem;
            } else {
                $rewardsItems[$cartItem['id']] = $cartItem;
            }

            if (in_array($cartItem, $excludedItems)) {
                if (!isset($excludedWeights)) {
                    $excludedWeights = $part->weight_lbs * $quantity;
                } else {
                    $excludedWeights += $part->weight_lbs * $quantity;
                }
            } else {
                if (!isset($rewardsWeights)) {
                    $rewardsWeights = $part->weight_lbs * $quantity;
                } else {
                    $rewardsWeights += $part->weight_lbs * $quantity;
                }
            }

            if (!isset($weights)) {
                $weights = $part->weight_lbs * $quantity;
            } else {
                $weights += $part->weight_lbs * $quantity;
            }

            if (in_array($cartItem, $excludedItems)) {
                $excludedVolume += $part->vol_ci * $quantity;
            } else {
                $rewardsVolume += $part->vol_ci * $quantity;
            }
            $volume += $part->vol_ci * $quantity;

            //TODO: This is only a temp solution. we really don't want to parse the part description string for the hazmat code. should be coming from m3
            $hideAir = $hideAir || strpos($part->part_desc, "ORM-D") !== false;
            $cartTotal += $cartItem['extendedPrice'];
        }

        if (count($weights) > 0) {
            if ($customerSegment->plant->plant == 'U51') {
                /* AUS REQUESTED NO RATE SHOPPING */
                return [['id' => -1, 'text' => trans('hilco::app.unable_to_calculate_shipping_rates')]];
            } else if ($toCountry == 'GB') {
                /* UK REQUESTED NO RATE SHOPPING */
                return [['id' => -1, 'text' => trans('hilco::app.unable_to_calculate_shipping_rates')]];
            } else {
                //The list of possible methods to show to the customer
//                $methodCodes = DB::connection()
//                    ->table('RateShopGroups')
//                    ->where('group_code', '=', $groupCode)
//                    ->pluck('delivery_method');
//
//                $defaultDelivery = null;
//                if ($defaultDeliveryMethod != null) {
//                    if (!in_array($defaultDeliveryMethod, $methodCodes)) {
//                        $defaultDelivery = DB::connection()
//                            ->table('DeliveryMethod_DeliveryTerm')
//                            ->leftJoin('RateShopAPIDetails', 'delivery_method', '=', 'delivery_method_code')
//                            ->where('delivery_method_code', '=', $defaultDeliveryMethod)
//                            ->select()
//                            ->first();
//
//                        $methodCodes[] = $defaultDeliveryMethod;
//                    }
//                }
//
////                if (($defaultDeliveryMethod == 'U21' || $defaultDeliveryMethod == 'PGW' || $defaultDeliveryMethod == 'U03')
////                    && !in_array($defaultDeliveryMethod, $methodCodes)) {
////                    array_push($methodCodes, $defaultDeliveryMethod);
////                }
//
//                if ($defaultDeliveryMethod != null && ($defaultDeliveryMethod == 'U39' || (isset($defaultDelivery->super_code) && $defaultDelivery->super_code == '1400'))) {
//                    if (($key = array_search('U03', $methodCodes)) !== false) {
//                        unset($methodCodes[$key]);
//                    }
//                } else {
//                    if (($key = array_search('U39', $methodCodes)) !== false) {
//                        unset($methodCodes[$key]);
//                    }
//                }
//
//                if ($hideAir) {
//                    $airCodes = DB::connection()
//                        ->table('RateShopAPIDetails')
//                        ->whereIn('delivery_method', $methodCodes)
//                        ->where('is_air', '=', true)
//                        ->select('delivery_method')
//                        ->get();
//
//                    foreach ($airCodes as $airCode) {
//                        if (($key = array_search($airCode->delivery_method, $methodCodes)) !== false) {
//                            unset($methodCodes[$key]);
//                        }
//                    }
//                }

                try {
                    $client = null;
                    $guzzle = new M3Request();
                    if(empty($guzzle->client)){
                        Log::error('Exception Caught while performing ' . __FUNCTION__ . ': Guzzle Http Client does not exist. Base URI may be missing!');
                        return trans('messages.temporarily_unavailable');
                    }else{
                        $client = $guzzle->client;
                    }

                    if(count($excludedItems)){
                        $excludedRates = json_decode($client->get('v4/' . 'getRates', ['query' => ['requestFrom' => 'b2b', 'groupCode' => $groupCode,'weight' => $excludedWeights, 'volume' => $excludedVolume, 'toCode' => $shippingAddress->postal_cd, 'toCountry' => $toCountry, 'fromCode' => $fromCode, 'fromCountry' => $fromCountry], 'headers' => ['Accept' => 'application/json']])->getBody()->getContents(), true);
                        $excludedShipVia = $excludedRates['details'];
                        if(count($rewardsItems)){
                            $rewardsRates = json_decode($client->get('v4/' . 'getRates', ['query' => ['requestFrom' => 'b2b', 'groupCode' => $groupCode,'weight' => $rewardsWeights, 'volume' => $rewardsVolume, 'toCode' => $shippingAddress->postal_cd, 'toCountry' => $toCountry, 'fromCode' => $fromCode, 'fromCountry' => $fromCountry], 'headers' => ['Accept' => 'application/json']])->getBody()->getContents(), true);
                            $rewardsShipVia = $rewardsRates['details'];
                        }
                    }

                    $rates = json_decode($client->get('v4/' . 'getRates', ['query' => ['requestFrom' => 'b2b', 'groupCode' => $groupCode,'weight' => $weights, 'volume' => $volume, 'toCode' => $shippingAddress->postal_cd, 'toCountry' => $toCountry, 'fromCode' => $fromCode, 'fromCountry' => $fromCountry], 'headers' => ['Accept' => 'application/json']])->getBody()->getContents(), true);
                    $shipVia = $rates['details'];
                } catch (\Exception $e) {
                    Log::critical('Exception caught while getting shipping rates: ' . $e->getMessage());
                    return [];
                }

//                if (isset($defaultDelivery->super_code) && isset($defaultDelivery->code_type)) {
//                    dd($shippingAddress->def_ship_via);
//                    if ($shippingAddress->def_ship_via != null && strlen($shippingAddress->def_ship_via) > 0) {
//                        $codes = DB::connection()
//                            ->table('RateShopAPIDetails as myCode')
//                            ->join('RateShopAPIDetails as otherCode', 'myCode.code_type', '=', 'otherCode.code_type')
//                            ->join('DeliveryMethod_DeliveryTerm', 'delivery_method_code', '=', 'otherCode.delivery_method')
//                            ->where('myCode.delivery_method', '=', $defaultDeliveryMethod)
//                            ->where(DB::raw('otherCode.super_code IS NOT NULL'))
//                            ->whereIn('otherCode.delivery_method', $methodCodes)
//                            ->select(['otherCode.delivery_method'])
//                            ->get();
//
//                        foreach ($codes as $code) {
//                            if (isset($excludedShipVia) and count($excludedShipVia)) {
//                                $excludedShipVia[$code] = [
//                                    'api_carrier' => null,
//                                    'residential' => null,
//                                    'hilco_carrier_code' => $code,
//                                    'carrier_desc' => null,
//                                    'rate' => 2,
//                                    'text' => null,
//                                    'service' => null,
//                                    'delivery_date' => null,
//                                    'delivery_time' => null,
//                                    'guaranteed' => null,
//                                    'def_ship_via' => $shippingAddress->def_ship_via,
//                                    'is_default' => $code == $defaultDeliveryMethod,
//                                ];
//                            }
//                            if (isset($rewardsShipVia) and count($rewardsShipVia)) {
//                                $rewardsShipVia[$code] = [
//                                    'api_carrier' => null,
//                                    'residential' => null,
//                                    'hilco_carrier_code' => $code,
//                                    'carrier_desc' => null,
//                                    'rate' => 2,
//                                    'text' => null,
//                                    'service' => null,
//                                    'delivery_date' => null,
//                                    'delivery_time' => null,
//                                    'guaranteed' => null,
//                                    'def_ship_via' => $shippingAddress->def_ship_via,
//                                    'is_default' => $code == $defaultDeliveryMethod,
//                                ];
//                            }
//                            $shipVia[$code] = [
//                                'api_carrier' => null,
//                                'residential' => null,
//                                'hilco_carrier_code' => $code,
//                                'carrier_desc' => null,
//                                'rate' => 2,
//                                'text' => null,
//                                'service' => null,
//                                'delivery_date' => null,
//                                'delivery_time' => null,
//                                'guaranteed' => null,
//                                'def_ship_via' => $shippingAddress->def_ship_via,
//                                'is_default' => $code == $defaultDeliveryMethod,
//                            ];
//                        }
//                    }
//                } else {
//                    if (isset($excludedShipVia) and count($excludedShipVia)) {
//                        $excludedShipVia[$defaultDeliveryMethod] = [
//                            'api_carrier' => null,
//                            'residential' => null,
//                            'hilco_carrier_code' => $defaultDeliveryMethod,
//                            'carrier_desc' => null,
//                            'rate' => 2,
//                            'text' => null,
//                            'service' => null,
//                            'delivery_date' => null,
//                            'delivery_time' => null,
//                            'guaranteed' => null,
//                            'def_ship_via' => $shippingAddress->def_ship_via,
//                            'is_default' => true,
//                        ];
//                    }
//                    if (isset($rewardsShipVia) and count($rewardsShipVia)) {
//                        $rewardsShipVia[$defaultDeliveryMethod] = [
//                            'api_carrier' => null,
//                            'residential' => null,
//                            'hilco_carrier_code' => $defaultDeliveryMethod,
//                            'carrier_desc' => null,
//                            'rate' => 2,
//                            'text' => null,
//                            'service' => null,
//                            'delivery_date' => null,
//                            'delivery_time' => null,
//                            'guaranteed' => null,
//                            'def_ship_via' => $shippingAddress->def_ship_via,
//                            'is_default' => true,
//                        ];
//                    }
//                    $shipVia[$defaultDeliveryMethod] = [
//                        'api_carrier' => null,
//                        'residential' => null,
//                        'hilco_carrier_code' => $defaultDeliveryMethod,
//                        'carrier_desc' => null,
//                        'rate' => 2,
//                        'text' => null,
//                        'service' => null,
//                        'delivery_date' => null,
//                        'delivery_time' => null,
//                        'guaranteed' => null,
//                        'def_ship_via' => $shippingAddress->def_ship_via,
//                        'is_default' => true,
//                    ];
//                }
            }
//            else {
//                $shipmentArray = [];
//                $excludedShipmentArray = [];
//                $rewardsShipmentArray = [];
//                if (count($excludedItems)) {
//                    foreach ($excludedWeights as $excludedWeight) {
//                        $excludedShipmentArray[] = ['weight' => $excludedWeight, 'volume' => $excludedVolume, 'toCode' => $shippingAddress->postal_cd, 'fromCode' => $fromCode];
//                    }
//                    $excludedShipVia = RateShop::rateShopCanada($groupCode, $excludedShipmentArray);
//                }
//                if (count($rewardsItems)) {
//                    foreach ($rewardsWeights as $rewardsWeight) {
//                        $rewardsShipmentArray[] = ['weight' => $rewardsWeight, 'volume' => $rewardsVolume, 'toCode' => $shippingAddress->postal_cd, 'fromCode' => $fromCode];
//                    }
//                    $rewardsShipVia = RateShop::rateShopCanada($groupCode, $rewardsShipmentArray);
//                }
//                foreach ($weights as $weight) {
//                    $shipmentArray[] = ['weight' => $weight, 'volume' => $volume, 'toCode' => $shippingAddress->postal_cd, 'fromCode' => $fromCode];
//                }
//                $shipVia = RateShop::rateShopCanada($groupCode, $shipmentArray);
//            }
        }

        if (!isset($excludedShipVia)) $excludedShipVia = [];
        foreach ($excludedShipVia as $key => $value) {
            if (isset($value['rate']) && $value['rate'] > 0) {
                $excludedShipVia[$key]['is_default'] = $value['delivery_method'] == $defaultDeliveryMethod;
            } else {
                if (is_null($excludedShipVia[$key]['rate'])) unset($excludedShipVia[$key]);
            }
        }

        if (!isset($rewardsShipVia)) $rewardsShipVia = [];
        foreach ($rewardsShipVia as $key => $value) {
            if (isset($value['rate']) && $value['rate'] > 0) {
                $rewardsShipVia[$key]['is_default'] = $value['delivery_method'] == $defaultDeliveryMethod;
            } else {
                if (is_null($rewardsShipVia[$key]['rate'])) unset($rewardsShipVia[$key]);
            }
        }

        if (!isset($shipVia)) $shipVia = [];
        foreach ($shipVia as $key => $value) {
            if (isset($value['rate']) && $value['rate'] > 0) {
                $shipVia[$key]['is_default'] = $value['delivery_method'] == $defaultDeliveryMethod;
            } else {
                if (is_null($shipVia[$key]['rate'])) unset($shipVia[$key]);
            }
        }

        if (isset($shipVia['U21'])) {
            $shipVia['U21']['rate'] = 0;
        }

//        $baseRates = collect();
//        $excludedBaseRates = collect();
//        $rewardsBaseRates = collect();
//
//        if (isset($shipVia['U06'])) {
//            $baseRates['U06'] = array_get($shipVia, 'U06.rate', 0);
//        }
//
//        if (isset($shipVia['U03'])) {
//            $baseRates['U03'] = array_get($shipVia, 'U03.rate', 0);
//        } else if (isset($shipVia['U39'])) {
//            $baseRates['U39'] = array_get($shipVia, 'U39.rate', 0);
//        }
//
//        if (isset($excludedShipVia['U06'])) {
//            $excludedBaseRates['U06'] = array_get($excludedShipVia, 'U06.rate', 0);
//        }
//
//        if (isset($excludedShipVia['U03'])) {
//            $excludedBaseRates['U03'] = array_get($excludedShipVia, 'U03.rate', 0);
//        } else if (isset($excludedShipVia['U39'])) {
//            $excludedBaseRates['U39'] = array_get($excludedShipVia, 'U39.rate', 0);
//        }
//
//        if (isset($rewardsShipVia['U06'])) {
//            $rewardsBaseRates['U06'] = array_get($rewardsShipVia, 'U06.rate', 0);
//        }
//
//        if (isset($rewardsShipVia['U03'])) {
//            $rewardsBaseRates['U03'] = array_get($rewardsShipVia, 'U03.rate', 0);
//        } else if (isset($rewardsShipVia['U39'])) {
//            $rewardsBaseRates['U39'] = array_get($rewardsShipVia, 'U39.rate', 0);
//        }
//
//        $netOrderTotal = $cartStatus['discountedTotal']; //'discounts.rewards', 0) - array_get($cartStatus, 'discounts.promotion', 0);
//        $netRewardsOrderTotal = 0;
//
//        if (count($rewardsItems)) {
//            if (count($excludedItems)) {
//                foreach ($rewardsItems as $rewardsItem) {
//                    $netRewardsOrderTotal += $rewardsItem['extendedPrice'];
//                }
//                $rewardsCarrierCodes = rewards()->carrierCode($netRewardsOrderTotal, $rewardsBaseRates, $cartStatus['rewardsTier']);
//            } else {
//                $rewardsCarrierCodes = rewards()->carrierCode($netOrderTotal, $baseRates, $cartStatus['rewardsTier']);
//            }
//
//            if ($rewardsCarrierCodes) {
//                foreach ($rewardsCarrierCodes as $rewardsCarrierCode) {
//                    if (count($excludedItems)) {
//                        $baseRate = array_get($rewardsBaseRates, $rewardsCarrierCode['base_code'], 0) +
//                            array_get($excludedBaseRates, $rewardsCarrierCode['base_code'], 0);
//                        if ($baseRate > 0) {
//                            $shipVia[$rewardsCarrierCode['code']] = [
//                                'id' => $rewardsCarrierCode['code'],
//                                'api_carrier' => null,
//                                'residential' => null,
//                                'hilco_carrier_code' => $rewardsCarrierCode['code'],
//                                'carrier_desc' => $rewardsCarrierCode['description'],
//                                'rate' => $baseRate,
//                                'rate_string' => number_format($rewardsCarrierCode['rate'], 2),
//                                'text' => $rewardsCarrierCode['description'] . ' - ' . $this->formatPrice($rewardsCarrierCode['rate']),
//                                'service' => null,
//                                'delivery_date' => null,
//                                'delivery_time' => null,
//                                'guaranteed' => null,
//                                'base_rate' => $baseRate,
//                                'base_rate_string' => number_format($baseRate, 2),
//                                'discount_amount' => $rewardsCarrierCode['rate'],
//                                'is_default' => $rewardsCarrierCode['code'] == $defaultCarrierCode,
//                            ];
//                        }
//                    } else {
//                        $baseRate = array_get($baseRates, $rewardsCarrierCode['base_code'], 0);
//                        if ($baseRate > 0) {
//                            $shipVia[$rewardsCarrierCode['code']] = [
//                                'id' => $rewardsCarrierCode['code'],
//                                'api_carrier' => null,
//                                'residential' => null,
//                                'hilco_carrier_code' => $rewardsCarrierCode['code'],
//                                'carrier_desc' => $rewardsCarrierCode['description'],
//                                'rate' => $baseRate,
//                                'rate_string' => number_format($rewardsCarrierCode['rate'], 2),
//                                'text' => $rewardsCarrierCode['description'] . ' - ' . $this->formatPrice($rewardsCarrierCode['rate']),
//                                'service' => null,
//                                'delivery_date' => null,
//                                'delivery_time' => null,
//                                'guaranteed' => null,
//                                'base_rate' => $baseRate,
//                                'base_rate_string' => number_format($baseRate, 2),
//                                'discount_amount' => $rewardsCarrierCode['rate'],
//                                'is_default' => $rewardsCarrierCode['code'] == $defaultCarrierCode,
//                            ];
//                        }
//                    }
//                }
//            }
//        }

//        $customer = b2b()->activeCustomer();
//        $customerCategory = $customer->overrideCustomerCategory;
//        $category = $customerCategory->cust_category_desc;
//        if ($category === 'OPTICIAN' || $category === 'OPTOMETRIST' || $category === 'OPHTHALMOLOGIST' || $category === 'LEASED DOCTOR') {
//            if (count($excludedItems) && count($rewardsItems)) {
//                if ($netRewardsOrderTotal >= 150) {
//                    if (isset($rewardsBaseRates['1000']) && isset($excludedBaseRates['1000'])) {
//                        $baseRate = array_get($rewardsBaseRates, '1000');
//                        $rate = round($baseRate / 4, 2);
//                        $baseRate += array_get($excludedBaseRates, '1000');
//                        $shipVia['1409'] = [
//                            'id' => '1409',
//                            'api_carrier' => null,
//                            'residential' => null,
//                            'hilco_carrier_code' => '1409',
////                            'carrier_desc' => 'Promotional 25% off Ground Shipping',
//                            'carrier_desc' => 'Promotional Ground Shipping',
//                            'rate' => $baseRate,
//                            'rate_string' => number_format($rate, 2),
////                            'text' => 'Promotional 25% off Ground Shipping' . ' - ' . $this->formatPrice($rate),
//                            'text' => 'Promotional Ground Shipping' . ' - ' . $this->formatPrice($rate),
//                            'service' => null,
//                            'delivery_date' => null,
//                            'delivery_time' => null,
//                            'guaranteed' => null,
//                            'base_rate' => $baseRate,
//                            'base_rate_string' => number_format($baseRate, 2),
//                            'discount_amount' => $rate,
//                            'is_default' => 1409 == $defaultCarrierCode,
//                        ];
//                    }
//                }
//            } else if(count($rewardsItems)){
//                if ($netOrderTotal >= 150) {
//                    // If baserates doens't exist, don't offer 1409
//
//                    if (isset($baseRates['1000'])) {
//                        $baseRate = array_get($baseRates, '1000');
//                        $rate = round($baseRate / 4, 2);
//                        $shipVia['1409'] = [
//                            'id' => '1409',
//                            'api_carrier' => null,
//                            'residential' => null,
//                            'hilco_carrier_code' => '1409',
////                            'carrier_desc' => 'Promotional 25% off Ground Shipping',
//                            'carrier_desc' => 'Promotional Ground Shipping',
//                            'rate' => $baseRate,
//                            'rate_string' => number_format($rate, 2),
////                            'text' => 'Promotional 25% off Ground Shipping' . ' - ' . $this->formatPrice($rate),
//                            'text' => 'Promotional Ground Shipping' . ' - ' . $this->formatPrice($rate),
//                            'service' => null,
//                            'delivery_date' => null,
//                            'delivery_time' => null,
//                            'guaranteed' => null,
//                            'base_rate' => $baseRate,
//                            'base_rate_string' => number_format($baseRate, 2),
//                            'discount_amount' => $rate,
//                            'is_default' => 1409 == $defaultCarrierCode,
//                        ];
//
//                    }
//
//                }
//            }
//        }

        if ($this->activeCountry() == 'US') { // hacky wrapper if-statement to force promotion to only apply to US customers (per Bob Woyton's email at 3:14pm on 12/0117)
            $legacyCode = LegacyCarrierCodeMapping::where('carrier_code', '=', '1000')->first();
            $baseRate = array_get($shipVia, $legacyCode['delivery_method'], 0);
            // check if order qualifies for shipping discount promotions and add them to shipVia list
            $totalShippingDiscounts = 0;
            $triggeredPromotions = array_get($cartStatus, 'triggeredPromotions');
            foreach ($triggeredPromotions as $triggeredPromotion) {
                $shippingDiscountActions = $triggeredPromotion->shippingDiscountActions;
                foreach ($shippingDiscountActions as $shippingDiscountAction) {
                    if ($totalShippingDiscounts < $baseRate['rate']) {
                        if ($shippingDiscountAction->is_percent) {
                            $totalShippingDiscounts += round(($baseRate['rate'] * $shippingDiscountAction->amount / 100), 2);
                        } else {
                            $totalShippingDiscounts += $shippingDiscountAction->amount;
                        }
                    }
                }
            }
            if ($totalShippingDiscounts > 0) {
                if ($totalShippingDiscounts > $baseRate['rate']) {
                    $totalShippingDiscounts = $baseRate['rate'];
                }
                $promoLegacyCode = LegacyCarrierCodeMapping::where('carrier_code', '=', '5001')->first();


                $shipVia[$promoLegacyCode['delivery_method']] = [
                    'carrier_name' => $baseRate['carrier_name'],
                    'api_code' => null,
                    'carrier_desc' => 'Promotional Ground Shipping',
                    'rate' => $baseRate['rate'],
                    'discounted_rate' => $totalShippingDiscounts,
                    'delivery_datetime' => null,
                    'delivery_method' => $promoLegacyCode['delivery_method'],
                    'is_default' => $promoLegacyCode['delivery_method'] == $defaultDeliveryMethod,
                ];
            }
        }

        if (count($shipVia) == 0) {
            $shipVia[0] = ['id' => -1, 'text' => trans('hilco::app.unable_to_calculate_shipping_rates')];
        }

        return array_values($shipVia);
    }

//  ******************************** RATE SHOP ********************************


    public function hasSentFeedback()
    {
        $user = auth()->user();
        $has = ($user) ? auth()->user()->webFeedbackResponses()->count() > 0 : false;
        return $has;
    }

    public function getShipmentRates(CustomerShippingAddress $customerShippingAddress)
    {
        $cartComposer = App::make('cartSingleton');
        $cartStatus = $cartComposer->getCartStatus();
        $rateType = array_get($this->activeWebSilo, 'shipment_rate_type', 'api');
        $rates = new Collection;

        if ($rateType == 'api') {
            $carriers = array_get($this->activeWebSilo, 'carriers', []);
            $package = new Package(array_get($cartStatus, 'totalWeight', 0), array_get($cartStatus, 'totalVolume', 0), $customerShippingAddress->postal_cd, $customerShippingAddress->country);
            foreach ($carriers as $carrier) {

            }
        } else {
            $rates = array_get($this->activeWebSilo, 'shipmentFlatRates', false);
            krsort($rates);
            $cartValue = 0;
            if ($rateType == 'flatByWeight') $cartValue = array_get($cartStatus, 'totalWeight', 0);
            if ($rateType == 'flatByPrice') $cartValue = array_get($cartStatus, 'discountedTotal', 0);
            foreach ($rates as $threshold => $rate) {
                if ($cartValue >= $threshold) {
                    $rates->push([
                        'rate' => $rate,
                    ]);
                    break;
                }
            }
        }

        return $rates;
    }

    public function getAPIShipmentRates($cartStatus, $customerShippingAddress)
    {

    }

    public function allowBannerCarousel($lpModel)
    {
        if ($this->activeWebSilo->id == '2') {
            if ($lpModel instanceof WebGroup || $lpModel instanceof WebCategory || $lpModel instanceof WebCollection) {
                return false;
            }
        }
        return true;
    }

    /**
     * Retrieves product models from an array of productType, productId pairs
     *
     * @param $products
     */
    public function retrieveProductModels($products)
    {
        $idsByType = [];
        foreach ($products as $product) {
            $type = array_get($product, 'productType');
            $id = array_get($product, 'productId');
            array_set($idsByType, "$type.$id", $id);
        }

        $map = Relation::morphMap();
        $models = collect();
        foreach ($idsByType as $type => $ids) {
            $type = normalizeProductType($type);
            $className = array_get($map, $type, false);
            if ($className === false) continue;
            $classModels = $className
                ::hasVisibleChildren()
                ->hasVisibleParents()
                ->isVisible()
                ->findMany($ids);
            foreach ($classModels as $model) {
                $models->push($model);
            }
        }

        return $models;
    }

    public function defaultRole()
    {
        $defaultRoleSlug = config('hilco.defaultWebRoleSlug');
        $webRole = WebRole::where('slug', '=', $defaultRoleSlug)->first();

        return $webRole;
    }

    public function catalogsWebLink()
    {
        $activeWebSilo = $this->activeWebSilo();
        $activePlantId = $this->activePlant()->id;
        $catalogsWebLinkId = $activeWebSilo->catalogs_weblink_id;
        $linkJoins = $activeWebSilo->webLinkWebSiloJoins->where('weblink_id', $catalogsWebLinkId)->keyBy('plant_id');
        $validLink = array_get($linkJoins, "$activePlantId", array_get($linkJoins, '-1'));
//        if (array_get($validLink, 'is_visible')) {
//
//        }
        return $validLink;

    }

    function promosHaveCartHeaders($cartStatus)
    {
        $triggeredPromotions = $cartStatus['triggeredPromotions'];
        $almostQualifiedPromotionsInfo = $cartStatus['almostQualifiedPromotionsInfo'];
        if (!is_null($triggeredPromotions)) {
            foreach ($triggeredPromotions as $promotion) {
                if (strlen($promotion->cart_header)) {
                    return true;
                }
            }
        }

        if(!is_null($almostQualifiedPromotionsInfo)){
            foreach($almostQualifiedPromotionsInfo as $key => $almostQualifiedPromoInfo){
                if(isset($almostQualifiedPromoInfo['promotion'])){
                    $promotion = $almostQualifiedPromoInfo['promotion'];
                    if(strlen($promotion->almost_header) > 0){
                        return true;
                    }
                }
            }
        }
        return false;

    }

    function promosHaveCheckoutHeaders($cartStatus)
    {
        $triggeredPromotions = $cartStatus['triggeredPromotions'];
        $almostQualifiedPromotionsInfo = $cartStatus['almostQualifiedPromotionsInfo'];
        if (!is_null($triggeredPromotions)) {
            foreach ($triggeredPromotions as $promotion) {
                if (strlen($promotion->checkout_header)) {
                    return true;
                }
            }
        }

        if(!is_null($almostQualifiedPromotionsInfo)){
            foreach($almostQualifiedPromotionsInfo as $key => $almostQualifiedPromoInfo){
                if(isset($almostQualifiedPromoInfo['promotion'])){
                    $promotion = $almostQualifiedPromoInfo['promotion'];
                    if(strlen($promotion->almost_header) > 0){
                        return true;
                    }
                }
            }
        }
        return false;

    }

    public function shippingErrorMessage()
    {
        $activeSegment = $this->activeSegment();
        if (isset($activeSegment->plant) && $activeSegment->plant->plant == 'U51') {
            return trans('messages.please_refer_to_freight_details');
        } else {
            return trans('messages.shippingError');
        }
    }

    public function defaultLocale()
    {
        return 'us_EN';
    }

    public function defaultLanguage()
    {
        return 'en';
    }

    public function isProductRestrictedByAddress ($partNumber) {
        $bannedPartStateCount =
            BannedPartsState
                ::where('part_no', $partNumber)
                ->where('state', $this->activeShippingAddress->state)
                ->where('country', $this->activeShippingAddress->country)
                ->count()
        ;
        return $bannedPartStateCount > 0;
    }

    public function isProductRestrictedByCustomer($partNumber) {
        $bannedPartCustomerCount =
            BannedPartsCustomer
                ::where('customer_id', $this->activeCustomer->id)
                ->where('part_no', $partNumber)
                ->count()
        ;
        return $bannedPartCustomerCount > 0;
    }

    public function doesCartHaveBannedProductRestrictions() {
        $bannedPartsByCustomer =
            BannedPartsCustomer
                ::where('customer_id', $this->activeCustomer->id)
                ->whereIn('part_no', function ($query) {
                    $query
                        ->select('part_number')
                        ->from('WebCarts')
                        ->where('webuser_id', auth()->user()->id)
                        ->where('deleted_at', '=', '0000-00-00 00:00:00')
                    ;
                })
                ->get()
        ;

        $bannedPartsByState =
            BannedPartsState
                ::where('state', $this->activeShippingAddress->state)
                ->where('country', $this->activeShippingAddress->country)
                ->whereIn('part_no', function ($query) {
                    $query
                        ->select('part_number')
                        ->from('WebCarts')
                        ->where('webuser_id', auth()->user()->id)
                        ->where('deleted_at', '=', '0000-00-00 00:00:00')
                    ;

                })
                ->get()
        ;

        return
            (isset($bannedPartsByCustomer) && count($bannedPartsByState) > 0) ||
            (isset($bannedPartsByCustomer) && count($bannedPartsByCustomer) > 0)
        ;
    }

    public function cartHasColdShipParts() {
        $coldShipPartsCount =
            WebCart
                ::where('webuser_id', auth()->user()->id)
                ->whereExists(function($query) {
                    return
                        $query
                            ->select(DB::raw('1'))
                            ->from('Parts')
                            ->where('refrig_flag', '=', '1')
                            ->whereRaw('part_no = part_number')
                    ;
                })
                ->count()
        ;
        return $coldShipPartsCount > 0;
    }

//  ******************************** PASSWORDS ********************************
    public function getPasswordValidationMinLength()
    {
        return 8;
    }

    public function getPasswordValidationRegex()
    {
        return '/^(?=.*[a-zA-Z])(?=.*\d).+$/';
    }

    public function getPasswordValidationString()
    {
        $passwordValidationMinLength = $this->getPasswordValidationMinLength();
        $passwordValidationRegex = $this->getPasswordValidationRegex();
        $passwordValidationString = "min:$passwordValidationMinLength|regex:$passwordValidationRegex";
        return $passwordValidationString;
    }

    public function generateRandomPasswordString($length = null, $passwordValidationRegex = null)
    {
        if (!isset($length)) {
            $length = $this->getPasswordValidationMinLength();
        }

        if (!isset($passwordValidationRegex)) {
            $passwordValidationRegex = $this->getPasswordValidationRegex();
        }

        $allowedCharacters = '346789abcdefghkmnpqrtuvwxyzABCDEFGHJKMNPRTUVWXY';
        $allowedCharactersLength = strlen($allowedCharacters);
        $randomPasswordString = '';
        for ($i = 0; $i < $length; $i++) {
            $randomPasswordString .= $allowedCharacters[rand(0, $allowedCharactersLength - 1)];
        }

        if (!preg_match($passwordValidationRegex, $randomPasswordString)) {
            $randomPasswordString = $this->generateRandomPasswordString($length, $passwordValidationRegex);
        }

        return $randomPasswordString;
    }
//  ******************************** PASSWORDS ********************************

//  ***************************** TRANSLATIONS ***************************
    public function availableLanguages() {
        $languages = AvailableLanguage::where('deleted_at', '=', '0000-00-00 00:00:00')->get();
        $languageList = [];
        foreach ($languages as $language){
            $languageList[$language->language_code] = $language->language_name;
        }

        if (config('app.env') !== 'production') {
            $languageList['foo'] = 'foo';
        }
        return $languageList;
    }

    public function getB2BTranslation($bladePath, $translationKey, $placeHolder) {
        // FOO Test Handler
        if (b2b()->activeLanguage() == 'foo') {
            return 'FOO';
        }
        // FOO Test Handler

        return $this->getApplicationTranslation('b2b', $bladePath, $translationKey, $placeHolder);
    }

    /**
     * Gets the ApplicationTranslation record identified by the
     * $application, $bladePage, $translationKey, and current $activeLanguage
     * If a record isn't found for the current $activeLanguage, return the
     * translation text record for English (en)
     * If somehow there isn't an English translation for the intended text,
     * return the value in $placeHolder.
     * @param $application
     * @param $component
     * @param $translationKey
     * @param $placeHolder
     * @return mixed
     */
    public function getApplicationTranslation($application, $component, $translationKey, $placeHolder) {
        $translation = $placeHolder;

        $applicationTranslationKey =
            ApplicationTranslationKey::byApplicationComponentKey($application, $component, $translationKey)->first();

        if ($applicationTranslationKey) {
            $applicationTranslation =
                ApplicationTranslation
                    ::byTranslationKeyLanguage($applicationTranslationKey->id, b2b()->activeLanguage())->first();
            if ($applicationTranslation) {
                $translation = $applicationTranslation->translation;
            } else {
                $defaultTranslation =
                    ApplicationTranslation
                        ::byTranslationKeyLanguage($applicationTranslationKey->id, 'en')->first();
                if ($defaultTranslation) {
                    $translation = $defaultTranslation->translation;
                }
            }
        }

        return $translation;
    }

    public function getB2BDataTablesTranslationParameters($bladePath) {
        return $this->getDataTablesTranslationParameters('b2b', $bladePath);
    }

    public function getDataTablesTranslationParameters($application, $component) {
        // FOO Test Handler
        if (b2b()->activeLanguage() == 'foo') {
            return config('datatables.language_foo');
        }
        // FOO Test Handler

        if (!ApplicationTranslationKey::byApplicationComponent($application, $component)->get()->count()) {
            $component = 'datatables.default';
        }

        $translationsArray = config('datatables.language_placeholders');
        foreach ($translationsArray as $languageKey => $languageValue) {
            if (is_array($languageValue)) {
                foreach($languageValue as $subKey => $subValue) {
                    $translationsArray[$languageKey][$subKey] =
                        $this->getApplicationTranslation($application, $component,
                            $languageKey.'.'.$subKey, $translationsArray[$languageKey][$subKey]);
                }
            } else {
                $translationsArray[$languageKey] =
                    $this->getApplicationTranslation($application, $component,
                        $languageKey, $translationsArray[$languageKey]);
            }
        }

        return $translationsArray;
    }
//  **********************************************************************

    public function getLanguages()
    {
        $languages = AvailableLanguage::where('deleted_at', '=', '0000-00-00 00:00:00')->get();
        $languageList = [];
        foreach ($languages as $language){
            $languageList[$language->language_code] = $language->language_name;
        }
        return $languageList;
    }

    public function getApplications(){
        $apps = ApplicationTranslationKey::distinct()->pluck('application')->toArray();
        $appList = [];
        foreach($apps as $app){
            $appList[$app] = strtoupper($app);
        }
        return $appList;
    }

    public function getRefrigAssetId() {
        $webAsset = WebAsset::where('asset_key', '=', config('hilco.hilcoRefrigWarningImageKey'))->first();
        if(!empty($webAsset)){
            return $webAsset->id;
        }
    }
}
