<?php

namespace Hilco\Models;

use DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Venturecraft\Revisionable\RevisionableTrait;

class WebGroup extends WebModel implements HasLandingPageInterface, HasWebSiloOverridesInterface {
	protected $table = "WebGroups";
	protected $fillable = ['name', 'slug', 'webSiloIds' ,'is_visible', 'detail'];
	protected $casts = [
	    'is_visible' => 'boolean',
        'layout' => 'array',
	];

    protected $hierarchyParent = false;

	public function webSilos() {
		return $this->belongsToMany(WebSilo::class, 'WebGroup_WebSilo', 'webgroup_id', 'websilo_id')->withPivot('display_order');
	}

	public function webCategories() {
		return $this->belongsToMany(WebCategory::class, 'WebCategory_WebGroup', 'webgroup_id', 'webcategory_id')->withPivot('display_order');
	}

    public function visibleWebCategories() {
        return $this->belongsToMany(WebCategory::class, 'WebCategory_WebGroup', 'webgroup_id', 'webcategory_id')
            ->withPivot('display_order', 'is_featured')
            ->webVisible()
            ->displayOrder()
        ;
    }

    public function scopeDisplayOrder(Builder $query) {
        return $query->orderBy('pivot_display_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeAlphabetical(Builder $query) {
        return $query->orderBy('name', 'asc');
    }

	public function webCategoriesOrdered() {
        return $this->webCategories()->orderBy('pivot_display_order')->orderBy('name')
            ;
    }

	public function webLandingPages() {
		return $this->belongsToMany(WebLandingPage::class, 'WebGroup_WebLandingPage', 'webgroup_id', 'weblandingpage_id');
	}

	public function getFeaturedChildrenAttribute() {
        return $this->visibleWebCategories()->wherePivot('is_featured', 1)->get();
    }

    public function getChildrenAttribute() {
        $children = $this->visibleWebCategories();
        if (request()->get('ap', false) == 1) $children->webSiloApproved();

        return $children->get();
    }

    public function setWebSiloIdsAttribute($ids) {
		$this->save();
        $this->createRevisionRecord('webSiloIds', $this->webSilos->pluck('id')->toJson(), json_encode($ids));
        $this->webSilos()->sync($ids);
	}

	public function scopeVisible(Builder $query) {
		$query
			->where('WebGroups.is_visible', '1')
            ->whereHas('webSilos', function ($query) {
                return $query->visible();
            })
		;
//
//        if (b2b()->activeCountry() != 'US') {
//            $query->where('slug', '!=', 'pharmaceuticals');
//        }

        return $query;
	}


    public function scopeWebVisible(Builder $query) {
        return $query
            ->where('WebGroups.is_visible', '1')
            ->where(function ($where) {
                $where
                    ->whereDoesntHave('webSiloOverride')
                    ->orWhereHas('webSiloOverride', function ($query) {
                        return $query->where('WebSiloOverrides.is_visible', '=', 1);
                    })
                ;
            })
            ->whereHas('webCategories', function ($webCategoriesQuery) {
                return $webCategoriesQuery->webVisible();
            })
            ->with('webSilos')
        ;
    }

    public function scopeWebSiloApproved(Builder $query) {
        return $query
            ->whereHas('webCategories', function ($webCategoriesQuery) {
                return $webCategoriesQuery->webSiloApproved();
            })
            ;
    }

	public function scopeInSilo($query, $webSiloId) {
        return $query->whereHas('webSilos', function ($query) use ($webSiloId) {
            $query->where('websilo_id', $webSiloId);
        });
    }

//	public function scopeForMenu(Builder $query) {
//        $query->visible()
//            ->join('WebCategory_WebGroup as WCWG', 'WCWG.webgroup_id', '=', 'WebGroups.id')
//            ->join('WebCollection_WebCategory as WLWC', 'WLWC.webcategory_id', '=', 'WCWG.webcategory_id')
//            ->join('WebCategories', 'WebCategories.id', '=', 'WCWG.webcategory_id')
//            ->join('WebCollections', 'WebCollections.id', '=', 'WLWC.webcollection_id')
//            ->orderBy('WCWG.display_order', 'asc')
//            ->orderBy('WebCategories.name', 'asc')
//            ->orderBy('WLWC.display_order', 'asc')
//            ->orderBy('WebCollections.name', 'asc')
//            ->select(DB::raw('WebGroups.*'))
//            ->groupBy('WebGroups.id')
//        ;
//        return $query;
//    }

    public function getHierarchyPathsAttribute() {
        $paths = [];
        foreach ($this->webSilos as $webSilo) {
            $paths[] = [
                'webSilo' => $webSilo,
            ];
        }
        return $paths;
    }

    public function slugUrl($approvedOnly = false) {
	    $params = [$this->slug];
	    if ((request()->get('ap', false) == 1) || $approvedOnly) $params['ap'] = 1;
        return route('group.slug', $params);
    }

    public function getParentRelationName() {
        return false;
    }

    public function getPrimaryImage() {
	    $asset = array_get($this, 'webSiloOverride.primaryImage', null);
        if (!is_null($asset)) return $asset;

        $asset = $this->assetsByType('primary')->first();
        if (!is_null($asset)) return $asset;

        $webCategory = $this->visibleWebCategories->first();
        if ($webCategory) return $webCategory->getPrimaryImage();
    }

    use HasAssets, HasSlug, HasLandingPage, RevisionableTrait, ManyToManyRevisionableTrait, HasWebSiloOverrides;
}
