<?php

namespace Visionware\DataManager\Console\Commands;

use Config;
use DB;
use Illuminate\Console\Command;
use InvalidArgumentException;
use Storage;

class ReverseEngineer extends DataManagerCommand {
    protected $signature = 'datamanager:reverse
            {connection : The name of the database connection for which to generate a schema }
            {output_file : The name of the file to create }
            {--description : Describe the schema}
    ';

    protected $description = 'Generates a schema.json from a MySQL Database';

    private $db;

    public function datamanager_handle() {
        $this->db = DB::connection($this->argument('connection'));
        $db_name = $this->db->getDatabaseName();
        $tables = $this->db->select('SHOW TABLES');
        $output_tables = [];
        foreach ($tables as $table_object) {
            $member = "Tables_in_$db_name";
            $table = $table_object->$member;
//            if (!($table == 'Customers' || $table == 'CustomerCategories')) continue;
            $output_tables[$table] = $this->parse_table($table);
        }

        $output_tables = $this->parse_old_definition($output_tables);

        $output = [
            'description' => $this->option('description') ? $this->option('description') : '',
            'created'     => date('r'),
            'tables'      => $output_tables,
        ];

        file_put_contents($this->argument('output_file'), json_encode($output, JSON_PRETTY_PRINT | JSON_NUMERIC_CHECK));
    }

    private function parse_old_definition($output_tables) {
        $indef = json_decode(Storage::disk('local')->get('old/hilco-import-def.json'), true);
        foreach ($indef['tables'] as $xx => $info) {

            $table = $info['table'];
            if (!isset($output_tables[$table])) continue;
//            dd($info);
            $output_tables[$table]['import_file'] = $info['importFile'];
//            $map = array_flip($info['fieldMapping']);
            foreach ($info['fieldMapping'] as $field => $column) {
//                if (!isset(array_pluck($output_tables[$table]['columns'], 'name', 'name')[$column])) continue;
                if (!isset($output_tables[$table]['columns'][$column])) {
                    $this->error("Expecting $column in $table, but there is none defined!");
                    continue;
                }
                $output_tables[$table]['columns'][$column]['import_field'] = $field;
            }
            if (isset($info['transformations'])) {
                foreach ($info['transformations'] as $column => $transformation) {
                    if (!isset($output_tables[$table]['columns'][$column])) continue;
                    $output_tables[$table]['columns'][$column]['import_transformation'] = $transformation;
                }
            }
            if (isset($info['foreign'])) {
                foreach ($info['foreign'] as $foreign) {
                    $index_column = $foreign['localID'];
                    $output_tables[$table]['indices'][$index_column]['foreign_table'] = $foreign['table'];
                    $output_tables[$table]['indices'][$index_column]['columns'][$index_column]['foreign_column'] = $foreign['foreignID'];
                    $output_tables[$table]['indices'][$index_column]['foreign_table'] = $foreign['table'];
                    $f_keys = array_combine($foreign['foreignKey'], $foreign['localKey']);
                    $foreign_sequence = 1;
                    foreach ($f_keys as $foreign_key => $local_key) {
                        $output_tables[$table]['indices'][$index_column]['import_join_columns'][] = [
                            'local'   => $local_key,
                            'foreign' => $foreign_key,
                            'sequence' => $foreign_sequence++,
                        ];
                    }
                }
            }
        }
        return $output_tables;
    }

    private function parse_table($table) {
        $columns = $this->parse_columns($table);
        $indices = $this->parse_indices($table);

        return [
            'name'        => $table,
            'columns'     => $columns,
            'indices'     => $indices,
//            'import_file' => null,
        ];
    }

    private function parse_columns($table) {
        $columns = [];
        foreach ($this->db->select("SHOW COLUMNS IN $table") as $column_object) {

            $columns[$column_object->Field] = [
                'name'                  => $column_object->Field,
                'type'                  => $column_object->Type,
                'null'                  => $column_object->Null == "YES",
                'default'               => $column_object->Default,
                'extra'                 => $column_object->Extra,
//                'import_field'          => null,
//                'import_transformation' => null,
            ];
//            if ($table == 'CustomerEmails') print_r($column_object);
        }

        return $columns;
    }

    private function parse_indices($table) {
        $indices = [];
        foreach ($this->db->select("SHOW INDEX IN $table") as $index_object) {
            if ($index_object->Collation != 'A') throw new \UnexpectedValueException(
                "Collation not supported by schema creator!"
            );
            if ($index_object->Index_type != 'BTREE') throw new \UnexpectedValueException(
                "Index type " . $index_object->Index_type . " not supported by schema creator!"
            );
            $indices[$index_object->Key_name]['name'] = $index_object->Key_name;
            $indices[$index_object->Key_name]['unique'] = $index_object->Non_unique != 1;
            $col = [
                'name'     => $index_object->Column_name,
                'sequence' => $index_object->Seq_in_index,
            ];
            $indices[$index_object->Key_name]['columns'][$index_object->Column_name] = $col;
        }

        return $indices;
    }
}
