<?php

namespace Visionware\DataManager\Console\Commands;

use DB;
use Storage;
use Symfony\Component\Console\Output\OutputInterface;

class Ingest extends DataManagerCommand {
    protected $signature = 'datamanager:ingest {definition? : The name of the definition in the config file to use for ingestion} {--force : Force update}';

    protected $description = 'Ingests data files and stuff';

    protected function datamanager_handle() {
        foreach ($this->definition['import_order'] as $importName) {
            $this->importFile($importName);
        }
    }

    private function importFile($importName) {
        $current_definition = $this->definition['tables'][$importName];
        $table_name = $current_definition['name'];
        $db = DB::connection($this->config['history-database']);
        $import_file_name = $current_definition['import_file'];

        $local_path = sys_get_temp_dir() . "/$import_file_name";
        $remote_path = $this->config['import_source.prepend_path'] . $import_file_name;
        $disk = Storage::disk($this->config['import_source.disk']);
        if (!$disk->exists($remote_path)) {
            $this->error("File $remote_path not found, skipping...");
            return false;
        }

        $this->info("Checking last modified date for $import_file_name...");
        $last_modified_ut = $disk->lastModified($remote_path);
        $last_modified = date('Y-m-d H:i:s', $last_modified_ut);

        try {
            $db_last_modified = $db->selectOne("SELECT MAX(last_modified) lm FROM {$table_name}_files")->lm;
        }  catch (\PDOException $e) {
            $this->error($e->getMessage());
            $this->error('Error while importing file! Skipping...');

            return false;
        }

        if ($last_modified == $db_last_modified && $this->option('force') == null) {
            $this->info("Table is up-to-date, skipping");
            return;
        }

        $this->info("Downloading file $import_file_name...");
        if (false === @file_put_contents($local_path, $disk->get($remote_path))) {
            $this->error("Error while downloading $remote_path, skipping...");
            return;
        }

        $load_file_path = $local_path . '.load';

        $local_file_object = new \SplFileObject($local_path, 'r');
        if ($local_file_object === false) $this->error("Unable to open $local_path");
        $local_file_object->setCsvControl("\t");
        $local_file_object->setFlags(
            \SplFileObject::READ_CSV
            | \SplFileObject::READ_AHEAD
            | \SplFileObject::SKIP_EMPTY
            | \SplFileObject::DROP_NEW_LINE
        );
        $local_file_object->seek(PHP_INT_MAX);
        $line_count = $local_file_object->key();
        $this->info("Opening file $local_path ($line_count lines)...");
        $local_file_object->rewind();

        $source_fields = $local_file_object->current();
        $local_file_object->next();

        $destination_columns = [];
        $field_column_map = [];
        foreach ($current_definition['columns'] as $column) {
            if (!isset($column['import_field'])) continue;
            $destination_columns[] = $column['name'];
            $field_column_map[$column['import_field']] = $column['name'];
        }

        if (count($source_fields) != count($destination_columns)) {
            $this->error("Import file field count does not match definition!\nSource Fields: " . print_r($source_fields, true) . "\nDestination Columns: " . print_r($destination_columns, true));
            return false; //return on error
        }

        $values = [];
        $this->info("Building LOAD DATA file...");
        $load_file_object = new \SplFileObject($load_file_path, 'w');
        $current_line = 0;
        while (!$local_file_object->eof()) {
            $raw = $local_file_object->current();
            if ($values === false) return; //return on error
            $input_row = @array_combine($source_fields, $raw);
            if ($input_row !== false) {
                foreach ($field_column_map as $field => $column) {
                    $output_row[$column] = str_replace(',', '\\,', trim($input_row[$field]));
                }
                $output_row['last_modified'] = $last_modified;
                $output_row['sequence'] = $current_line;
                $output_row['record_hash'] = md5(implode('', $output_row));

                $load_file_object->fwrite(implode(',', $output_row) . "\n");
            } else {
                $this->error("Line $current_line has invalid field count, skipping line.");
            }

            $local_file_object->next();
            $current_line++;
        }
        $load_file_object->fflush();
        $columns_string = implode(', ', array_keys($output_row));
        $sql = "LOAD DATA LOCAL INFILE '$load_file_path' IGNORE INTO TABLE `{$table_name}_staging` FIELDS TERMINATED BY ',' ($columns_string)";
        try {
            $this->info("Loading data into staging table...");
            $this->info("$sql", OutputInterface::VERBOSITY_VERY_VERBOSE);
            $db->table("{$table_name}_staging")->truncate();
            $db->getPdo()->exec($sql);
            $db->beginTransaction();
            $columns_string = 'record_hash, ' . implode(', ', $destination_columns);
            $this->info("Updating records table from staging...");
            $db->statement("INSERT IGNORE INTO `{$table_name}_records` ($columns_string) SELECT $columns_string FROM `{$table_name}_staging`");
            $this->info("Updating files table from staging...");
            $db->statement("INSERT IGNORE INTO `{$table_name}_files` (last_modified, record_id, sequence) SELECT s.last_modified, r.id, s.sequence FROM `{$table_name}_staging` s JOIN `{$table_name}_records` r USING (record_hash)");
            $db->commit();
        } catch (\PDOException $e) {
            $this->error($e->getMessage());
            $this->error('Error while importing file! Skipping...');

            return false;
        }
        $this->info("Finished with file $local_path\n");
    }
}
