<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 1/27/16
 * Time: 4:40 PM
 */

namespace Visionware\DataManager;

use DB;
use Monolog\Logger;
use Visionware\DataManager\Grammars\GrammarInterface;
use Visionware\DataManager\Schema\Schema;
use Visionware\DataManager\Schema\SchemaInfoInterface;

class SchemaManager {
    private $ops;

    /**
     * @var Schema
     */
    protected $new_schema;
    /** @var Schema  */
    protected $current_schema;
    /**
     * @var GrammarInterface
     */
    protected $grammar;
    protected $name;
    protected $sql;

    private $recreate;

    /**
     * @var Logger
     */
    private $logger;

    public function __construct(GrammarInterface $grammar, $name, $logger, Schema $new_schema, SchemaInfoInterface $current_schema = null) {
        $this->grammar = $grammar;
        $this->name = $name;
        $this->new_schema = $new_schema;
        $this->current_schema = $current_schema ? $current_schema : null;
        $this->logger = $logger;
    }

    public function setRecreate() {
        $this->recreate = true;
    }

    public function migrate(&$log = []) {
        $this->ops = [];

        if ($this->recreate) {
            $this->ops[] = ['action' => 'drop_schema', 'schema' => $this->name];
            $this->current_schema = null;
        }

        if (is_null($this->current_schema)) {
            $this->ops[] = ['action' => 'create_schema', 'schema' => $this->name];
            $this->ops[] = ['action' => 'use_schema', 'name' => $this->name];
            $this->ops[] = ['action' => 'create_functions'];
            $this->ops[] = ['action' => 'create_metadata'];
            $current_tables = [];
        } else {
            $current_tables = $this->current_schema->table_names();
        }

        $new_tables = $this->new_schema->table_names();
        $tables_to_add = array_diff($new_tables, $current_tables);
        $tables_to_drop = array_diff($current_tables, $new_tables);
        foreach ($tables_to_add as $table_name) $this->ops[] = ['action' => 'create_table', 'table_name' => $table_name];
        foreach ($tables_to_drop as $table_name) $this->ops[] = ['action' => 'drop_table', 'table_name' => $table_name];
        foreach ($new_tables as $table_name) {
            if (in_array($table_name, $tables_to_drop) || in_array($table_name, $tables_to_add)) continue;

            $create_index_ops = [];
            $drop_index_ops = [];
            $create_column_ops = [];
            $drop_column_ops = [];
            $change_ops = [];

            $current_indices = $this->current_schema->index_names($table_name);
            $new_indices = $this->new_schema->index_names($table_name);
            $indices_to_add = array_diff($new_indices, $current_indices);
            $indices_to_drop = array_diff($current_indices, $new_indices);
            foreach ($indices_to_add as $index_name) $create_index_ops[] = ['action' => 'create_index', 'table_name' => $table_name, 'index_name' => $index_name];
            foreach ($indices_to_drop as $index_name) $drop_index_ops[] = ['action' => 'drop_index', 'table_name' => $table_name, 'index_name' => $index_name];
            foreach ($new_indices as $index_name) {
                if (in_array($index_name, $indices_to_add) || in_array($index_name, $indices_to_drop)) continue;
                if ($this->current_schema->index($table_name, $index_name) != $this->new_schema->index($table_name, $index_name)) {
                    $drop_index_ops[] = ['action' => 'drop_index', 'table_name' => $table_name, 'index_name' => $index_name];
                    $create_index_ops[] = ['action' => 'create_index', 'table_name' => $table_name, 'index_name' => $index_name];
                }
            }

            $current_columns = $this->current_schema->column_names($table_name);
            $new_columns = $this->new_schema->column_names($table_name);
            $columns_to_add = array_diff($new_columns, $current_columns);
            $columns_to_drop = array_diff($current_columns, $new_columns);
            foreach ($columns_to_add as $column_name) $create_column_ops[] = ['action' => 'create_column', 'table_name' => $table_name, 'column_name' => $column_name];
            foreach ($columns_to_drop as $column_name) $drop_column_ops[] = ['action' => 'drop_column', 'table_name' => $table_name, 'column_name' => $column_name];
            foreach ($new_columns as $column_name) {
                if (in_array($column_name, $columns_to_add) || in_array($column_name, $columns_to_drop)) continue;
                if ($this->current_schema->column($table_name, $column_name) != $this->new_schema->column($table_name, $column_name)) {
                    $change_ops[] = ['action' => 'change_column', 'table_name' => $table_name, 'column_name' => $column_name];
                }
            }

            foreach ($create_column_ops as $op) $this->ops[] = $op;
            foreach ($create_index_ops as $op) $this->ops[] = $op;
            foreach ($drop_index_ops as $op) $this->ops[] = $op;
            foreach ($drop_column_ops as $op) $this->ops[] = $op;
            foreach ($change_ops as $op) $this->ops[] = $op;
        }
        if (count($this->ops) && !is_null($this->new_schema)) $this->ops = array_prepend($this->ops, ['action' => 'use_schema', 'name' => $this->name]);
        $this->sql = [];
        foreach ($this->ops as $op) {
            $action = $op['action'];
            unset($op['action']);
            $this->logger->info($this->name . ': ' . $action . ' ' . implode(', ', array_values($op)));
            call_user_func_array([$this, $action], $op);

        }
        if (count($this->sql)) $this->update_metadata();
        return $this->sql;
    }

    protected function create_schema($name) {
        $this->sql[] = $this->grammar->create_schema($name);
    }

    protected function drop_schema($name) {
        $this->sql[] = $this->grammar->drop_schema($name);
    }

    protected function create_functions() {
        $this->sql[] = $this->grammar->uuid_to_bin();
        $this->sql[] = $this->grammar->uuid_from_bin();
        $this->sql[] = $this->grammar->alpha_num();
    }

    protected function use_schema($name) {
        $this->sql[] = $this->grammar->use_schema($name);
    }

    private function create_metadata() {
        $table_name = 'datamanager_metadata';

        $columns = [
            $this->grammar->column_definition('metadata_type', 'VARCHAR(255)'),
            $this->grammar->column_definition('metadata_key', 'VARCHAR(255)'),
            $this->grammar->column_definition('int_value', 'INT(11)', true, 'null'),
            $this->grammar->column_definition('text_value', 'VARCHAR(255)', true, 'null'),
            $this->grammar->column_definition('longtext_value', 'LONGTEXT', true, 'null'),
            $this->grammar->column_definition('date_value', 'datetime', true, 'null'),
        ];
        $indices = [
            $this->grammar->index_definition($table_name, 'PRIMARY KEY', ['metadata_type', 'metadata_key'], ''),
        ];
        $this->sql[] = $this->grammar->create_table($table_name, $columns, $indices);
    }

    private function update_metadata() {
        $this->sql[] = $this->grammar->replace(
            'datamanager_metadata',
            ['metadata_type', 'metadata_key', 'longtext_value'],
            [
                '"definition"',
                '"definition"',
                "'" . addslashes(json_encode($this->new_schema->definition())) . "'"
            ]
        );
    }

    private function build_column_string($column, $field = null) {
        return $this->grammar->column_definition($column['name'], $column['type'], $column['nullable'], $column['default'], $column['extra'], array_get($field, 'cst_in_version', 'all'));
    }
    private function build_index_string($table_name, $index) {
        return $this->grammar->index_definition($table_name, $index['type'], $index['columns'], $index['name']);
    }

    protected function create_table($table_name) {
        $columns = $this->new_schema->columns($table_name);
        $indices = $this->new_schema->indices($table_name);
        $column_strings = [];
        foreach ($columns as $column) {
            $field = $this->new_schema->field_from_column($table_name, $column);
            $column_string = $this->build_column_string($column, $field);
            if ($column_string !== null) {
                $column_strings[] = $column_string;
            }
        }
        $index_strings = [];
        foreach ($indices as $index) $index_strings[] = $this->build_index_string($table_name, $index);

        $this->sql[] = $this->grammar->create_table($table_name, $column_strings, $index_strings);
        if ($this->new_schema->post_create_table_sql($table_name)) {
            $this->sql[] = $this->new_schema->post_create_table_sql($table_name);
        }
    }

    protected function drop_table($table_name) {
        $this->sql[] = $this->grammar->drop_table($table_name);

        if ($this->new_schema->post_drop_table_sql($table_name)) {
            $this->sql[] = $this->new_schema->post_drop_table_sql($table_name);
        }
    }

    protected function create_column($table_name, $column_name) {
        $column_string = $this->build_column_string($this->new_schema->column($table_name, $column_name));
        if ($column_string !== null) {
            $this->sql[] = $this->grammar->create_column($table_name, $column_string);
            if ($this->new_schema->post_create_column_sql($table_name, $column_name)) {
                $this->sql[] = $this->new_schema->post_create_column_sql($table_name, $column_name);
            }
        }
    }

    protected function drop_column($table_name, $column_name) {
        $this->sql[] = $this->grammar->drop_column($table_name, $column_name);
        if ($this->new_schema->post_drop_column_sql($table_name, $column_name)) {
            $this->sql[] = $this->new_schema->post_drop_column_sql($table_name, $column_name);
        }
    }
    
    protected function create_index($table_name, $index_name) {
        $this->sql[] = $this->grammar->create_index($table_name, $this->build_index_string($table_name, $this->new_schema->index($table_name, $index_name)));
    }
    
    protected function drop_index($table_name, $index_name) {
        $this->sql[] = $this->grammar->drop_index($table_name, $index_name);
    }

    protected function change_column($table_name, $column_name) {
        $column_string = $this->build_column_string($this->new_schema->column($table_name, $column_name));
        if ($column_string !== null) {
            $this->sql[] = $this->grammar->change_column($table_name, $column_name, $column_string);
            if ($this->new_schema->post_change_column_sql($table_name, $column_name)) {
                $this->sql[] = $this->new_schema->post_change_column_sql($table_name, $column_name);
            }
        }
    }
}