<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 4/18/16
 * Time: 12:25 PM
 */

namespace Visionware\DataManager\Schema;

class HistorySchema extends Schema {
    private $extra_tables = [
        'file_record',
        'files',
        'records',
        'snapshots',
        'staging',
    ];

    protected function filter_table($table_name, $table) {
        return array_key_exists('imported_from', $table);
    }
    
    public function post_create_table_sql($table_name) {
        if (ends_with($table_name, "_staging")) { // this is to make sure only one view is created for the set of extra tables
            $table_name = str_replace("_staging", '', $table_name);

            return <<<SQL
CREATE VIEW `{$table_name}_latest` AS
SELECT r.*, `f`.`last_modified` AS `import_file_last_modified`
FROM {$table_name}_records r
JOIN {$table_name}_file_record j ON (j.record_hash = r.record_hash)
JOIN {$table_name}_files f ON (j.file_id = f.id)
JOIN {$table_name}_snapshots sn ON (f.id = sn.file_id)
WHERE sn.last_modified IN ( SELECT max(last_modified) FROM {$table_name}_snapshots);
SQL;
        }
        return false;
    }

    public function post_drop_table_sql($table_name) {
        if (ends_with($table_name, "_staging")) {
            $table_name = str_replace("_staging", '', $table_name);

            return "DROP VIEW `{$table_name}_latest`;";
        }
        return false;
    }

    public function post_change_column_sql($table_name, $column_name) {
        return $this->post_drop_table_sql($table_name) . " " . $this->post_create_table_sql($table_name);
    }

    public function post_create_column_sql($table_name, $column_name) {
        return $this->post_change_column_sql($table_name, $column_name);
    }

    public function post_drop_column_sql($table_name, $column_name) {
        return $this->post_change_column_sql($table_name, $column_name);
    }

    public function tables() {
        $parent_tables = parent::tables();
        $tables = [];
        foreach ($parent_tables as $parent_table_name => $parent_table) {
            foreach ($this->extra_tables as $extra_table_name) {
                $table_name = $parent_table_name . '_' . $extra_table_name;
                $tables[$table_name] = $this->table($table_name);
            }
        }
        return $tables;
    }

    public function table($table_name) {
        foreach ($this->extra_tables as $extra_table_name) {
            if (ends_with($table_name, "_$extra_table_name")) {
                $parent_table_name = str_replace("_$extra_table_name", '', $table_name);
                return call_user_func_array([$this, $extra_table_name . '_table'], ['parent_table_name' => $parent_table_name, 'table_name' => $table_name]);
            }
        }
        return parent::table($table_name);
    }

    public function file_record_table($parent_table_name, $table_name) {
        $fields = [
            $this->new_column('file_id', 'int(11) unsigned'),
            $this->new_column('record_hash', 'varchar(32)'),
            $this->new_column('sequence', 'int(11) unsigned'),
        ];
        $indices = [
            $this->new_index('PRIMARY KEY', ['file_id', 'record_hash', 'sequence']),
        ];
        return ['table' => 'file_record_' . $table_name, 'fields' => $fields, 'indices' => $indices];
    }

    public function files_table($parent_table_name, $table_name) {
        $fields = [
            $this->new_column('id', 'int(11) unsigned', false, '', 'auto_increment'),
            $this->new_column('file_name', 'varchar(128)'),
            $this->new_column('last_modified', 'datetime'),
        ];
        $indices = [
            $this->new_index('PRIMARY KEY', ['id'], ''),
            $this->new_index('UNIQUE KEY', ['file_name', 'last_modified'], 'unique_key'),
        ];
        return ['table' => 'files_' . $table_name, 'fields' => $fields, 'indices' => $indices];
    }

    public function records_table($parent_table_name, $table_name) {
        $fields = [
            $this->new_column('record_hash', 'varchar(32)'),
        ];
        $indices = [
            $this->new_index('PRIMARY KEY', ['record_hash']),
        ];
        $fields = array_merge($fields, $this->get_parent_fields($parent_table_name));
        $unique_key = parent::index($parent_table_name, 'unique_key');
        $unique_key['type'] = 'KEY';
        $indices[] = $unique_key;
        return ['table' => 'records_' . $table_name, 'fields' => $fields, 'indices' => $indices];
    }

    public function snapshots_table($parent_table_name, $table_name) {
        $fields = [
            $this->new_column('last_modified', 'datetime'),
            $this->new_column('file_id', 'int(11) unsigned'),
        ];
        $indices = [
            $this->new_index('UNIQUE KEY', ['last_modified', 'file_id'], 'unique_key'),
        ];
        return ['table' => 'snapshots_' . $table_name, 'fields' => $fields, 'indices' => $indices];
    }

    public function staging_table($parent_table_name, $table_name) {
        $fields = [
            $this->new_column('last_modified', 'datetime'),
            $this->new_column('record_hash', 'varchar(32)'),
            $this->new_column('sequence', 'int(11) unsigned'),
            $this->new_column('file_name', 'varchar(128)'),
        ];
        $fields = array_merge($fields, $this->get_parent_fields($parent_table_name));
        $indices = [
            $this->new_index('PRIMARY KEY', ['record_hash']),
        ];
        return ['table' => 'staging_' . $table_name, 'fields' => $fields, 'indices' => $indices];
    }

    private function get_parent_fields($parent_table_name) {
        $fields = [];
        foreach (parent::fields($parent_table_name) as $parent_field) {
            if (array_key_exists('foreign', $parent_field)) continue;
            if (array_key_exists('no_import', $parent_field) && $parent_field['no_import']) continue;
            if ($parent_field['type'] != 'text' && strcasecmp(substr($parent_field['type'], 0, 7), 'varchar')) $parent_field['type'] = 'varchar(32)';
            $fields[] = $parent_field;
        }
        return $fields;
    }
}