<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/13/16
 * Time: 10:24 AM
 */

namespace Visionware\DataManager;

use Monolog\Handler\RotatingFileHandler;
use Monolog\Logger;
use Storage;
use Config;
use DB;
use Symfony\Bridge\Monolog\Handler\ConsoleHandler;
use Symfony\Component\Console\Output\OutputInterface;
use Visionware\DataManager\Exceptions\RemoteFileNotFoundException;
use Visionware\DataManager\Schema\Schema;

class DataManagerHelper {

    protected $definition;
    protected $history_db;
    protected $live_db;

    public function __construct() {
        $this->load_definition();
        $this->history_db = DB::connection(Config::get('datamanager.history-database-connection'));
        $this->live_db = DB::connection(Config::get('datamanager.live-database-connection'));
    }

    /**
     * @return Ingester
     */
    public function getIngester() {
        return new Ingester(new Schema($this->definition));
    }

    public function getImporter() {
        return new Importer(new Schema($this->definition));
    }

    public function getSchema($mode) {
        $schemaclassname = 'Visionware\\DataManager\\Schema\\' . ucfirst($mode) . 'Schema';
        $schema = new $schemaclassname($this->definition);
        return $schema;
    }

    public function getLogger($name, OutputInterface $output = null) {
        $logger = new Logger($name);
        $handler = new RotatingFileHandler(
            storage_path("/logs/datamanager/$name.log"),
            7,
            Logger::INFO,
            true
        );
        $logger->pushHandler($handler);

        if ($output) {
            $console_handler = new ConsoleHandler(
                $output, true, [
                    OutputInterface::VERBOSITY_NORMAL       => Logger::NOTICE,
                    OutputInterface::VERBOSITY_VERBOSE      => Logger::INFO,
                    OutputInterface::VERBOSITY_VERY_VERBOSE => Logger::DEBUG,
                    OutputInterface::VERBOSITY_DEBUG        => Logger::DEBUG,
                ]
            );
            $logger->pushHandler($console_handler);
        }

        return $logger;
    }

    public function definition() {
        return $this->definition;
    }
    
    public function live_db() {
        return $this->live_db;
    }

    public function history_db() {
        return $this->history_db;
    }

    public function db_for_mode($mode) {
        if ($mode == 'live') return $this->live_db();
        else if ($mode == 'history') return $this->history_db();
        return null;
    }
    
    public function import_source_disk() {
        return Storage::disk(Config::get('datamanager.import_source.disk'));
    }
    
    public function import_source_path() {
        return Config::get('datamanager.import_source.prepend_path');
    }

    private function load_definition() {
        $definition = [];
        $files = Storage::disk(Config::get('datamanager.disk'))->files(Config::get('datamanager.path'));
        $definition['tables'] = [];
        foreach ($files as $file) {
            $pi = pathinfo($file);
            if ($pi['extension'] != 'yml') continue;
            $yaml = yaml_parse_file($file);
            foreach ($yaml as $item) {
                if (array_key_exists('table', $item)) $definition['tables'][$item['table']] = $item;
            }
        }
        $this->definition = $definition;
    }
}