<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 2/9/16
 * Time: 12:33 PM
 */

namespace Visionware\DataManager;

use Exception;

class DefinitionValidator {
    static private $require_fields = [
        'columns' => [
            'name',
            'column_type',
            'type',
            'null',
            'default',
            'extra',
        ],
        'indices' => [
            'name',
            'index_type',
            'unique',
            'columns',
        ],
        'index_columns' => [
            'name',
            'sequence'
        ],
        'import_join_columns' => [
            'local',
            'foreign',
            'sequence'
        ],
        'tables' => [
            'name',
            'columns',
            'indices',
            'table_type',
        ]
    ];

    static private function dot($items) { return implode('.', $items);}
    static private function error($path_array, $msg) { return self::dot($path_array) . " - $msg";}

    static private function validate_self(&$errors, $object, $key, $type, $path_array) {
        foreach (self::$require_fields[$type] as $field) {
            if (!array_key_exists($field, $object)) $errors[] = self::error($path_array, "Object has no '$field' field defined");
        }
        if (array_key_exists('name', $object) && $key != $object['name']) $errors[] = self::error($path_array, "Object key does not match defined name field");
    }

    static public function validate_column(&$errors, $definition, $table_key, $column_key) {
        $table = $definition['tables'][$table_key];
        $column = $table['columns'][$column_key];
        $indexed_columns = [];
        foreach ($table['indices'] as $index_key => $index) {
            foreach ($index['columns'] as $index_column_key => $index_column) {
                $indexed_columns[$index_column['name']] = true;
            }
        }
        $path_array = ['tables', $table_key, 'columns', $column_key];
        self::validate_self($errors, $column, $column_key, 'columns', $path_array);

        if (array_key_exists('import_field', $column) && !array_key_exists('import_file', $table)) $errors[] = self::error($path_array, "No import_file defined for table");

//        if ((substr($column['name'], -3) == '_id') || $column['name'] == 'id') {
//            if ($column['type'] != 'binary(16)') $errors[] = self::error($path_array, "ID column is not type binary(16)");
//
//            $is_in_indices = false;
//            foreach ($table['indices'] as $indexKey=>$index) {
//                foreach ($index['columns'] as $iColumnKey=>$iColumn) {
//                    if ($iColumn['name'] == $column['name']) {
//                        $is_in_indices = true;
//                        break 2;
//                    }
//                }
//            }
//            if (!$is_in_indices) $errors[] = self::error($path_array, "ID column is not defined as an index");
//        }
        switch ($column['column_type']) {
            case 'primary':
                if (!array_key_exists($column['name'], $indexed_columns)) $error[] = self::error($path_array, "Column is not defined as an index");
                if ($column['type'] != 'binary(16)') $errors[] = self::error($path_array, "ID column is not type binary(16)");
                break;
            case 'foreign_key':
            case 'foreign_id':
                if (!array_key_exists('references_column', $column)) $errors[] = self::error($path_array, "Foreign column is missing references_column definition");
                if (!array_key_exists('references_on', $column)) $errors[] = self::error($path_array, "Foreign table is missing references_on definition");
                if (!@array_key_exists($column['references_on'], $definition['tables'])) $errors[] = self::error($path_array, "Foreign table is missing");
                if (!@array_key_exists($column['references_column'], $definition['tables'][$column['references_on']]['columns'])) $errors[] = self::error($path_array, "Foreign table is missing foreign column definition");
                break;
            case 'column':
            case 'timestamp':
            case 'deleted_flag':
                break;
            default:
                $errors[] = self::error($path_array, "Unknown column_type");
        }
    }

    static public function validate_index(&$errors, $definition, $table_key, $index_key) {
        $table = $definition['tables'][$table_key];
        $index = $table['indices'][$index_key];
        $path_array = ['tables', $table_key, 'indices', $index_key];
        self::validate_self($errors, $index, $index_key, 'indices', $path_array);
        foreach ($index['columns'] as $index_column_key => $index_column) self::validate_index_column($errors, $definition, $table_key, $index_key, $index_column_key);
        if (array_key_exists('import_join_columns', $index)) {
            if (!array_key_exists('foreign_table', $index)) $errors[] = self::error($path_array, "Index is missing foreign_table definition");
            foreach ($index['import_join_columns'] as $import_join_column_key => $import_join_column) self::validate_import_join_column($errors, $definition, $table_key, $index_key, $import_join_column_key);
        }
        switch ($index['index_type']) {
            case 'primary';
            case 'unique_key':
                if ($index['unique'] !== true) $errors[] = self::error($path_array, "Index is required to be unique");
                break;
            case 'index':
            case 'foreign_key':
                break;
            default:
                $errors[] = self::error($path_array, "Unknown index_type");
        }
    }

    static public function validate_index_column(&$errors, $definition, $table_key, $index_key, $index_column_key) {
        $table = $definition['tables'][$table_key];
        $index = $table['indices'][$index_key];
        $index_column = $index['columns'][$index_column_key];
        $path_array = ['tables', $table_key, 'indices', $index_key, 'columns', $index_column_key];
        self::validate_self($errors, $index_column, $index_column_key, 'index_columns', $path_array);
        if (array_key_exists('foreign_table', $index) && !array_key_exists('foreign_column', $index_column)) $errors[] = self::error($path_array, "Index column is missing foreign_column definition");
        if (array_key_exists('foreign_column', $index_column) && !array_key_exists('foreign_table', $index)) $errors[] = self::error($path_array, "Index is missing foreign_table definition");
        if (!array_key_exists($index_column['name'], $table['columns'])) $errors[] = self::error($path_array, "Index column is not defined as a table column");
        if ($index['index_type'] == 'unique_key' && $table['table_type'] == 'imported' && $table['columns'][$index_column['name']]['column_type'] == 'foreign_id') $errors[] = self::error($path_array, "unique_key cannot contain id columns");
        if ($index['index_type'] == 'foreign_key' && count($index['columns']) > 1) $errors[] = self::error($path_array, "foreign_key index type cannot have more than one column");
        if ($table['table_type'] == 'imported' && $index['index_type'] == 'foreign_key' && !array_key_exists('import_join_columns', $index)) $errors[] = self::error($path_array, "foreign_key index type must have import_join_columns");

    }

    static public function validate_import_join_column(&$errors, $definition, $table_key, $index_key, $import_join_column_key) {
        $table = $definition['tables'][$table_key];
        $index = $table['indices'][$index_key];
        $import_join_column = $index['import_join_columns'][$import_join_column_key];
        $path_array = ['tables', $table_key, 'indices', $index_key, 'import_join_columns', $import_join_column_key];
        self::validate_self($errors, $import_join_column, $import_join_column_key, 'import_join_columns', $path_array);
        if (!array_key_exists($import_join_column['local'], $table['columns'])) $errors[] = self::error($path_array, 'Local column is not defined in table');
        if (!array_key_exists($import_join_column['foreign'], $definition['tables'][$index['foreign_table']]['columns'])) $errors[] = self::error($path_array, 'Foreign column is not defined in foreign table');
    }

    static public function validate_table(&$errors, $definition, $table_key) {
        $table = $definition['tables'][$table_key];
        $path_array = ['tables', $table_key];
        self::validate_self($errors, $table, $table_key, 'tables', $path_array);

        if (array_key_exists('import_file', $table) && !array_key_exists('unique_key', $table['indices'])) $errors[] = self::error($path_array, "Table has import_file but no unique_key index");
        foreach ($table['columns'] as $column_key => $column) self::validate_column($errors, $definition, $table_key, $column_key);
        foreach ($table['indices'] as $index_key => $index) self::validate_index($errors, $definition, $table_key, $index_key);

        if ($table['table_type'] == "imported") {
            if (!array_key_exists('import_file', $table)) $errors[] = self::errors($path_array, "Table is marked as imported type but has no import_file definition");
        } else if ($table['table_type'] == 'internal') {
        } else if ($table['table_type'] == 'jobson') {
        } else {
            $errors[] = self::error($path_array, "Unknown table_type");
        }
    }

    static public function validate_import(&$errors, $definition, $import_key) {
        $path_array = ['import_order', $import_key];
        if (!array_key_exists($import_key, $definition['tables'])) $errors[] = self::error($path_array, "Table is not defined");
        if (!array_key_exists('import_file', $definition['tables'][$import_key])) $errors[] = self::error($path_array, "Table has no import_file defined");
    }

    static public function validate($definition) {
        $errors = [];
        try {
            foreach ($definition['import_order'] as $import_key) self::validate_import($errors, $definition, $import_key);
            foreach ($definition['tables'] as $table_key => $table) self::validate_table($errors, $definition, $table_key);
        } catch (Exception $e) {
            $errors[] = "EXCEPTION CAUGHT: " . $e->getMessage() . " in file " . $e->getFile() . " at line " .  $e->getLine();
        }
        if (count($errors) > 0) array_unshift($errors, 'DEFINITION VALIDATION FAILED!');

        return $errors;
    }
}