<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/13/16
 * Time: 10:16 AM
 */

namespace Visionware\DataManager;

use DB;
use Exception;
use Illuminate\Support\Collection;
use Monolog\Logger;
use Visionware\DataManager\Definition\TableDefinition;
use Visionware\DataManager\Facades\DataManager;
use Carbon\Carbon;
use Illuminate\Database\Connection;

/**
 * Class Importer
 * @package Visionware\DataManager
 * @method emergency(string $msg, array $context = [])
 * @method alert(string $msg, array $context = [])
 * @method critical(string $msg, array $context = [])
 * @method error(string $msg, array $context = [])
 * @method warning(string $msg, array $context = [])
 * @method notice(string $msg, array $context = [])
 * @method info(string $msg, array $context = [])
 */
class Importer extends DataManagerProcess {
    /** @var Connection */
    protected $liveDb;
    /** @var Connection */
    protected $importDb;
    protected $insertSelectPairs;
    protected $updates;
    protected $joins;
    protected $wheres;
    protected $fromDb;
    protected $fromTable;
//    protected $filesTable;
    protected $iterations;
//    protected $sqlLogFile;
    protected $lastIngested;
    protected $noTransactions;
    protected $noPostSql;
    protected $onlyPostSql;
    protected $only;

    public function __construct($schema) {
        parent::__construct($schema);
        $this->liveDb = DataManager::getLiveConnection();
        $this->importDb = DataManager::getImportConnection();
        $this->noTransactions = false;
        $this->noPostSql = false;
        $this->onlyPostSql = false;
        $this->only = false;
    }

    function __destruct() {
//        fclose($this->sqlLogFile);
    }

    protected function setDestinationTable(TableDefinition $definition) {
        $this->definition = $definition;
        $this->tableName = $definition->name();
        $this->fromDb = $this->importDb->getDatabaseName();
        if (!$this->definition->isSummaryTable()) {
            $this->fromTable = "{$definition->name()}_latest";
        } else {
            $this->fromTable = "{$this->definition->getSummarySourceTable()}_latest";
        }
        $this->insertSelectPairs = new Collection;
        $this->updates = new Collection;
        $this->joins = new Collection;
        $this->wheres = new Collection;
        $this->iterations = 1;
//        $this->sqlLogFile = fopen(storage_path("logs/statements.log"), 'w');
        $this->lastIngested = false;
    }

    public function skipPostSql() {
        $this->noPostSql = true;
        $this->onlyPostSql = false;
    }

    public function onlyPostSql() {
        $this->noPostSql = false;
        $this->onlyPostSql = true;
    }

    public function go() {
        $allStart = Carbon::now();
        try {
            $this->info("Acquiring import lock...");
            DataManager::acquireImportLock();
            if (!$this->noTransactions) $this->liveDb->beginTransaction();
            if (!$this->onlyPostSql) {
                foreach ($this->schema->tablesToImport() as $tableName) {
                    if ($this->only && !($tableName == $this->only)) continue;
                    $this->setDestinationTable($tableName->definition());
                    $this->importTable();
                }
            }
            /** THIS IS NO LONGER NECESSARY, USE THE datamanager:updater COMMAND INSTEAD */
//            if (!$this->noPostSql) {
//                $this->notice('Running post-import SQLs...');
//                $sqls = DataManagerHelper::getPostImportSql();
//
//                $newCommitSequences = array();
//                foreach ($sqls as $sql) {
//                    if (strpos($sql, '%%LOCK-TABLE:') !== false) {
//                        $table = str_replace('%%LOCK-TABLE:', '', $sql);
//                        if (!isset($newCommitSequences[$table])) {
//                            $newCommitSequences[$table] = $this->liveDb->table($table)->lockForUpdate()->max('commit_sequence') + 1;
//                        }
//                        $this->notice("Locking Table: $table, New Commit Sequence: {$newCommitSequences[$table]}");
//                        continue;
//                    }
//
//                    if (strpos($sql, '%%COMMIT-SEQUENCE:') !== false) {
//                        preg_match_all('/%%COMMIT-SEQUENCE:([^%]+)%%/', $sql, $matches, PREG_SET_ORDER );
//                        foreach ($matches as $match) {
//                            $tag = $match[0];
//                            $table = $match[1];
//                            if (isset($newCommitSequences[$table])) {
//                                $sql = str_replace($tag, $newCommitSequences[$table], $sql);
//                            }
//                        }
//                    }
//
//                    if (strpos($sql, '%%BREAK%%') !== false) {
//                        break;
//                    }
//
//                    $this->setLogPrefix('"' . substr($sql, 0, 40) . '..."');
//                    $this->notice("Executing statement...");
//                    $this->info($sql);
//                    $qStart = Carbon::now();
//                    $count = $this->liveDb->affectingStatement($sql);
//                    $qDiff = Carbon::now()->diffForHumans($qStart, true);
//                    $this->notice("Affected $count rows in $qDiff total");
//                }
//            }
            if (!$this->noTransactions) $this->liveDb->commit();
        } catch (Exception $e) {
            $this->error("Caught exception while importing table! Rolling back...", ['exception' => $e]);
            if (!$this->noTransactions) $this->liveDb->rollBack();
        } finally {
            $this->info("Releasing import lock...");
            DataManager::releaseImportLock();
        }
        $diff = Carbon::now()->diffForHumans($allStart, true);
        $this->notice("Finished import process in $diff total");
    }

    public function noTransactions() {
        $this->noTransactions = true;
    }

    public function only($tableName) {
        $this->only = $tableName;
    }

    protected function pushInsert($insert, $select) {
        $this->insertSelectPairs->put($insert, "$select AS $insert");
    }
    
    protected function pushJoin($joinStatement) {
        $this->joins->push($joinStatement);
    }

    protected function pushUpdate($updateStatement) {
        $this->updates->push($updateStatement);
    }

    protected function pushWhere($whereStatement) {
        $this->wheres->push($whereStatement);
    }

    public function importTable() {
        $start = Carbon::now();
        $this->setLogPrefix($this->definition->name());
        $this->notice("Importing table...");
        $joinedTables = [];
        if ($this->definition->hasUuid()) {
            $this->pushInsert('id', "COALESCE({$this->compileField('id', $this->definition->name())}, UuidToBin(UUID()))");
        }
        $join = ['table' => $this->definition->name(), 'using' => $this->definition->key()];
        $joinedTables[$this->definition->isSummaryTable() ? $this->definition->getSummarySourceTable() : $this->definition->name()] = true;
        $this->pushJoin($this->compileJoin($join, $this->definition->name()));
        foreach ($this->definition->fields() as $field) {
            if ($field->skipImport()) continue;
            if ($field->hasImportField()) { //exists in History DB
                $this->pushInsert($field->name(), $this->compileField($field->name()));
            } else if ($field->hasForeign()) {
                $joins = $field->getJoins();
                foreach ($joins as $join) {
                    if ($join['table'] == $this->definition->name()) $this->iterations++;
                    $alias = $field->name() . '_' . $join['table'] . '_Join';
                    $this->pushInsert($field->name(), $this->compileField($field->foreign(), $alias));
                    $joinedTables[$join['table']] = true;
                    $this->pushJoin($this->compileJoin($join, $alias));
                }
            } else if ($field->hasImportedFrom()) {
                $this->pushInsert($field->name(), $this->compileField($field->name()));
            }
            $this->pushUpdate("{$field->name()} = VALUES({$field->name()})");
        }

        if (!$this->isThereUpdatedData(array_keys($joinedTables)) && !$this->force) {
            $this->info("No new data, skipping table.");
            return;
        }

        $newCommitSequence = false;
        if ($this->definition->isSyncedDown()) {
            $this->info('Getting new commit sequence number');
            $newCommitSequence = $this->liveDb->table($this->definition->name())->lockForUpdate()->max('commit_sequence') + 1;
            $this->info("New commit sequence number: $newCommitSequence");
        }

        $this->pushInsert('date_modified', $this->compileField('date_modified'));
        $this->pushUpdate("date_modified = VALUES(date_modified)");
        if ($this->definition->isSyncedDown()) {
//            $this->pushInsert('date_uploaded', $this->compileField('date_modified'));
//            $this->pushUpdate("date_uploaded = VALUES(date_modified)");
        }

        if (!$this->force) {
            $this->pushWhere("(`{$this->definition->name()}`.`date_modified` IS NULL OR `{$this->fromTable}`.`date_modified` > `{$this->definition->name()}`.`date_modified`)");
        } else {
//            $this->pushUpdate("date_created = VALUES(date_modified)");
//            $this->pushUpdate("date_uploaded = VALUES(date_modified)");
        }
        $this->pushUpdate("deleted_at = 0");
        $insertSql = $this->compileInsert($newCommitSequence);
        $updateSql = $this->compileUpdateDeleted($newCommitSequence);

        for ($count = 1; $count <= $this->iterations; $count++) {

            $this->info($this->definition->name() . ": Inserting into live (iteration $count)...");
            $this->logSql($insertSql);
            $this->liveDb->affectingStatement($insertSql);
            $this->info($this->definition->name() . ": Updating deleted records (iteration $count)...");
            $this->logSql($updateSql);
            $this->liveDb->affectingStatement($updateSql);
        }

        $end = Carbon::now();
        $diff = $end->diffForHumans($start, true);
        $sql = $this->compileUpdateLastImport($end->diffInSeconds($start, true));
        $this->logSql($sql);
        $this->liveDb->affectingStatement($sql);

        $this->notice("Completed importing table in $diff total");
    }

    protected function isThereUpdatedData($joinedTables) {
        foreach ($joinedTables as $joinedTable) {
            $this->debug("Checking $joinedTable for new data...");
            $joinDefinition = $this->definition->schema()->tables()->get($joinedTable);
            $joinFromTable = $joinDefinition->isSummaryTable() ? $joinDefinition->getSummarySourceTable() : $joinedTable;
            $lastImported = $this->liveDb->table($joinedTable)->value(DB::raw('MAX(date_modified)'));
            $lastIngested = $this->importDb->table("{$joinFromTable}_latest")->value(DB::raw('MAX(date_modified)'));
            $this->debug("Last $joinedTable imported time: $lastImported");
            $this->debug("Last $joinedTable ingested time: $lastIngested");
            if ($lastIngested > $lastImported) {
                $this->lastIngested = $lastIngested = $this->importDb->table($this->fromTable)->value(DB::raw('MAX(date_modified)'));;
                return true;
            }
        }
        return false;
    }

    protected function compileInsert($newCommitSequence) {
        $insertString = implode(",\n  ", $this->insertSelectPairs->keys()->toArray());
        $selectString = implode(",\n  ", $this->insertSelectPairs->values()->toArray());
        $joinsString = implode("\n", $this->joins->toArray());
        $updatesString = implode(",\n  ", $this->updates->values()->toArray());
        $wheresString = implode(' AND ', $this->wheres->values()->toArray());
        $commitSequenceInsertString = $newCommitSequence ? ', commit_sequence' : '';
        $commitSequenceSelectString = $newCommitSequence ? ", $newCommitSequence" : '';
        $commitSequenceUpdateString = $newCommitSequence ? ", `commit_sequence` = $newCommitSequence" : '';

        if (strlen($wheresString)) $wheresString = "WHERE $wheresString";
        return <<<SQL
INSERT IGNORE INTO `{$this->liveDb->getDatabaseName()}`.`{$this->definition->name()}` (
  $insertString $commitSequenceInsertString
)
SELECT
  $selectString $commitSequenceSelectString
FROM `{$this->fromDb}`.`{$this->fromTable}`
$joinsString
$wheresString 
ON DUPLICATE KEY UPDATE
  $updatesString $commitSequenceUpdateString
;
SQL;
    }
    
    protected function compileUpdateDeleted($newCommitSequence) {
        $destinationDb = $this->liveDb->getDatabaseName();
        $definition = $this->definition->name();
        $sourceDb = $this->fromDb;
        $sourceTable = $this->fromTable;
        $joinStrings = [];
        $commitSequenceString = $newCommitSequence ? ", `commit_sequence` = $newCommitSequence" : '';
        foreach ($this->definition->key() as $keyColumn) {
            $joinStrings[] = "`$definition`.`$keyColumn` = `$sourceTable`.`$keyColumn`";
        }
        $joinString = implode(' AND ', $joinStrings);
        return <<<SQL
UPDATE `$destinationDb`.`$definition`
LEFT OUTER JOIN `$sourceDb`.`$sourceTable`
ON ($joinString)
SET `$definition`.`deleted_at` = NOW() $commitSequenceString 
WHERE `$sourceTable`.`record_hash` IS NULL
AND `$definition`.`deleted_at` = 0;
SQL;

    }

//    protected function compileDestinationField($column) {
//        return $this->compileField($column, $this->fromTable);
//    }

//    protected function compileHistoryField($column) {
//        return $this->compileField($column, $this->fromTable);
//    }

    protected function compileField($column, $table = false) {
        if ($table === false) $table = $this->fromTable;
        return "`$table`.`$column`";
    }

    protected function compileUpdateLastImport($duration) {
        return <<<SQL
INSERT IGNORE INTO `{$this->liveDb->getDatabaseName()}`.`last_imported`
(table_name, last_imported, import_duration)
VALUES ('{$this->definition->name()}', '{$this->lastIngested}', $duration)
ON DUPLICATE KEY UPDATE last_imported = VALUES(last_imported), import_duration = VALUES(import_duration)
SQL;
    }

    protected function compileJoin($joinArray, $alias) {
        $linkString = '';
        if (array_key_exists('using', $joinArray)) {
            $linkString = 'USING (' . implode(', ', $joinArray['using']) . ')';
        } else {
            $onStrings = [];
            foreach ($joinArray['ons'] as $on) {
                $onStrings[] = "`$alias`.`$on[foreign]` = `{$this->fromTable}`.`$on[local]`";
            }
            $linkString = 'ON (' . implode(' AND ', $onStrings) . ')';
        }

        return "LEFT OUTER JOIN `{$this->liveDb->getDatabaseName()}`.`$joinArray[table]` AS $alias $linkString";
    }

    private function logSql($sql) {
//        fputs($this->sqlLogFile, "$sql\n\n");
        $this->debug($sql);
    }
}