<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/20/16
 * Time: 10:22 AM
 */

namespace Visionware\DataManager\Info;

use Closure;
use Illuminate\Support\Collection;
use Visionware\DataManager\Actions\ActionManager;
use Visionware\DataManager\Definition\SchemaDefinition;
use Visionware\DataManager\Definition\TableDefinition;
use Visionware\DataManager\Info\SchemaInfo;

class TableInfo extends Info {
    /**
     * @var SchemaInfo
     */
    private $schema;
    private $name;
    private $columns;
    private $indices;
    private $dependencies;

    protected function __construct() {
        parent::__construct();
        $this->columns = new Collection;
        $this->indices = new Collection;
        $this->dependencies = new Collection;
    }

    public function setName($name) {
        $this->name = $name;
    }

    public function setSchema($schema) {
        $this->schema = $schema;
    }

    public function setDefinition(TableDefinition $definition) {
        $this->putOther('definition', $definition);
    }

    public function name() {
        return $this->name;
    }

    /**
     * If this table is imported from 
     * @return string
     */
    public function getSourceTableName() {
        $schemaName = $this->schema()->schemaName();
        if (strlen($schemaName)) return "`$schemaName`.`{$this->name()}`";
        return "`{$this->name()}`";
    }
    /**
     * @return Collection
     */
    public function columns() {
        return $this->columns;
    }

    public function definition() {
        return $this->getOther('definition');
    }
    
    public function clone() {
        $instance = self::create($this->name());
        foreach ($this->columns() as $column) $instance->putColumn($column->clone());
        foreach ($this->indices() as $index) $instance->putIndex($index->clone());

        $instance->setDefinition($this->definition());
        return $instance;
    }

    public function setDerived() {
        $this->putOther('derived', true);
    }

    public function isDerived() {
        return $this->getOther('derived', false);
    }

    public function putColumn(ColumnInfo $column) {
        $column->setTable($this);
        $this->columns->put($column->name(), $column);
    }

    public function prependColumn(ColumnInfo $column) {
        $column->setTable($this);
        $this->columns->prepend($column, $column->name());
    }

    public function setColumns(Collection $columns) {
        $this->columns = new Collection;
        foreach ($columns as $column) $this->putColumn($column);
    }

    public function deleteColumn($column) {
        $name = ($column instanceof ColumnInfo) ? $column->name() : $column;
        $this->columns()->offsetUnset($name);
    }

    public function putIndex(IndexInfo $index) {
        $index->setTable($this);
        $this->indices->put($index->name(), $index);
    }

    public function prependIndex(IndexInfo $index) {
        $index->setTable($this);
        $this->indices->prepend($index, $index->name());
    }

    public function setIndices(Collection $indices) {
        $this->indices = new Collection;
        foreach ($indices as $index) $this->putIndex($index);
    }

    public function clearIndices() {
        $this->indices = new Collection;
    }
    
    public function indices() {
        return $this->indices;
    }

    public function schema() {
        return $this->schema;
    }

    public function shouldIngest() {
        $definition = $this->definition();
        if (is_null($definition)) return false;
        return $this->definition()->isInHistory();
    }

    public function shouldImport() {
        $definition = $this->definition();
        if (is_null($definition)) return false;
        return $this->definition()->isImported();
    }

    public function getPrimaryKeyColumns() {
        $primaryKey = $this->primaryKey();
        if ($primaryKey) return $this->primaryKey()->columns();
        else return [];
    }

    public function primaryKey() {
        foreach ($this->indices() as $index) {
            if ($index->isPrimaryKey()) return $index;
        }
        return false;
    }
    
    public function columnsToAdd(TableInfo $current) {
        return $this
            ->columns()
            ->only(
                array_diff(
                    $this->columns()->keys()->toArray(),
                    $current->columns()->keys()->toArray()
                )
            )
        ;
    }

    public function columnsToDrop(TableInfo $currentTable) {
        return $currentTable
            ->columns()
            ->only(
                array_diff(
                    $currentTable->columns()->keys()->toArray(),
                    $this->columns()->keys()->toArray()
                )
            )
        ;
    }

    private function columnsToCompare(TableInfo $currentTable) {
        return $this
            ->columns()
            ->only(
                array_intersect(
                    $this->columns()->keys()->toArray(),
                    $currentTable->columns()->keys()->toArray()
                )
            )
            ;
    }

    public function columnsToChange(TableInfo $currentTable) {
        return $this->columnsToCompare($currentTable)->filter(function ($column, $key) use ($currentTable) {
            $currentColumn = $currentTable->columns()->get($key);
            return !$column->compareTo($currentColumn);
        });
    }

    public function indicesToAdd(TableInfo $current) {
        return $this
            ->indices()
            ->only(
                array_diff(
                    $this->indices()->keys()->toArray(),
                    $current->indices()->keys()->toArray()
                )
            )
            ;
    }

    public function indicesToDrop(TableInfo $current) {
        return $current
            ->indices()
            ->only(
                array_diff(
                    $current->indices()->keys()->toArray(),
                    $this->indices()->keys()->toArray()
                )
            )
            ;
    }

    private function indicesToCompare(TableInfo $current) {
        return $this
            ->indices()
            ->only(
                array_intersect(
                    $this->indices()->keys()->toArray(),
                    $current->indices()->keys()->toArray()
                )
            )
            ;
    }
    
    public function dependentTables() {
        $depends = new Collection;
        foreach ($this->definition()->fields() as $field) {
            $dependentTable = false;
            if ($field->onTable()) $dependentTable = $this->schema()->tables()->get($field->onTable());
            else if ($field->importedFrom()) $dependentTable = $this->schema()->tables()->get($field->importedFrom());
            if ($dependentTable && $dependentTable->name() != $this->name()) $depends->push($dependentTable);
        }
        return $depends;
    }

    public function buildDependencies() {
        if (!$this->shouldImport()) return;
        foreach ($this->dependentTables() as $dependentTable) {
            $this->dependsOn($dependentTable);
        }
    }

    public function dependsOn(TableInfo $tableInfo) {
        $this->dependencies->push($tableInfo);
    }

    public function resolveDependencies(Collection $resolved) {
        foreach ($this->dependencies as $dependency) {
            if (!$resolved->contains($dependency->name())) $dependency->resolveDependencies($resolved);
        }
        $resolved->put($this->name(), $this);
    }

    public function dependencyTree(Collection $resolved, Collection $output, $level = 0) {
//        if ($level) {
//            print "\\\n\\\n" . str_repeat('+--', $level) . "+--+";
//        }
        foreach ($this->dependencies as $dependency) {
            if (!$resolved->contains($dependency->name())) {
                $dependency->dependencyTree($resolved, $output, $level + 1);
            }
        }
        $output->push(str_repeat('--', $level) . ($level ? ' ' : '' ) . $this->name());
        $resolved->put($this->name(), $this);
    }

    public function indicesToChange(TableInfo $current) {
        $result = $this->indicesToCompare($current)->filter(function ($index, $key) use ($current) {
            $currentIndex = $current->indices()->get($key);
            return !$index->compareTo($currentIndex);
        });
        return $result;
    }

    public static function createFromDefinition(TableDefinition $table) {
        $instance = new static();
        $instance->setName($table->name());
        $instance->setDefinition($table);
        foreach ($table->fields() as $fieldDefinition) {
            $columnInfo = ColumnInfo::createFromDefinition($fieldDefinition);
            $instance->putColumn($columnInfo);
        }
        foreach ($table->indices() as $indexDefinition) {
            $indexInfo = IndexInfo::createFromDefinition($indexDefinition);
            $instance->putIndex($indexInfo);
        }
        return $instance;
    }

    public static function create($name, $columns = null, $indices = null) {
        $instance = new static();
        $instance->setName($name);
        if ($columns) $instance->setColumns($columns);
        if ($indices) $instance->setIndices($indices);
        return $instance;
    }

    public function __toString() {
        return "{$this->name()}";
    }

    public function __toDebugArray() {
        return [
            'name' => $this->name(),
        ];
    }
}