<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 1/27/16
 * Time: 4:40 PM
 */

namespace Visionware\DataManager;

use DB;

class SchemaCreator {
    protected $definition;
//    protected $drops;
    protected $default_statements = [
        'create_schema' => 'CREATE DATABASE IF NOT EXISTS `%name%`;',
        'use_schema' => 'USE `%name%`;',
        'uuid_to_bin' => 'CREATE FUNCTION UuidToBin(_uuid BINARY(36)) RETURNS BINARY(16) LANGUAGE SQL  DETERMINISTIC  CONTAINS SQL  SQL SECURITY INVOKER RETURN UNHEX(CONCAT( SUBSTR(_uuid, 15, 4), SUBSTR(_uuid, 10, 4), SUBSTR(_uuid,  1, 8), SUBSTR(_uuid, 20, 4), SUBSTR(_uuid, 25) ));',
        'uuid_from_bin' => 'CREATE FUNCTION UuidFromBin(_bin BINARY(16)) RETURNS CHAR(36) LANGUAGE SQL  DETERMINISTIC  CONTAINS SQL  SQL SECURITY INVOKER RETURN LCASE(CONCAT_WS(\'-\', HEX(SUBSTR(_bin,  5, 4)), HEX(SUBSTR(_bin,  3, 2)), HEX(SUBSTR(_bin,  1, 2)), HEX(SUBSTR(_bin,  9, 2)), HEX(SUBSTR(_bin, 11)) ));',
        'create_table' => 'CREATE TABLE `%table%` (%columns_string%, %indices_string%) ENGINE=InnoDB DEFAULT CHARSET=utf8;',
        'metadata_table' => 'CREATE TABLE `datamanager_metadata` ( `metadata_type` VARCHAR(32) NOT NULL, `metadata_key` VARCHAR(32) NOT NULL, `int_value` INT(11) DEFAULT NULL, `text_value` VARCHAR(255) DEFAULT NULL, `longtext_value` LONGTEXT DEFAULT NULL, `date_value` datetime DEFAULT NULL, PRIMARY KEY (`metadata_type`, `metadata_key`) ) ENGINE=InnoDB DEFAULT CHARSET=utf8;',
        'metadata_insert' => 'INSERT INTO `datamanager_metadata` (`metadata_type`, `metadata_key`, `longtext_value`) VALUES ("definition", "definition", \'%definition%\');',
    ];
    protected $statements = [
        'schema' => [
            'create_schema' => 'create_schema',
            'use_schema'    => 'use_schema',
            'uuid_to_bin' => 'uuid_to_bin',
            'uuid_from_bin' => 'uuid_from_bin',
        ],
        'table'  => [
            'create_table' => 'create_table',
        ],
    ];
    protected $sql_schema_name;
    protected $table_name_postfix;
    protected $skip_uuid_functions = false;

    public function __construct($definition, $drops = false, $sql_schema_name) {
        $this->definition = $definition;
        $this->sql_schema_name = $sql_schema_name;
        $this->table_name_postfix = '';
    }

    public function build_schema() {
        $sqls = $this->build_schema_statements();
        foreach ($this->definition['tables'] as $table) {
            $table = $this->filter_table($table);
            if ($table === false) continue;

            foreach ($table['columns'] as $column_key => $column) {
                $new_column = $this->filter_column($column, $table);
                if ($new_column !== false) $table['columns'][$column_key] = $new_column;
                else unset($table['columns'][$column_key]);
            }
            foreach ($table['indices'] as $index_key => $index) {
                $new_index = $this->filter_index($index, $table);
                if ($new_index !== false) $table['indices'][$index_key] = $new_index;
                else unset($table['indices'][$index_key]);
            }

            $sqls = array_merge($sqls, $this->build_table_statements($table));
        }
        return $sqls;
    }

    private function expand_sql_template($template, $variables) {
        $matches = false;
        preg_match_all('/%(.*?)%/', $template, $matches);
        foreach ($matches[1] as $match) {
            $template = str_replace("%$match%", $variables[$match], $template);
        }
        return $template;
    }

    private function build_schema_statements() {
        $sqls = [];
        foreach ($this->statements['schema'] as $key => $value) {
            $sql_template = (array_key_exists($value, $this->default_statements)) ? $this->default_statements[$value] : $value;
            $vars = [
                'name' => $this->sql_schema_name,
            ];

            $sqls[$key] = $this->expand_sql_template($sql_template, $vars);
        }
        $vars = [
            'definition' => addslashes(json_encode($this->definition)),
        ];
        $sqls['metadata_table'] = $this->expand_sql_template($this->default_statements['metadata_table'], $vars);
        $sqls['metadata_insert'] = $this->expand_sql_template($this->default_statements['metadata_insert'], $vars);
        return $sqls;
    }

    private function build_table_statements($in_table) {
        $sqls = [];
        foreach ($this->statements['table'] as $key => $value) {
            $table = $in_table;
            if (method_exists($this, "filter_$key")) $table = call_user_func([$this, "filter_$key"], $table);

            $sql_template = (array_key_exists($value, $this->default_statements)) ? $this->default_statements[$value] : $value;

            $column_strings = [];
            foreach ($table['columns'] as $column) $column_strings[] = $this->stringify_column($column);
            $index_strings = [];
            foreach ($table['indices'] as $index) $index_strings[] = $this->stringify_index($index);

            $vars = [
                'table' => $table['name'],
                'columns_string' => implode(', ', $column_strings),
                'indices_string' => implode(', ', $index_strings),
            ];

            $sqls[$table['name'].'.'.$key] = $this->expand_sql_template($sql_template, $vars);
        }
        return $sqls;
    }

    protected function filter_table($table) { return $table; }
    protected function filter_column($column, $table) { return $column; }
    protected function filter_index($index, $table) { return $index; }

    private function stringify_column($column) {
            $column_string = "`$column[name]` $column[type]";
            if (!$column['null']) $column_string .= ' NOT NULL';

            if (!is_null($column['default']) && strlen($column['default'])) $column_string .= " DEFAULT $column[default]";
            else if ($column['null']) $column_string .= " DEFAULT NULL";

            if (array_key_exists('extra', $column) && $column['extra']) $column_string .= " $column[extra]";

            return $column_string;
    }

    private function stringify_index($index) {
        $index_string = '';

        if ($index['name'] == 'PRIMARY') {
            $index_string = 'PRIMARY ';
            $index['name'] = '';
        } else {
            $index['name'] = " `$index[name]`";
            if ($index['unique']) $index_string .= 'UNIQUE ';
        }
        uasort(
            $index['columns'],
            function ($a, $b) {
                return $a['sequence'] - $b['sequence'];
            }
        );
        $columns_string = implode("`, `", array_pluck($index['columns'], 'name'));

        $index_string .= "KEY$index[name] (`$columns_string`)";
        return $index_string;
    }
}