<?php

namespace Visionware\DataManager\Console\Commands;

use Carbon\Carbon;
use DB;
use Exception;
use Storage;
use Symfony\Component\Console\Output\OutputInterface;

class Import extends DataManagerCommand {
    protected $datamanager_command_name = 'datamanager:import';
    protected $datamanager_signature = ' {--force : Import tables regardless of when they were last imported}';

    protected $description = 'Imports from history database into live database';
    protected $live_sql = [];

    protected function datamanager_handle() {
        $start = Carbon::now();
        $history = DB::connection($this->config['history-database-connection'])->getDatabaseName();
        $live = DB::connection($this->config['live-database-connection'])->getDatabaseName();
        $db = DB::connection($this->config['live-database-connection']);

        $to_import_count = count($this->definition['import_order']);
        $imported_count = 0;
        $db->beginTransaction();
        try {
            foreach ($this->definition['import_order'] as $table_name) {
                /** 1) Get most recent data from history table and load it into the staging table
                 *  2) Populate the staging table with foreign key ID values by using the key fields defined
                 *  3) Update the live tables with the filled-in staging table values
                 */
                $imported_count++;

                $tbl_start = Carbon::now();
                $this->current_target = $table_name;
                $this->info("Importing table ($imported_count of $to_import_count)...");

                $definition = $this->definition['tables'][$table_name];

                $source_table = $history . '.' . $table_name . "_latest";
                $get_date_table = $history . '.' . $table_name . "_files";
                $live_table = $live . '.' . $table_name;

                $source_last_modified = $db->table($get_date_table)->max('last_modified');
                if ($this->option('force') !== true) {
                    $last_updated =
                        $db->table('datamanager_metadata')
                            ->where('metadata_type', '=', 'last_updated')
                            ->where('metadata_key', '=', $table_name)
                            ->first()->date_value;

                    if (!is_null($last_updated) && $last_updated > $source_last_modified) {
                        $this->info("Table contains most recent data already, skipping", ['last_modified' => $source_last_modified, 'last_updated' => $last_updated]);
                        continue;
                    }
                }

                $unique_key = $definition['indices']['unique_key']['columns'];
                uasort(
                    $unique_key,
                    function ($a, $b) {
                        return $a['sequence'] - $b['sequence'];
                    }
                );
                $unique_key_columns = [];
                foreach ($unique_key as $xx => $yy) $unique_key_columns[] = $yy['name'];
                $unique_key_columns_string = implode(', ', $unique_key_columns);

                $joins = [];
                $joins[$table_name] = "LEFT OUTER JOIN $live_table USING ($unique_key_columns_string)";
                $table_name_mapping = [];
                $letter = "A";
                foreach ($definition['indices'] as $index_key => $index) {
                    if ($index['index_type'] != 'foreign_key'
                        && !array_key_exists(
                            'import_join_columns',
                            $index
                        )
                    ) {
                        continue;
                    }
                    $foreign_table_base = "$live.$index[foreign_table]";
                    $foreign_table = $index['foreign_table'];
                    while (array_key_exists($foreign_table, $joins)) {
                        $foreign_table =
                            $index['foreign_table'] . $letter++;
                    }
                    uasort(
                        $index['import_join_columns'],
                        function ($a, $b) {
                            return $a['sequence'] - $b['sequence'];
                        }
                    );
                    $join_ons = [];
                    foreach ($index['import_join_columns'] as $import_join_column) {
                        $join_ons[] = $foreign_table
                            . '.'
                            . $import_join_column['foreign']
                            . ' = '
                            . $source_table
                            . '.'
                            . $import_join_column['local'];
                    }
                    $join_ons_string = implode(' AND ', $join_ons);

                    $joins[$foreign_table] = "LEFT OUTER JOIN $foreign_table_base $foreign_table ON ($join_ons_string)";
                    $table_name_mapping[$index['foreign_table']] = $foreign_table;
                }
                $joins_string = implode(" ", $joins);

                $select_columns = [];
                $live_columns = [];
                $live_updates = [];
                foreach ($definition['columns'] as $column_key => $column) {
                    if (!in_array($column['column_type'], ['primary', 'column', 'foreign_key', 'foreign_id', 'deleted_flag'])) continue;
                    if ($column['column_type'] == 'primary') {
                        $select_columns[] = "COALESCE($live_table.id, UuidToBin(UUID()))";
                    } else if ($column['column_type'] == 'deleted_flag') {
                        $select_columns[] = "NULL";
                    } else {
                        if ($column['column_type'] == 'foreign_id') {
                            $select_columns[] =
                                $table_name_mapping[$column['references_on']]
                                . ".$column[references_column] $column[name]";
                        } else {
                            $select_columns[] = $this->import_transform($column, $source_table) . " $column[name]";
                        }
                    }

                    $live_columns[] = $column['name'];

                    if ($column['column_type'] == 'deleted_flag') $live_updates[] = "$column[name]=NULL";
                    else if ($column['column_type'] != 'primary') $live_updates[] = "$column[name]=VALUES($column[name])";

                }

                $select_columns_string = implode(', ', $select_columns);
                $live_columns_string = implode(', ', $live_columns);
                $live_updates_string = implode(', ', $live_updates);

                $this->info("Updating live table $live_table...");
                $sql =
                    "INSERT IGNORE INTO $live_table ($live_columns_string) SELECT $select_columns_string FROM $source_table $joins_string ON DUPLICATE KEY UPDATE $live_updates_string;";
                $this->debug($sql);
                $db->affectingStatement($sql);

                $this->info('Updating deleted records...');
                $sql = "UPDATE $live_table LEFT OUTER JOIN $source_table USING ($unique_key_columns_string) SET $live_table.deleted_at = NOW() WHERE $source_table.id IS NULL AND $live_table.deleted_at IS NULL;";
                $this->debug($sql);
                $db->affectingStatement($sql);

                $this->info('Updating last modified metadata record...');
                $sql = "REPLACE INTO datamanager_metadata (`metadata_type`, `metadata_key`, `date_value`) VALUES ('last_updated', '$table_name', '$source_last_modified')";
                $this->debug($sql);
                $db->affectingStatement($sql);

                $this->notice("Completed importing table in " . Carbon::now()->diffForHumans($tbl_start, true));
            }
        } catch (Exception $e) {
            $db->rollBack();
            $this->error('Exception caught', ['exception' => $e]);
            return;
        }
        $this->info("Committing...");
        $db->commit();
        $this->info("Import complete in " . Carbon::now()->diffForHumans($start, true));
    }

    private function import_transform($column, $table) {
        if (!array_key_exists('import_transformation', $column)) return "$table.$column[name]";
        return str_replace('$$$', "$table.$column[name]", $column['import_transformation']);
    }
}

//    protected function updateschema() {
//        foreach ($this->definition['tables'] as $tt => $table) {
//            $type = 'internal';
//            if (array_key_exists('import_file', $table)) $type = 'imported';
//
//            $this->definition['tables'][$tt]['table_type'] = $type;
//
//            $ij_columns = [];
//            $ii_columns = [];
//            foreach ($table['indices'] as $ii => $index) {
//                $type = 'index';
//                if ($index['name'] == 'PRIMARY') {
//                    $type = 'primary';
//                } else {
//                    if ($index['name'] == 'unique_key') {
//                        $type = 'unique_key';
//                    } else {
//                        if (array_key_exists('foreign_table', $index)) {
//                            if (array_key_exists('import_join_columns', $index)) {
//                                $type = 'foreign_key';
//                                foreach ($index['import_join_columns'] as $ij => $ij_column) {
//                                    $ij_columns[$ij_column['local']] = [
//                                        'references' => $ij_column['foreign'],
//                                        'on'         => $index['foreign_table'],
//                                    ];
//                                }
//                            } else $type = 'foreign_id';
//
//                            foreach ($index['columns'] as $ic => $index_column) {
//                                $ii_columns[$index_column['name']] = [
//                                    'references' => $index_column['foreign_column'] ?? 'id',
//                                    'on'         => $index['foreign_table'],
//                                ];
//                                if (!array_key_exists('foreign_column', $index_column))
//                                    $this->definition['tables'][$tt]['indices'][$ii]['columns'][$ic]['foreign_column'] =
//                                        'id';
//                            }
//                        }
//                    }
//                }
//                $this->definition['tables'][$tt]['indices'][$ii]['index_type'] = $type;
//            }
//
//            foreach ($table['columns'] as $cc => $column) {
//                $type = 'column';
//                if ($column['name'] == 'id') {
//                    $type = 'primary';
//                } else {
//                    if (substr($column['name'], -3) == '_id') {
//                        $type = 'foreign_id';
////                    print_r($tt);
////                    print_r($column);
////                    print_r($ii_columns);
//
//                        $this->definition['tables'][$tt]['columns'][$cc]['references_column'] =
//                            $ii_columns[$column['name']]['references'];
//                        $this->definition['tables'][$tt]['columns'][$cc]['references_on'] =
//                            $ii_columns[$column['name']]['on'];
//                    } else {
//                        if ($column['name'] == 'date_created') {
//                            $type = 'timestamp';
//                        } else {
//                            if ($column['name'] == 'date_modified') {
//                                $type = 'timestamp';
//                            } else {
//                                if ($column['name'] == 'deleted_at') {
//                                    $type = 'deleted_flag';
//                                } else {
//                                    if (array_key_exists($column['name'], $ij_columns)) {
//                                        $type = 'foreign_key';
//                                        $this->definition['tables'][$tt]['columns'][$cc]['references_column'] =
//                                            $ij_columns[$column['name']]['references'];
//                                        $this->definition['tables'][$tt]['columns'][$cc]['references_on'] =
//                                            $ij_columns[$column['name']]['on'];
//                                    }
//                                }
//                            }
//                        }
//                    }
//                }
//                $this->definition['tables'][$tt]['columns'][$cc]['column_type'] = $type;
//            }
//        }
//        print json_encode($this->definition, JSON_PRETTY_PRINT | JSON_NUMERIC_CHECK);
//
//        return;
//    }
