<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 2/18/16
 * Time: 5:20 PM
 */

namespace Visionware\DataManager;

use Monolog\Formatter\LineFormatter;
use Monolog\Handler\MissingExtensionException;
use Monolog\Handler\SocketHandler;
use Monolog\Logger;

class VisionwareSlackHandler extends SocketHandler {
    /**
     * Slack API token
     * @var string
     */
    private $token;

    /**
     * Slack channel (encoded ID or name)
     * @var string
     */
    private $channel;

    /**
     * Name of a bot
     * @var string
     */
    private $username;

    /**
     * Emoji icon name
     * @var string
     */
    private $iconEmoji;

    /**
     * Whether the message should be added to Slack as attachment (plain text otherwise)
     * @var bool
     */
    private $useAttachment;

    /**
     * Whether the the context/extra messages added to Slack as attachments are in a short style
     * @var bool
     */
    private $useShortAttachment;

    /**
     * Whether the attachment should include context and extra data
     * @var bool
     */
    private $includeContextAndExtra;

    /**
     * @var LineFormatter
     */
    private $lineFormatter;

    /**
     * @param string      $token                  Slack API token
     * @param string      $channel                Slack channel (encoded ID or name)
     * @param string      $username               Name of a bot
     * @param bool        $useAttachment          Whether the message should be added to Slack as attachment (plain text otherwise)
     * @param string|null $iconEmoji              The emoji name to use (or null)
     * @param int         $level                  The minimum logging level at which this handler will be triggered
     * @param bool        $bubble                 Whether the messages that are handled can bubble up the stack or not
     * @param bool        $useShortAttachment     Whether the the context/extra messages added to Slack as attachments are in a short style
     * @param bool        $includeContextAndExtra Whether the attachment should include context and extra data
     */
    public function __construct($token, $channel, $username = 'Monolog', $useAttachment = true, $iconEmoji = null, $level = Logger::CRITICAL, $bubble = true)
    {
        if (!extension_loaded('openssl')) {
            throw new MissingExtensionException('The OpenSSL PHP extension is required to use the SlackHandler');
        }

        parent::__construct('ssl://slack.com:443', $level, $bubble);

        $this->token = $token;
        $this->channel = $channel;
        $this->username = $username;
        $this->iconEmoji = trim($iconEmoji, ':');
        $this->useAttachment = $useAttachment;
        $this->lineFormatter = new LineFormatter;
    }

    /**
     * {@inheritdoc}
     *
     * @param  array  $record
     * @return string
     */
    protected function generateDataStream($record)
    {
        $content = $this->buildContent($record);

        return $this->buildHeader($content) . $content;
    }

    /**
     * Builds the body of API call
     *
     * @param  array  $record
     * @return string
     */
    private function buildContent($record)
    {
        $dataArray = $this->prepareContentData($record);

        return http_build_query($dataArray);
    }

    /**
     * Prepares content data
     *
     * @param  array $record
     * @return array
     */
    protected function prepareContentData($record)
    {
        $dataArray = array(
            'token'       => $this->token,
            'channel'     => $this->channel,
            'username'    => $this->username,
            'text'        => '',
            'attachments' => array()
        );

        if ($this->useAttachment) {
            $attachment = array(
                'fallback' => $record['message'],
                'color'    => $this->getAttachmentColor($record['level']),
                'fields'   => array()
            );
            $level_map = [
                Logger::NOTICE => 'I LIKE WHAT YOU GOT',
                Logger::ERROR => 'BOO! DISQUALIFIED!!'
            ];
            $prepend_message = (array_key_exists($record['level'], $level_map)) ? $level_map[$record['level']].  "\n" : '';
            $attachment['title'] = $record['channel'] . "->" . $record['extra']['current_definition'] . '->' . $record['extra']['current_target'];
            $attachment['text'] = $prepend_message . $record['message'];
            $attachment['fields'][] = [
                'title' => 'Level',
                'value' => $record['level_name'],
                'short' => true,
            ];
            $attachment['fields'][] = [
                'title' => 'Timestamp',
                'value' => $record['datetime']->format("c"),
                'short' => true,
            ];
            $attachment['fields'][] = [
                'title' => 'Duration',
                'value' => $record['extra']['last_duration'],
                'short' => true,
            ];

            if (!empty($record['context'])) {
                $attachment['fields'][] = [
                    'title' => "Context",
                    'value' => $this->stringify($record['context']),
                    'short' => true,
                ];
            }

            $dataArray['attachments'] = json_encode(array($attachment));
        } else {
            $dataArray['text'] = $record['message'];
        }

        if ($this->iconEmoji) {
            $dataArray['icon_emoji'] = ":{$this->iconEmoji}:";
        }
        return $dataArray;
    }

    /**
     * Builds the header of the API Call
     *
     * @param  string $content
     * @return string
     */
    private function buildHeader($content)
    {
        $header = "POST /api/chat.postMessage HTTP/1.1\r\n";
        $header .= "Host: slack.com\r\n";
        $header .= "Content-Type: application/x-www-form-urlencoded\r\n";
        $header .= "Content-Length: " . strlen($content) . "\r\n";
        $header .= "\r\n";

        return $header;
    }

    /**
     * {@inheritdoc}
     *
     * @param array $record
     */
    protected function write(array $record)
    {
        parent::write($record);
        $this->closeSocket();
    }

    /**
     * Returned a Slack message attachment color associated with
     * provided level.
     *
     * @param  int    $level
     * @return string
     */
    protected function getAttachmentColor($level)
    {
        switch (true) {
            case $level >= Logger::ERROR:
                return 'danger';
            case $level >= Logger::WARNING:
                return 'warning';
            case $level >= Logger::INFO:
                return 'good';
            default:
                return '#e3e4e6';
        }
    }

    /**
     * Stringifies an array of key/value pairs to be used in attachment fields
     *
     * @param array $fields
     * @access protected
     * @return string
     */
    protected function stringify($fields)
    {
        $string = '';
        foreach ($fields as $var => $val) {
            $string .= $var.': '.$this->lineFormatter->stringify($val)." | ";
        }

        $string = rtrim($string, " |");

        return $string;
    }
}

