<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 1/27/16
 * Time: 4:40 PM
 */

namespace Visionware\DataManager;

use DB;
use Visionware\DataManager\Grammars\GrammarInterface;

class SchemaManager {
    private $ops;

    protected $new_definition;
    protected $current_definition;
    /**
     * @var GrammarInterface
     */
    protected $grammar;
    protected $name;
    protected $sql;

//    protected $valid_scopes = [];

    public function __construct(GrammarInterface $grammar, $name, $new_definition, $current_definition = false) {
        $this->grammar = $grammar;
        $this->name = $name;
        $this->new_definition = $new_definition;
        $this->current_definition = $current_definition ? $current_definition : false;
        $this->current_definition = false;
    }

    public function migrate() {
        $this->ops = [];

        if ($this->current_definition === false) {
            $this->ops[] = ['action' => 'create_schema', 'schema' => $this->name];
            $this->ops[] = ['action' => 'create_metadata'];
        }

        $current = $this->current_definition ? array_keys($this->current_definition['tables']) : [];
        $new = array_keys($this->new_definition['tables']);
        $added = array_diff($new, $current);
        $removed = array_diff($current, $new);
        foreach ($added as $table) $this->ops[] = ['action' => 'create_table', 'table' => $table];
        foreach ($removed as $table) $this->ops[] = ['action' => 'drop_table', 'table' => $table];
        foreach ($new as $table) {
            if (in_array($table, $removed)) continue;
            if (in_array($table, $added)) continue;

            $current_columns = [];
            foreach ($this->current_definition['tables'][$table]['fields'] as $field) $current_columns[] = $field['name'];
            $new_columns = [];
            foreach ($this->new_definition['tables'][$table]['fields'] as $field) $new_columns[] = $field['name'];
            $added_columns = array_diff($new_columns, $current_columns);
            $removed_columns = array_diff($current_columns, $new_columns);
            foreach ($added_columns as $col) $this->ops[] = ['action' => 'create_column', 'column' => $col, 'table' => $table];
            foreach ($removed_columns as $col) $this->ops[] = ['action' => 'drop_column', 'column' => $col, 'table' => $table];
            foreach ($new_columns as $col) {
                $current_column = $this->lookup_column($this->current_definition, $col, $table);
                $new_column = $this->lookup_column($this->new_definition, $col, $table);
                if ($current_column != $new_column) {
                    $this->ops[] = ['action' => 'drop_column', 'column' => $col, 'table' => $table];
                    $this->ops[] = ['action' => 'create_column', 'column' => $col, 'table' => $table];
                }
            }
        }

        if (count($this->ops)) $this->ops[] = ['action' => 'update_metadata'];
        $this->sql = [];
        foreach ($this->ops as $op) {
            $action = $op['action'];
            unset($op['action']);
            call_user_func_array([$this, $action], $op);
        }
        return $this->sql;
    }

    protected function create_schema($name) {
        $this->sql[] = $this->grammar->create_schema($name);
        $this->sql[] = $this->grammar->use_schema($name);
        $this->sql[] = $this->grammar->uuid_to_bin();
        $this->sql[] = $this->grammar->uuid_from_bin();
    }

    private function create_metadata() {
        $columns = [
            $this->grammar->column_definition('metadata_type', 'VARCHAR(32)'),
            $this->grammar->column_definition('metadata_key', 'VARCHAR(32)'),
            $this->grammar->column_definition('int_value', 'INT(11)', true, 'null'),
            $this->grammar->column_definition('text_value', 'VARCHAR(255)', true, 'null'),
            $this->grammar->column_definition('longtext_value', 'LONGTEXT', true, 'null'),
            $this->grammar->column_definition('date_value', 'datetime', true, 'null'),
        ];
        $indices = [
            $this->grammar->index_definition('PRIMARY KEY', '', ['metadata_type', 'metadata_key']),
        ];
        $this->sql[] = $this->grammar->create_table('datamanager_metadata', $columns, $indices);
    }

    private function update_metadata() {
        $this->sql[] = $this->grammar->replace(
            'datamanager_metadata',
            ['metadata_type', 'metadata_key', 'longtext_value'],
            [
                '"definition"',
                '"definition"',
                "'" . addslashes(json_encode($this->new_definition)) . "'"
            ]
        );
    }

    protected function create_table($table) {
        $info = $this->filter_table($table, $this->new_definition['tables'][$table]);
        if ($info === false) return;
//        if (count($this->valid_scopes) && !in_array($info['scope'], $this->valid_scopes)) return;

        $column_strings = [];
        $index_strings = [];
        if (array_key_exists('has_uuid', $info)) {
            $column_strings[] = $this->grammar->uuid_column_definition('id');
            $index_strings[] = $this->grammar->index_definition('PRIMARY KEY', '', 'id');
        }
        $imported_froms = [];
        foreach ($info['fields'] as $field) {
            $key = $field['name'];
            $field = $this->filter_column($key, $field);
            if ($field === false) continue;
            $column_strings[] = $this->grammar->column_definition($key, $field['type'], isset($field['nullable']) ? $field['nullable'] : false, isset($field['default']) ? $field['default'] : false);
            if (array_key_exists('foreign', $field)) $index_strings[] = $this->grammar->index_definition('KEY', $key, $key);
            if (array_key_exists('imported_from', $field)) $imported_froms[$field['imported_from']] = $field['imported_from'];
        }
        $column_strings[] = $this->grammar->column_definition('date_created', 'datetime', true, 'CURRENT_TIMESTAMP');
        $column_strings[] = $this->grammar->column_definition('date_modified', 'datetime', true, 'CURRENT_TIMESTAMP', 'ON UPDATE CURRENT_TIMESTAMP');
        $column_strings[] = $this->grammar->column_definition('deleted_at', 'datetime', true, 'NULL');
        if (array_key_exists('key', $info)) {
            $index_strings[] = $this->grammar->index_definition('UNIQUE KEY', 'unique_key', $info['key']);
        }
        $this->sql[] = $this->grammar->create_table($table, $column_strings, $index_strings);
    }

    private function lookup_column($definition, $column, $table) {
        $info = false;
        foreach ($definition['tables'][$table]['fields'] as $field) {
            if ($field['name'] == $column) $info = $field;
        }
        return $info;
    }

    private function create_column($column, $table) {
        $info = $this->lookup_column($this->new_definition, $column, $table);
        if (!$info) die("THIS CAN'T HAPPEN");
        $this->sql[] = $this->grammar->create_column($table, $this->grammar->column_definition($column, $info['type'], isset($info['nullable']) ? $info['nullable'] : false, isset($info['default']) ? $info['default'] : false));
    }

    private function drop_column($column, $table) {
        $this->sql[] = $this->grammar->drop_column($table, $column);
    }

//    private function alter_column($column, $table) {
//        $info = $this->lookup_column($this->new_definition, $column, $table);
//        if (!$info) die("THIS CAN'T HAPPEN");
//        $this->sql[] = $this->grammar->alter_column($table, $this->grammar->column_definition($column, $info['type'], isset($info['nullable']) ? $info['nullable'] : false, isset($info['default']) ? $info['default'] : false));
//    }

    protected function filter_table($table, $info) { return $info; }
    protected function filter_column($column, $info) { return $info; }
//    protected function filter_index($index, $table) { return $index; }
}