<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 2/29/16
 * Time: 10:39 AM
 */

namespace Visionware\DataManager\Console\Commands;

use Carbon\Carbon;
use DB;
use Illuminate\Support\Facades\Storage;

class Importer extends DataManagerCommand {
    protected $datamanager_command_name = 'datamanager:import';
    protected $datamanager_signature = '
        {--ingest : If specified, will only ingest (along with other modes that are explicitly specified)}
        {--import : If specified, will only import (along with other modes that are explicitly specified)}
        {--post : If specified, will only post-process (along with other modes that are explicitly specified)}
        {--pre : If specified, will only post-process (along with other modes that are explicitly specified)}
        {--force : Process even if database is up-to-date}
        {--skip= : Comma separated list of tables to skip (overrides --only values)}
        {--only= : Comma separated list of tables to import (will skip everything else) }
    ';
    protected $description = 'Ingests and imports data';
    private $to_ingest;
    private $to_import;

    protected function datamanager_handle() {
        $start = Carbon::now();

        $should_ingest = $this->option('ingest');
        $should_import = $this->option('import');

        if (!$should_ingest && !$should_import) {
            $should_ingest = true;
            $should_import = true;
        }
        $skip = is_null($this->option('skip')) ? [] : explode(',', $this->option('skip'));
        $only = is_null($this->option('only')) ? [] : explode(',', $this->option('only'));
        $this->to_ingest = [];
        foreach ($this->definition['tables'] as $import_table) {
            if (!array_key_exists('imported_from', $import_table)) continue;
            if (count($only) && !in_array($import_table['table'], $only)) continue;
            if (in_array($import_table['table'], $skip)) continue;
            $this->to_ingest[] = $import_table['table'];
        }

        $this->to_import = [];
        $nodes = [];
        foreach ($this->definition['tables'] as $import_table) {
            $nodes[$import_table['table']] = new Node($import_table['table']);
            $should_import = false;
            if (array_key_exists('imported_from', $import_table)) $should_import = true;
            foreach ($import_table['fields'] as $field) if (array_key_exists('imported_from', $field)) $should_import = true;
            if (!$should_import) continue;

            if (count($only) && !in_array($import_table['table'], $only)) continue;
            if (in_array($import_table['table'], $skip)) continue;

            $this->to_import[] = $import_table['table'];
        }

        $nodes['BASE'] = new Node('BASE');
        foreach ($this->to_import as $name) {
            $info = $this->getTable($name);
            foreach ($info['fields'] as $field) {
                $fname = false;
                if (array_key_exists('on_table', $field)) $fname = $field['on_table'];
                else if (array_key_exists('imported_from', $field)) $fname = $field['imported_from'];
                if ($fname && $fname != $name) $nodes[$name]->dependsOn($nodes[$fname]);
            }
            $nodes['BASE']->dependsOn($nodes[$name]);
        }
        $order = [];
        $nodes['BASE']->resolve($order);
        foreach ($order as $k=>$v) if ($v == 'BASE') unset($order[$k]);
        $this->to_import = $order;

        if ($should_ingest) $this->ingest();
        if ($should_import) $this->import();

        $this->notice("Process complete in " . Carbon::now()->diffForHumans($start, true) . " total\n");
    }

    private function getTable($name) {
        foreach ($this->definition['tables'] as $table) if ($table['table'] == $name) return $table;
    }

    protected function ingest() {
        $start = Carbon::now();
        foreach ($this->to_ingest as $ingest) {
            $this->current_target = $ingest;
            $this->ingest_file($ingest);
        }
        $this->info("Ingestion complete in " . Carbon::now()->diffForHumans($start, true));
    }

    private function ingest_file($table_name) {
        $start = Carbon::now();

        $this->current_target = $table_name;
        $current_definition = $this->definition['tables'][$table_name];
        $db = DB::connection($this->config['history-database-connection']);
        $import_files = [$current_definition['imported_from']];

        try {
            $this->info("Ingesting " . count($import_files) . " file(s) for table $table_name");
            $snapshot_files = [];
            $new_files_count = 0;
            foreach ($import_files as $import_file_name) {
                $local_path = sys_get_temp_dir() . "/$import_file_name";
                $remote_path = $this->config['import_source.prepend_path'] . $import_file_name;
                $disk = Storage::disk($this->config['import_source.disk']);
                if (!$disk->exists($remote_path)) {
                    $this->error("File $remote_path not found, skipping...");
                    continue;
                }

                $this->info("Checking last modified date for $import_file_name...");
                $last_modified_ut = $disk->lastModified($remote_path);
                $last_modified = date('Y-m-d H:i:s', $last_modified_ut);

//                $db_last_modified = $db->selectOne("SELECT MAX(last_modified) lm FROM {$table_name}_files")->lm;
                $db_last_modified = $db->table($table_name . '_files')->where('file_name', '=', $import_file_name)->max(
                    'last_modified'
                )
                ;

                if (!$this->option('force') && $last_modified <= $db_last_modified) {
                    $file_id = $db->table("{$table_name}_files")->where('file_name', '=', $import_file_name)->where('last_modified', '=', $last_modified)->value('id');
                    $snapshot_files[$file_id] = $last_modified;
                    $this->info("File is up-to-date, skipping");
                    continue;
                }

                $this->info("Downloading file $import_file_name...");
                if (false === @file_put_contents($local_path, $disk->get($remote_path))) {
                    $this->error("Error while downloading $remote_path, skipping...", ['error' => error_get_last()]);
                    continue;
                }

                $load_file_path = $local_path . '.load';

                $local_file_object = new \SplFileObject($local_path, 'r');
                if ($local_file_object === false) {
                    $this->error("Unable to open $local_path");
                    continue;
                }
//            $local_file_object->setCsvControl("\t", '');
                $local_file_object->setFlags(
                    \SplFileObject::READ_AHEAD
                    | \SplFileObject::SKIP_EMPTY
                    | \SplFileObject::DROP_NEW_LINE
                );
                $local_file_object->seek(PHP_INT_MAX);
                $line_count = $local_file_object->key();
                $this->info("Opening file $local_path ($line_count lines)...");
                $local_file_object->rewind();

                $source_fields = explode("\t", $local_file_object->current());
                $local_file_object->next();

                $destination_columns = [];
                $field_column_map = [];
                foreach ($current_definition['fields'] as $field) {
                    if (!array_key_exists('import_field', $field)) continue;
                    $destination_columns[] = $field['name'];
                    $field_column_map[$field['import_field']] = $field['name'];
                }

                if (count($source_fields) != count($destination_columns)) {
                    $this->error(
                        "Import file field count does not match definition!",
                        ['source_fields' => $source_fields, 'destination_columns' => $destination_columns]
                    );
                    continue;
                }

                $values = [];
                $this->info("Building LOAD DATA file...");
                $load_file_object = new \SplFileObject($load_file_path, 'w');
                $current_line = 0;
                while (!$local_file_object->eof()) {
                    $output_row = [];
                    $raw = explode("\t", $local_file_object->current());
                    if ($values === false) return; //return on error
                    $input_row = @array_combine($source_fields, $raw);
                    if ($input_row !== false) {
                        foreach ($field_column_map as $field => $column) {
                            $output_row[$column] = str_replace([',', '"'], ['\\,', '\\"'], trim($input_row[$field]));
                            if ($output_row[$column] == '') $output_row[$column] = '\N';
                        }
                        $output_row['record_hash'] = md5(implode('', $output_row));
                        $output_row['file_name'] = $import_file_name;
                        $output_row['last_modified'] = $last_modified;
                        $output_row['sequence'] = $current_line;

                        $load_file_object->fwrite(implode(',', $output_row) . "\n");
                    } else {
                        $this->warn("Line $current_line has invalid field count, skipping line.", ['raw_line' => $raw]);
                    }

                    $local_file_object->next();
                    $current_line++;
                }
                $load_file_object->fflush();
                $columns_string = implode(', ', array_keys($output_row));

                $this->info("Truncating staging table...");
                $db->table("{$table_name}_staging")->truncate();

                $db->beginTransaction();

                $sql =
                    "LOAD DATA LOCAL INFILE '$load_file_path' IGNORE INTO TABLE `{$table_name}_staging` FIELDS TERMINATED BY ',' ($columns_string)";
                $this->info("Loading data into staging table...");
                $this->debug($sql);
                $db->getPdo()->exec($sql);
                $columns_string = 'record_hash, ' . implode(', ', $destination_columns);
                $this->info("Updating records table from staging...");
                $sql =
                    "INSERT IGNORE INTO `{$table_name}_records` ($columns_string) SELECT $columns_string FROM `{$table_name}_staging`";
                $this->debug($sql);
                $db->statement($sql);

                $this->info("Updating files table from staging...");
                $sql =
                    "INSERT IGNORE INTO `{$table_name}_files` (file_name, last_modified) SELECT s.file_name, s.last_modified FROM `{$table_name}_staging` s LIMIT 1";
                $this->debug($sql);
                $db->statement($sql);

                $file_id = $db->table("{$table_name}_files")->where('file_name', '=', $import_file_name)->where('last_modified', '=', $last_modified)->value('id');

                if ($this->option('force')) {
                    $this->info("Deleting old file_record rows...");
                    $sql = "DELETE FROM `{$table_name}_file_record` where file_id = $file_id";
                    $this->debug($sql);
                    $db->statement($sql);
                }

                $this->info("Updating file_record table from staging...");
                $sql =
                    "INSERT IGNORE INTO `{$table_name}_file_record` (file_id, record_hash, sequence) SELECT f.id, s.record_hash, s.sequence FROM `{$table_name}_staging` s JOIN `{$table_name}_files` f USING (last_modified)";
                $this->debug($sql);
                $db->statement($sql);
                $db->commit();

                $snapshot_files[$file_id] = $last_modified;
                $new_files_count++;
                $this->notice("Ingested file $local_path");
            }

            if ($new_files_count) {
                $snapshot_last_modified = max(array_values($snapshot_files));
                $this->info("Updating snapshots table from staging...");
                $sql = "DELETE FROM `{$table_name}_snapshots` WHERE last_modified = \"$snapshot_last_modified\"";
                $this->debug($sql);
                $db->statement($sql);
                foreach ($snapshot_files as $file_id => $nothing) {
                    $sql = "REPLACE INTO `{$table_name}_snapshots` (last_modified, file_id) VALUES (\"$snapshot_last_modified\", $file_id)";
                    $this->debug($sql);
                    $db->statement($sql);
                }
            }
        } catch (\PDOException $e) {
            $this->error('Exception while importing file! Skipping...', ['exception' => $e]);
            return;
        }
        $this->notice("Imported table $local_path in " . Carbon::now()->diffForHumans($start, true) . " total\n");
    }

    protected function import() {
        $start = Carbon::now();

        $to_import_count = count($this->to_import);
        $imported_count = 0;
        foreach ($this->to_import as $table_name) {
            $tbl_start = Carbon::now();
            $imported_count++;
            $this->current_target = $table_name;
            $this->info("Importing table ($imported_count of $to_import_count)...");
            if (array_key_exists('imported_from', $this->definition['tables'][$table_name])) $this->import_table($table_name);
            else $this->summarize_table($table_name);
            $this->notice("Completed importing table in " . Carbon::now()->diffForHumans($tbl_start, true));

        }
        $this->info("Import complete in " . Carbon::now()->diffForHumans($start, true));
    }

//    private function xsummarize_table($table_name) {
//        $definition = $this->definition['tables'][$table_name];
//
//        $history = DB::connection($this->config['history-database-connection'])->getDatabaseName();
//        $erpfeed = DB::connection($this->config['erpfeed-database-connection'])->getDatabaseName();
//        $db = DB::connection($this->config['erpfeed-database-connection']);
//
//        $destination_table = $erpfeed . '.' . $table_name;
//        $history_table = $history . '.' . $table_name . '_latest';
//        $source_table = false;
//        $insert_columns = [];
//        $select_columns = [];
////        $foreigners = [];
//        $joins = [];
//        foreach ($definition['fields'] as $field) {
//            if (array_key_exists('foreign', $field)) {
////                $foreigners[$field['name']] = $field;
//                $joins[$field['on_table']] = "LEFT JOIN $foreign_table_base "
//            }
//
//            if (array_key_exists('imported_from', $field)) $source_table = $field['imported_from'];
//            $insert_columns = $field['name'];
//
//            if (array_key_exists('foreign', $field)) {
////                $select_columns[] = $table_name_mapping
//            }
//        }
//    }

    private function summarize_table($table_name) {
        /** 1) Get most recent data from history table and load it into the staging table
         *  2) Populate the staging table with foreign key ID values by using the key fields defined
         *  3) Update the live tables with the filled-in staging table values
         */
        $definition = $this->definition['tables'][$table_name];

        $history = DB::connection($this->config['history-database-connection'])->getDatabaseName();
        $erpfeed = DB::connection($this->config['erpfeed-database-connection'])->getDatabaseName();
        $db = DB::connection($this->config['erpfeed-database-connection']);

        $destination_table = $erpfeed . '.' . $table_name;
        $source_table_name = false;
        foreach ($definition['fields'] as $field) if (array_key_exists('imported_from', $field)) $source_table_name = $field['imported_from'];

        $source_table = $history . '.' . $source_table_name . "_latest";
        $get_date_table = $history . '.' . $source_table_name . "_files";

        $source_last_modified = $db->table($get_date_table)->max('last_modified');
        $last_updated =
            $db->table('datamanager_metadata')
                ->where('metadata_type', '=', 'last_updated')
                ->where('metadata_key', '=', $table_name)
                ->value('date_value')
        ;
        if (!$this->option('force') && $last_updated >= $source_last_modified) {
            $this->info(
                "Table contains most recent data already, skipping",
                ['last_modified' => $source_last_modified, 'last_updated' => $last_updated]
            );
            return;
        }

        $unique_key_columns = $definition['key'];
        $unique_key_columns_string = implode(', ', $unique_key_columns);

        $iterations = 1;

        $foreigners = [];
        foreach ($definition['fields'] as $field) {
            $letter = "A";
            if (array_key_exists('extra_joins', $field)) {
                foreach ($field['extra_joins'] as $extra_join) {
                    $fake = ['name' => $field['name']];
                    if (array_key_exists('imported_from', $extra_join)) $fake['imported_from'] = $extra_join['imported_from'];
                    if (array_key_exists('on_table', $extra_join)) $fake['on_table'] = $extra_join['on_table'];
                    if (array_key_exists('using', $extra_join)) $fake['using'] = $extra_join['using'];
                    if (array_key_exists('join_on', $extra_join)) $fake['join_on'] = $extra_join['join_on'];
                    $foreigners[$field['name'] . $letter++] = $fake;
                }
            }
            if (array_key_exists('foreign', $field)) {
                $foreigners[$field['name']] = $field;
                if ($field['on_table'] == $table_name) $iterations++;
            }
        }

        $joins = [];
        $joins[$table_name] = "LEFT OUTER JOIN $destination_table USING ($unique_key_columns_string)";
        $table_name_mapping = [];
        foreach ($foreigners as $index_key => $index) {
            if (array_key_exists('on_table', $index)) {
                $foreign_table_base = "$erpfeed.$index[on_table]";
                $foreign_table = $index['on_table'];
            }
            else if (array_key_exists('imported_from', $index)) {
                $foreign_table_base = "$history.$index[imported_from]_latest";
                $foreign_table = $index['imported_from'];
            }
            if (array_key_exists('join_on', $index)) {
                $join_ons = [];
                foreach ($index['join_on'] as $import_join_column) {
                    $join_ons[] = $foreign_table
                        . '.'
                        . $import_join_column['foreign']
                        . ' = '
                        . $source_table
                        . '.'
                        . $import_join_column['local'];
                }
                $join_ons_string = implode(' AND ', $join_ons);

                $joins[$foreign_table] = "LEFT JOIN $foreign_table_base $foreign_table ON ($join_ons_string)";
            } else if (array_key_exists('using', $index)) {
                $joins[$foreign_table] = "LEFT JOIN $foreign_table_base $foreign_table USING ($index[using])";
            }
            $table_name_mapping[$index_key] = $foreign_table;
        }
        $joins_string = implode(" ", $joins);

        $destination_columns = ['id'];
        $select_columns = [];
        if (array_key_exists('has_uuid', $definition) && $definition['has_uuid']) {
            $select_columns[] = "COALESCE($destination_table.id, UuidToBin(UUID()))";
        }
        $destination_updates = [];
        foreach ($definition['fields'] as $column) {
            $column_key = $column['name'];
            if (array_key_exists('foreign', $column)) {
                $select_columns[] = $table_name_mapping[$column_key] . ".$column[foreign] $column[name]";
            } else {
                $select_columns[] = $source_table . '.' . $column['name'];
            }

            $destination_columns[] = $column['name'];
            $destination_updates[] = "$column[name]=VALUES($column[name])";
        }
        $destination_updates[] = "deleted_at=NULL";

        $select_columns_string = implode(', ', $select_columns);
        $destination_columns_string = implode(', ', $destination_columns);
        $destination_updates_string = implode(', ', $destination_updates);

        try {
            $db->beginTransaction();

            $this->info(sprintf("Updating live table %s (%d %s)...", $destination_table, $iterations, str_plural('iteration', $iterations)));
            while ($iterations > 0) {
                $this->info('Inserting into live...');
                $sql =
                    "INSERT IGNORE INTO $destination_table ($destination_columns_string) SELECT $select_columns_string FROM $source_table $joins_string ON DUPLICATE KEY UPDATE $destination_updates_string;";
                $this->debug($sql);
                $db->affectingStatement($sql);

                $this->info('Updating deleted records...');
                $sql =
                    "UPDATE $destination_table LEFT OUTER JOIN $source_table USING ($unique_key_columns_string) SET $destination_table.deleted_at = NOW() WHERE $source_table.record_hash IS NULL AND $destination_table.deleted_at IS NULL;";
                $this->debug($sql);
                $db->affectingStatement($sql);
                $iterations--;
            }

            $this->info('Updating last modified metadata record...');
            $sql =
                "REPLACE INTO datamanager_metadata (`metadata_type`, `metadata_key`, `date_value`) VALUES ('last_updated', '$table_name', '$source_last_modified')";
            $this->debug($sql);
            $db->affectingStatement($sql);

            $this->info("Committing...");
            $db->commit();

        } catch (Exception $e) {
            $db->rollBack();
            $this->error('Exception caught', ['exception' => $e]);

            return;
        }
    }


    private function import_table($table_name) {
        /** 1) Get most recent data from history table and load it into the staging table
         *  2) Populate the staging table with foreign key ID values by using the key fields defined
         *  3) Update the live tables with the filled-in staging table values
         */
        $definition = $this->definition['tables'][$table_name];

        $history = DB::connection($this->config['history-database-connection'])->getDatabaseName();
        $live = DB::connection($this->config['erpfeed-database-connection'])->getDatabaseName();
        $db = DB::connection($this->config['erpfeed-database-connection']);

        $source_table = $history . '.' . $table_name . "_latest";
        $get_date_table = $history . '.' . $table_name . "_files";
        $live_table = $live . '.' . $table_name;

        $source_last_modified = $db->table($get_date_table)->max('last_modified');
        $last_updated =
            $db->table('datamanager_metadata')
                ->where('metadata_type', '=', 'last_updated')
                ->where('metadata_key', '=', $table_name)
                ->value('date_value')
        ;
        if (!$this->option('force') && $last_updated >= $source_last_modified) {
            $this->info(
                "Table contains most recent data already, skipping",
                ['last_modified' => $source_last_modified, 'last_updated' => $last_updated]
            );
            return;
        }

        $unique_key_columns = $definition['key'];
        $unique_key_columns_string = implode(', ', $unique_key_columns);

        $iterations = 1;

        $foreigners = [];
        foreach ($definition['fields'] as $field) {
            if (array_key_exists('foreign', $field)) {
                $foreigners[$field['name']] = $field;
                if ($field['on_table'] == $table_name) $iterations++;
            }
        }

        $joins = [];
        $joins[$table_name] = "LEFT OUTER JOIN $live_table USING ($unique_key_columns_string)";
        $table_name_mapping = [];
        $letter = "A";
        foreach ($foreigners as $index_key => $index) {
            $foreign_table_base = "$live.$index[on_table]";
            $foreign_table = $index['on_table'];
            while (array_key_exists($foreign_table, $joins)) {
                $foreign_table =
                    $index['on_table'] . $letter++;
            }
            $join_ons = [];
            foreach ($index['join_on'] as $import_join_column) {
                $join_ons[] = $foreign_table
                    . '.'
                    . $import_join_column['foreign']
                    . ' = '
                    . $source_table
                    . '.'
                    . $import_join_column['local'];
                $table_name_mapping[$index_key] = $foreign_table;
            }
            $join_ons_string = implode(' AND ', $join_ons);

            $joins[$foreign_table] = "LEFT OUTER JOIN $foreign_table_base $foreign_table ON ($join_ons_string)";
        }
        $joins_string = implode(" ", $joins);

        $live_columns = ['id'];
        $select_columns = [];
        if (array_key_exists('has_uuid', $definition) && $definition['has_uuid']) {
            $select_columns[] = "COALESCE($live_table.id, UuidToBin(UUID()))";
        }
        $live_updates = [];
        foreach ($definition['fields'] as $column) {
            $column_key = $column['name'];
            if (array_key_exists('foreign', $column)) {
                $select_columns[] = $table_name_mapping[$column_key] . ".$column[foreign] $column[name]";
            } else {
                $select_columns[] = $this->import_transform($column, $source_table) . " $column[name]";
            }

            $live_columns[] = $column['name'];
            $live_updates[] = "$column[name]=VALUES($column[name])";
        }
        $live_updates[] = "deleted_at=NULL";

        $select_columns_string = implode(', ', $select_columns);
        $live_columns_string = implode(', ', $live_columns);
        $live_updates_string = implode(', ', $live_updates);

        try {
            $db->beginTransaction();

            $this->info(sprintf("Updating live table %s (%d %s)...", $live_table, $iterations, str_plural('iteration', $iterations)));
            while ($iterations > 0) {
                $this->info('Inserting into live...');
                $sql =
                    "INSERT IGNORE INTO $live_table ($live_columns_string) SELECT $select_columns_string FROM $source_table $joins_string ON DUPLICATE KEY UPDATE $live_updates_string;";
                $this->debug($sql);
                $db->affectingStatement($sql);

                $this->info('Updating deleted records...');
                $sql =
                    "UPDATE $live_table LEFT OUTER JOIN $source_table USING ($unique_key_columns_string) SET $live_table.deleted_at = NOW() WHERE $source_table.record_hash IS NULL AND $live_table.deleted_at IS NULL;";
                $this->debug($sql);
                $db->affectingStatement($sql);
                $iterations--;
            }

            $this->info('Updating last modified metadata record...');
            $sql =
                "REPLACE INTO datamanager_metadata (`metadata_type`, `metadata_key`, `date_value`) VALUES ('last_updated', '$table_name', '$source_last_modified')";
            $this->debug($sql);
            $db->affectingStatement($sql);

            $this->info("Committing...");
            $db->commit();

        } catch (Exception $e) {
            $db->rollBack();
            $this->error('Exception caught', ['exception' => $e]);

            return;
        }
    }

    private function import_transform($column, $table) {
        if (!array_key_exists('transformation', $column)) return "$table.$column[name]";

        return str_replace('$$$', "$table.$column[name]", $column['transformation']);
    }
}