<?php

namespace Visionware\DataManager\Console\Commands;

use Config;
use DataManager;
use DB;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Storage;
use Visionware\DataManager\Grammars\MysqlGrammar;
use Visionware\DataManager\Grammars\SqliteEmptyGrammar;
use Visionware\DataManager\Schema\MySqlSchemaInfo;
use Visionware\DataManager\Schema\SqliteSchemaInfo;
use Visionware\DataManager\SchemaManager;

class CreateSchemaCommand extends Command {
    protected $signature = 'datamanager:create
        {--output= : Output to a file instead of stdout.}
        {--mode= : Creation mode history, live, import, historyimport, sqlite, all-history (history, historyimport), all-live (live, import) }
        {--recreate : Includes drop database statements}
    ';

    protected $description = 'Builds create statements';

    public function handle() {
        
        if ($this->option('mode') == 'sqlite' && is_null($this->option('output'))) {
            $this->error("You must specify an output file when using SQLite mode!");
            return false;
        }

        $modes = $this->option('mode');
        if ($this->option('mode') == 'all-history') $modes = ['history', 'historyimport'];
        if ($this->option('mode') == 'all-live') $modes = ['live', 'import'];
        if (!is_array($modes)) $modes = [$modes];

        if (!is_array($modes)) $modes = [$modes];
        $buffer = [];
        $logger = DataManager::getLogger('creator', $this->getOutput());
        foreach ($modes as $mode) {
            if ($mode == 'sqlite') {

                $grammar = new SqliteEmptyGrammar();
            } else {
                $grammar = new MysqlGrammar();
            }

            $newStatements = [];
            $new = DataManager::getSchemaInfo($mode);
            $schemaName = DataManager::getDbNameForMode($mode);
            if ($this->option('recreate') || $mode == 'sqlite') {
                $newStatements[] = $grammar->dropSchema($schemaName);
                $newStatements = array_merge($newStatements, $this->compileCreateSchema($grammar, $schemaName));
                $newStatements = array_merge($newStatements, $new->toSql($grammar)->toArray());
            } else {
                try {
                    $current = DataManager::getSqlSchemaInfo($mode);
                } catch (Exception $e) {
                    $logger->warn("Exception caught while trying to get current schema. Proceeding with new schema creation.", [$e]);
                    $newStatements = $this->compileCreateSchema($grammar, $schemaName);
                }
                $newStatements = array_merge($newStatements, $new->migrateFrom($current, $grammar, $logger)->toArray());
            }
            if (count($newStatements)) {
                $newStatements = array_merge([$grammar->useSchema($schemaName)], $newStatements);
            }
            $buffer = array_merge($buffer, $newStatements);
        }

        if (!count($buffer)) {
            $logger->notice('No changes are necessary.');
        } else {
            if ($mode == 'sqlite') {
                $path = storage_path() . '/database.sqlite';
                touch($path);
                $db = DB::connection('sqlite');

//                $db->beginTransaction();
                foreach ($buffer as $sql) {
                    $sql = trim($sql);
                    if (!strlen($sql)) continue;
                    $sqls = explode("\n", $sql);
                    foreach ($sqls as $stmt) $db->statement(DB::raw($stmt));
                }
            }
            if ($this->option('output')) {
                file_put_contents($this->option('output'), implode("\n\n", $buffer));
                $logger->notice('Statements written out to file');
            } else {
                print implode("\n\n", $buffer);
            }
        }
    }

    private function compileCreateSchema($grammar, $name) {
        return [
            $grammar->createSchema($name),
            $grammar->useSchema($name),
            $grammar->createAlphaNumFunction(),
            $grammar->createUuidFromBinFunction(),
            $grammar->createUuidToBinFunction(),
        ];
    }
}