<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/13/16
 * Time: 10:16 AM
 */

namespace Visionware\DataManager;

use DB;
use Exception;
use Illuminate\Support\Collection;
use Monolog\Logger;
use Visionware\DataManager\Definition\TableDefinition;
use Visionware\DataManager\Facades\DataManager;
use Carbon\Carbon;
use Illuminate\Database\Connection;

/**
 * Class Transferer
 * @package Visionware\DataManager
 * @method emergency(string $msg, array $context = [])
 * @method alert(string $msg, array $context = [])
 * @method critical(string $msg, array $context = [])
 * @method error(string $msg, array $context = [])
 * @method warn(string $msg, array $context = [])
 * @method notice(string $msg, array $context = [])
 * @method info(string $msg, array $context = [])
 */
class Transferer extends DataManagerProcess {
    /** @var Connection */
    protected $importDb;
    /** @var Connection */
    protected $historyDb;
    protected $noTransactions;
    protected $only;


    public function __construct($schema) {
        parent::__construct($schema);
        $this->importDb = DataManager::getImportConnection();
        $this->historyDb = DataManager::getHistoryConnection();
        $this->noTransactions = false;
        $this->only = false;
    }

    public function go() {
        $allStart = Carbon::now();
        try {
            foreach ($this->schema->tablesToIngest() as $tableName) {
                if ($this->only && !($tableName == $this->only)) continue;
                $this->setTable($tableName->name());
                $latestTable = "{$tableName}_latest";
                $start = Carbon::now();

                $this->notice("Transferring table $tableName from history to import...");

                $historyLastModified = $this->historyDb->table($latestTable)->max('date_modified');
                $importLastModified = $this->importDb->table($latestTable)->max('date_modified');

                if ($historyLastModified <= $importLastModified && !$this->force) {
                    $this->notice("Table $tableName is up to date, skipping");
                    continue;
                }
                
                $columns = new Collection([
                    'record_hash' => 'record_hash'
                ]);
                foreach ($this->definition->getImportFields()->keys() as $columnName) {
                    $columns->put($columnName, $columnName);
                };
                $columns->put('date_modified', 'date_modified');
                $this->info("Transferring columns: " . $columns->keys()->implode(', '));

                $this->importDb->table($latestTable)->truncate();
                if (!$this->noTransactions) $this->importDb->beginTransaction();
                $query = $this->historyDb
                    ->table($latestTable)
                    ->select($columns->values()->toArray())
                    ->orderBy('record_hash')
                ;
                if (!$this->force && $importLastModified) $query->where('date_modified', '>', $importLastModified);
                $chunkSize = env('TRANSFER_CHUNK_SIZE', 10000);
                $query->chunk($chunkSize, function($rows) use ($latestTable, $columns, $chunkSize) {
                        $values = [];
                        foreach ($rows as $data) {
                            $row = (array)$data;
                            foreach ($row as $key=>$value) $row[$key] = $this->importDb->getPdo()->quote(trim($value));
                            $values[] = implode(', ', $row);
                        }
                        $valuesString = implode('), (', $values);
                        $sql = "INSERT INTO $latestTable (" . $columns->keys()->implode(', ') . ") VALUES ($valuesString);";
//                        $this->debug($sql);
                        $this->info("Inserting $chunkSize rows...");
                        $this->importDb->statement($sql);
                    })
                ;
                if (!$this->noTransactions) $this->importDb->commit();
                $end = Carbon::now();
                $diff = $end->diffForHumans($start, true);
                $this->notice("Transferred table " . $this->definition->name() . " in $diff total");
            }
        } catch (\Exception $e) {
            $this->error("Caught exception while transferring table! Rolling back...", ['exception' => $e]);
            if (!$this->noTransactions) $this->importDb->rollBack();
        }
        $diff = Carbon::now()->diffForHumans($allStart, true);
        $this->notice("Finished transfer process in $diff total");
    }

    public function noTransactions() {
        $this->noTransactions = true;
    }

    public function only($tableName) {
        $this->only = $tableName;
    }

    public function transferTable() {

    }

}