<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/13/16
 * Time: 10:16 AM
 */

namespace Visionware\DataManager;

use AWS;
use Closure;
use Visionware\DataManager\Definition\TableDefinition;
use Visionware\DataManager\Facades\DataManager;
use Carbon\Carbon;
use Illuminate\Database\Connection;
use Visionware\DataManager\Exceptions\IngestionFieldsMismatchException;
use Visionware\DataManager\Exceptions\RemoteFileDownloadException;
use Visionware\DataManager\Exceptions\RemoteFileNotFoundException;
use Visionware\DataManager\Exceptions\UnableToCreateLoadFileException;
use Visionware\DataManager\Info\SchemaInfo;
use Visionware\DataManager\Info\TableInfo;

class Ingester extends DataManagerProcess {
    /**
     * @var Connection
     */
    protected $db;
//    protected $snapshotFiles;
//    protected $newFilesCount;
    protected $allVersions;
    protected $currentFile;
    protected $s3;
    protected $skipTransactions;
    protected $newerThan;

    public function __construct(SchemaInfo $schema) {
        parent::__construct($schema);
        $this->db = DataManager::getHistoryConnection();
//        $this->snapshotFiles = [];
//        $this->newFilesCount = 0;
        $this->allVersions = false;
        $this->skipTransactions = false;
        $this->newerThan = 0;
    }

    public function allVersions() {
        $this->allVersions = true;
    }

    public function noTransactions() {
        $this->skipTransactions = true;
    }

    public function newerThan($date) {
        $this->newerThan = trim($date);
    }

    public function go() {
        $start = Carbon::now();
        $this->s3 = AWS::createClient('s3');


        $import_files = $this->definition->getIngestFiles();
        $this->notice("Starting to ingest table $this->tableName");
        $this->updateStartTime($this->tableName, $start);
        try {
            foreach ($import_files as $fileName) {
                $this->info("Starting to ingest $fileName");

                $this->ingestFileVersions($fileName);
                $this->info("Finished ingesting $fileName");
            }

//            if ($this->newFilesCount) {
//                $this->info("Updating snapshots table from staging and replacing latest...");
//                $this->updateSnapshots();
//            }
        } catch (\Exception $e) {
            $this->emergency("Caught exception while ingesting file $fileName for table {$this->tableName}! Rolling back...", ['exception' => $e]);
            if (!$this->skipTransactions) $this->db->rollBack();
            return;
        }

        $end = Carbon::now();
        $diff = $end->diffForHumans($start, true);
        $this->updateEndTime($this->tableName, $end, $end->diffInSeconds($start, true), $diff);
        $this->notice("Finished ingesting table " . $this->tableName . " in $diff total");
    }

//    private function getDbLastModified($fileName) {
//        $dbLastModified = $this->db->table($this->tableName . '_files')->where('file_name', '=', $fileName)->max(
//            'last_modified'
//        );
//        return $dbLastModified;
//    }

//    private function getLastModified($fileName) {
//        $disk = DataManager::getImportSourceDisk();
//        $path = DataManager::getImportSourcePath() . $fileName;
//        if (!$disk->exists($path)) {
//            throw new RemoteFileNotFoundException("File $path not found, skipping...");
//        }
//        $lastModifiedUt = $disk->lastModified($path);
//        return date('Y-m-d H:i:s', $lastModifiedUt);
//    }

    private function getIngestedVersions($fileName) {
        $versions = $this->db->table($this->tableName . '_files')
            ->where('file_name', $fileName)
            ->pluck('version_id')
        ;
        return $versions;
    }

    private function ingestFileVersions($fileName) {
        $ingestedVersions = $this->getIngestedVersions($fileName);
        $result = $this->s3->listObjectVersions([
            'Bucket' => env('AWS_S3_IMPORT_BUCKET'),
            'Prefix' => $fileName,
        ]);

        $versions = $result['Versions'];
        if ($this->allVersions) {
            usort($versions, function($a, $b) {
                if ($a['LastModified'] > $b['LastModified']) return 1;
                else if ($a['LastModified'] == $b['LastModified']) return -1;
                return 0;
            });
        }


        $updateLatest = false;

        foreach ($versions as $version) {
            if ($version['VersionId'] == 'null') continue;
            if ($version['LastModified']->format('Y-m-d') <= $this->newerThan) continue;
            $this->currentFile = $version;

            if ($this->force || !in_array($version['VersionId'], $ingestedVersions)) {
                $this->notice("Ingesting file {$this->currentFile['Key']} version {$this->currentFile['LastModified']->format('Y-m-d H:i:s')}...");
                $return = $this->ingestFile();
                if ($return !== false) $updateLatest = true;
            } else {
                $this->info("File {$this->currentFile['Key']} version {$this->currentFile['LastModified']->format('Y-m-d H:i:s')} has already been ingested, skipping...");
            }
            if (!$this->allVersions) break;
        }

        if ($updateLatest) {
            if (!$this->skipTransactions) $this->db->beginTransaction();
            $this->info("Updating latest table...");
            $this->updateLatest();
            if (!$this->skipTransactions) $this->db->commit();
        }
    }

    private function ingestFile() {
        $fileName = $this->currentFile['Key'];
        $lastModified = $this->currentFile['LastModified']->format('Y-m-d H:i:s');
//        $dbLastModified = $this->getDbLastModified($fileName);
//        $isModified = !($lastModified <= $dbLastModified);
//        if (!$isModified && !$this->force) {
//            $fileId = $this->db->table("{$this->tableName}_files")->where('file_name', '=', $fileName)->where('last_modified', '=', $lastModified)->value('id');
//            $this->snapshotFiles[$this->currentFile['Key']][$lastModified][$fileId] = $fileId;
//            $this->info("File is up-to-date, skipping");
//            return;
//        }
        $this->info("Downloading file...");
        $localPath = $this->downloadFile($fileName);
        $this->info("Building LOAD DATA file...");
        $destinationColumns = $this->getDestinationColumns();
//        $fieldColumnMap = $this->getFieldColumnMap();
        $loadFilePath = $localPath . '.load';
        $loadColumns = $this->buildLoadFile($localPath, $loadFilePath, $fileName, $lastModified);
        if ($loadColumns === false) return false;

        $this->info("Truncating staging table...");
        $this->db->table("{$this->tableName}_staging")->truncate();

        if (!$this->skipTransactions) $this->db->beginTransaction();

        $this->info("Loading data into staging table...");
        $this->loadData($loadFilePath, $loadColumns);

        $this->info("Updating records table from staging...");
        $this->updateRecords($destinationColumns);

        $this->info("Updating files table from staging...");
        $this->updateFiles();

        $fileId = $this->db->table("{$this->tableName}_files")->where('file_name', '=', $fileName)->where('modified_time', '=', $lastModified)->value('id');

//        if ($this->force) {
            $this->info("Deleting old file_record rows...");
            $this->deleteOldRecords($fileId);
//        }

        $this->info("Updating file_record table from staging...");
        $this->updateFileRecords();

        if (!$this->skipTransactions) $this->db->commit();

//        $this->snapshotFiles[$this->currentFile['Key']][$lastModified][$fileId] = $fileId;
//        $this->newFilesCount++;
        return true;
    }

    private function getDestinationColumns() {
        $destination_columns = [];
        foreach ($this->definition->fields() as $field) {
            if (!$field->hasImportField()) continue;
            $destination_columns[] = $field->name();
        }
        return $destination_columns;
    }

    private function downloadFile() {
        $localPath = sys_get_temp_dir() . DIRECTORY_SEPARATOR . $this->currentFile['Key'];

        $this->info("Downloading file {$this->currentFile['Key']} version {$this->currentFile['LastModified']->format('Y-m-d H:i:s')}...");
//        $disk = DataManager::getImportSourceDisk();
//        $remotePath = DataManager::getImportSourcePath() . $fileName;
//        if (false === @file_put_contents($localPath, $disk->get($remotePath))) {
//            throw new RemoteFileDownloadException("Error " . error_get_last() . " while downloading $remotePath, skipping...");
//        }
        $s3 = AWS::createClient('s3');
        $s3->getObject([
            'Bucket' => env('AWS_S3_IMPORT_BUCKET'),
            'Key' => $this->currentFile['Key'],
            'VersionId' => $this->currentFile['VersionId'],
            'SaveAs' => $localPath,
        ]);
        return $localPath;
    }

    private function buildLoadFile($localPath, &$loadFilePath, $fileName, $lastModified) {
        $localFileObject = new \SplFileObject($localPath, 'r');
        if ($localFileObject === false) {
            throw new UnableToCreateLoadFileException("Unable to open $localPath");
        }

        $localFileObject->setFlags(
            \SplFileObject::READ_AHEAD
            | \SplFileObject::SKIP_EMPTY
            | \SplFileObject::DROP_NEW_LINE
        );
        
        $localFileObject->seek(PHP_INT_MAX);
        $localFileObject->rewind();

        $sourceFields = explode("\t", $localFileObject->current());
        $localFileObject->next();

        $fieldColumnMap = $this->definition->getFieldColumnMap();
        $destinationColumns = array_values($fieldColumnMap);

        foreach ($fieldColumnMap as $src => $dest) {
            if (!in_array($src, $sourceFields)) {
                $this->emergency("Import file does not contain field $src", ['source_fields' => $sourceFields, 'field_column_map' => $fieldColumnMap]);
                if (!$this->allVersions) throw new IngestionFieldsMismatchException("Import file does not contain field $src");
                return false;
            }
        }
        if (count($sourceFields) != count($destinationColumns)) {
            $this->warn("Import file field count does not match definition!", ['source_fields' => $sourceFields, 'destination_columns' => $destinationColumns]);
        }

        $loadFileObject = new \SplFileObject($loadFilePath, 'w');
        $currentLine = 0;
        while (!$localFileObject->eof()) {
            $outputRow = [];
            $raw = explode("\t", $localFileObject->current());
            $inputRow = @array_combine($sourceFields, $raw);
            if ($inputRow !== false) {
                foreach ($fieldColumnMap as $field => $column) {
                    $outputRow[$column] = str_replace([',', '"'], ['\\,', '\\"'], trim($inputRow[$field]));
                }
                $special_fields = [
                    'record_hash' => md5(implode('', $outputRow)),
                    'file_name' => $fileName,
                    'file_modified_time' => $lastModified,
                    'file_version_id' => $this->currentFile['VersionId'],
                    'sequence' => $currentLine,
                ];
                $outputRow = array_merge($special_fields, $outputRow);
                $loadFileObject->fwrite(implode(',', $outputRow) . "\n");
            } else {
                $this->warn("Line $currentLine has invalid field count, skipping line.", ['raw_line' => $raw]);
            }

            $localFileObject->next();
            $currentLine++;
        }
        $loadFileObject->fflush();
        $sortFilePath = "$loadFilePath.sorted";
        $output = $return = false;
        @exec("sort $loadFilePath > $sortFilePath", $output, $return);
        if ($return === 0) $loadFilePath = $sortFilePath;

        return implode(', ', array_keys($outputRow));
    }

    private function loadData($load_file_path, $load_columns) {
        $sql =
            "LOAD DATA LOCAL INFILE '$load_file_path' IGNORE INTO TABLE `{$this->tableName}_staging` FIELDS TERMINATED BY ',' ($load_columns)";
        $this->debug($sql);
        $this->db->getPdo()->exec(str_replace("\\", "/", $sql));
    }

    private function updateRecords() {
        $columns = [];
        
        foreach ($this->definition->getImportFields() as $field) {
            if ($field->hasTransformation()) {
                $columns[$field->getUntransformedName()] = $field->name();
                $columns[$field->name()] = $field->getTransformation();
            } else {
                $columns[$field->name()] = $field->name();
            }
        }
        $insertString = '' . implode(', ', array_keys($columns));
        $selectString = implode(', ', array_values($columns));
//        if ($this->force) {
//            $sql = <<<SQLEND
//REPLACE INTO `{$this->tableName}_records` (record_hash, first_modified_time, latest_modified_time, $insertString)
//SELECT record_hash, COALESCE(MIN(`{$this->tableName}_files`.modified_time), file_modified_time, file_modified_time, $selectString
//FROM `{$this->tableName}_staging`
//LEFT JOIN `{$this->tableName}_file_record` USING (record_hash)
//LEFT JOIN `{$this->tableName}_files` ON (`{$this->tableName}_files`.id = `{$this->tableName}_file_record`.file_id)
//GROUP BY `{$this->tableName}_staging`.record_hash
//SQLEND;
//            $prefix = 'REPLACE';
//            $suffix = '';
//        } else {
            $sql = <<<SQLEND2
INSERT IGNORE INTO `{$this->tableName}_records` (record_hash, first_modified_time, latest_modified_time, $insertString)
SELECT record_hash, file_modified_time, file_modified_time, $selectString
FROM `{$this->tableName}_staging`
ON DUPLICATE KEY UPDATE latest_modified_time = VALUES(latest_modified_time)
SQLEND2;
//        }

        $this->debug($sql);
        $this->db->statement($sql);
    }

    private function updateLatest() {
        $columns = [];
        $tableName = $this->tableName;
        foreach ($this->definition->getImportFields() as $field) {
            $columns[$field->name()] = $field->name();
        }
        $selectString = "`{$tableName}_records`.`record_hash`, `" . implode("`, `{$tableName}_records`.`", array_keys($columns)) . '`, first_modified_time as date_modified';
        $insertString = 'record_hash, ' . implode(', ', array_keys($columns)) . ', date_modified';

        $sql = <<<SQL
INSERT INTO `{$tableName}_latest` ($insertString)
SELECT $selectString FROM `{$tableName}_records`
JOIN {$tableName}_file_record ON ({$tableName}_file_record.record_hash = {$tableName}_records.record_hash)
WHERE {$tableName}_file_record.file_id IN (
  SELECT id FROM {$tableName}_files
  JOIN (SELECT file_name, MAX(modified_time) max_modified_time FROM {$tableName}_files GROUP BY file_name) maxFiles
  ON (maxFiles.file_name = {$tableName}_files.file_name AND maxFiles.max_modified_time = {$tableName}_files.modified_time)
)
SQL;
        $this->debug($sql);
        $this->db->table("{$tableName}_latest")->truncate();
        $this->db->statement($sql);
    }

    private function updateFiles() {
        $sql =
            "INSERT IGNORE INTO `{$this->tableName}_files` (file_name, modified_time, version_id) SELECT s.file_name, s.file_modified_time, s.file_version_id FROM `{$this->tableName}_staging` s LIMIT 1";
        $this->debug($sql);
        $this->db->statement($sql);
    }

    private function deleteOldRecords($file_id) {
        $sql = "DELETE FROM `{$this->tableName}_file_record` where file_id = $file_id";
        $this->debug($sql);
        $this->db->statement($sql);
    }

    private function updateFileRecords() {
        $sql =
            "INSERT IGNORE INTO `{$this->tableName}_file_record` (file_id, record_hash, sequence) SELECT f.id, s.record_hash, s.sequence FROM `{$this->tableName}_staging` s JOIN `{$this->tableName}_files` f ON (f.file_name = s.file_name AND f.modified_time = s.file_modified_time)";
        $this->debug($sql);
        $this->db->statement($sql);
    }

//    private function updateSnapshots() {
//        foreach ($this->snapshotFiles as $file => $snapshots) {
//            foreach ($snapshots as $modified => $files) {
//                $sql = "DELETE FROM `{$this->tableName}_snapshots` WHERE last_modified = \"$modified\"";
//                $this->debug($sql);
//                $this->db->statement($sql);
//                foreach ($files as $fileId) {
//                    $sql = "REPLACE INTO `{$this->tableName}_snapshots` (last_modified, file_id) VALUES (\"$modified\", $fileId)";
//                    $this->debug($sql);
//                    $this->db->statement($sql);
//                }
//            }
//        }
//    }

    private function updateStartTime($table, Carbon $time) {
        $timeString = $time->toDateTimeString();
        $sql = <<<SQL
REPLACE INTO ingestions (table_name, start_time, end_time, total_time, state)
VALUES ('$table', '$timeString', null, -1, 'ingesting')
;
SQL;
        $this->debug($sql);
        $this->db->statement($sql);
    }

    private function updateEndTime($table, Carbon $time, $tookSeconds, $humanDiff) {
        $timeString = $time->toDateTimeString();
        $sql = <<<SQL
UPDATE ingestions 
SET end_time = '$timeString', total_time = $tookSeconds, state = 'done in $humanDiff'
WHERE table_name = '$table'
;
SQL;
        $this->debug($sql);
        $this->db->statement($sql);
    }
}