<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/13/16
 * Time: 10:16 AM
 */

namespace Visionware\DataManager;

use DB;
use Exception;
use Illuminate\Support\Collection;
use Monolog\Logger;
use Visionware\DataManager\Definition\TableDefinition;
use Visionware\DataManager\Facades\DataManager;
use Carbon\Carbon;
use Illuminate\Database\Connection;

/**
 * Class Importer
 * @package Visionware\DataManager
 * @method emergency(string $msg, array $context = [])
 * @method alert(string $msg, array $context = [])
 * @method critical(string $msg, array $context = [])
 * @method error(string $msg, array $context = [])
 * @method warning(string $msg, array $context = [])
 * @method notice(string $msg, array $context = [])
 * @method info(string $msg, array $context = [])
 */
class Importer extends DataManagerProcess {
    /** @var Connection */
    protected $liveDb;
    /** @var Connection */
    protected $importDb;
    protected $insertSelectPairs;
    protected $updates;
    protected $joins;
    protected $wheres;
    protected $fromDb;
    protected $fromTable;
    protected $iterations;
    protected $lastIngested;
    protected $noTransactions;
    protected $noPostSql;
    protected $onlyPostSql;
    protected $only;

    public function __construct($schema) {
        parent::__construct($schema);
        $this->liveDb = DataManager::getLiveConnection();
        $this->importDb = DataManager::getImportConnection();
        $this->noTransactions = false;
        $this->noPostSql = false;
        $this->onlyPostSql = false;
        $this->only = false;
    }

    protected function setDestinationTable(TableDefinition $definition) {
        $this->definition = $definition;
        $this->tableName = $definition->name();
        $this->fromDb = $this->importDb->getDatabaseName();
        if (!$this->definition->isSummaryTable()) {
            $this->fromTable = "{$definition->name()}_latest";
        } else {
            $this->fromTable = "{$this->definition->getSummarySourceTable()}_latest";
        }
        $this->insertSelectPairs = new Collection;
        $this->updates = new Collection;
        $this->joins = new Collection;
        $this->wheres = new Collection;
        $this->iterations = 1;
        $this->lastIngested = false;

        $this->setLogPrefix($this->definition->name());
    }

    public function skipPostSql() {
        $this->noPostSql = true;
        $this->onlyPostSql = false;
    }

    public function onlyPostSql() {
        $this->noPostSql = false;
        $this->onlyPostSql = true;
    }

    public function go() {
        $allStart = Carbon::now();
        try {
            $this->info("Acquiring import lock...");
            DataManager::acquireImportLock();
            if (!$this->noTransactions) $this->liveDb->beginTransaction();
            if (!$this->onlyPostSql) {
                $noNewData = [];
                foreach ($this->schema->tablesToImport() as $tableName) {
                    if ($this->only && !($tableName == $this->only)) continue;
                    $this->setDestinationTable($tableName->definition());
                    if ($this->importTable() == -1) $noNewData[$tableName->name()] = $tableName->name();
                }
                foreach ($this->schema->tablesToImport() as $tableName) {
                    if ($this->only && !($tableName == $this->only)) continue;
                    if (in_array($tableName->name(), $noNewData)) continue;
                    $this->setDestinationTable($tableName->definition());
                    $this->updateForeignKeys();
                }
            }
            if (!$this->noTransactions) $this->liveDb->commit();
        } catch (Exception $e) {
            $this->error("Caught exception while importing table! Rolling back...", ['exception' => $e]);
            if (!$this->noTransactions) $this->liveDb->rollBack();
        } finally {
            $this->info("Releasing import lock...");
            DataManager::releaseImportLock();
        }
        $diff = Carbon::now()->diffForHumans($allStart, true);
        $this->notice("Finished import process in $diff total");
    }

    public function noTransactions() {
        $this->noTransactions = true;
    }

    public function only($tableName) {
        $this->only = $tableName;
    }

    protected function pushInsert($insert, $select) {
        $this->insertSelectPairs->put($insert, "$select AS $insert");
    }
    
    protected function pushJoin($joinStatement) {
        $this->joins->push($joinStatement);
    }

    protected function pushUpdate($updateStatement) {
        $this->updates->push($updateStatement);
    }

    protected function pushWhere($whereStatement) {
        $this->wheres->push($whereStatement);
    }

    public function importTable() {
        $start = Carbon::now();

        $this->notice("Importing table...");
        $joinedTables = [];
        if ($this->definition->hasUuid()) {
            $this->pushInsert('id', "COALESCE({$this->compileField('id', $this->definition->name())}, UuidToBin(UUID()))");
        }
        $join = ['table' => $this->definition->name(), 'using' => $this->definition->key()];
        $joinedTables[$this->definition->name()] = true;
        $this->pushJoin($this->compileJoin($join, $this->definition->name()));
        foreach ($this->definition->fields() as $field) {
            if ($field->skipImport()) continue;
            if ($field->hasImportField()) { //exists in History DB
                $this->pushInsert($field->name(), $this->compileField($field->name()));
            } else if ($field->hasForeign()) {
                $joins = $field->getJoins();
                foreach ($joins as $join) {
                    if ($join['table'] == $this->definition->name()) $this->iterations++;
                    $alias = $field->name() . '_' . $join['table'] . '_Join';
                    $this->pushInsert($field->name(), $this->compileField($field->foreign(), $alias));
                    $joinedTables[$join['table']] = true;
                    $this->pushJoin($this->compileJoin($join, $alias));
                }
            } else if ($field->hasImportedFrom()) {
                $this->pushInsert($field->name(), $this->compileField($field->name()));
            }
            $this->pushUpdate("{$field->name()} = VALUES({$field->name()})");
        }

        if (!$this->isThereUpdatedData(array_keys($joinedTables)) && !$this->force) {
            $this->info("No new data, skipping table.");
            return -1;
        }

        $newCommitSequence = false;
        if ($this->definition->isSyncedDown()) {
            $this->info('Getting new commit sequence number');
            $newCommitSequence = $this->liveDb->table($this->definition->name())->lockForUpdate()->max('commit_sequence') + 1;
            $this->info("New commit sequence number: $newCommitSequence");
        }

        $this->pushInsert('date_modified', $this->compileField('date_modified'));
        $this->pushUpdate("date_modified = VALUES(date_modified)");

        if (!$this->force) {
            $this->pushWhere("(`{$this->definition->name()}`.`date_modified` IS NULL OR `{$this->fromTable}`.`date_modified` > `{$this->definition->name()}`.`date_modified`)");
        }
        $this->pushUpdate("deleted_at = 0");
        $insertSql = $this->compileInsert($newCommitSequence);
        $updateSql = $this->compileUpdateDeleted($newCommitSequence);

        $is_append_only = $this->definition->getIsAppendOnly();
        for ($count = 1; $count <= $this->iterations; $count++) {
            $this->info($this->definition->name() . ": Inserting into live (iteration $count)...");
            $this->logSql($insertSql);
            $rows = $this->liveDb->affectingStatement($insertSql);
            $this->debug("Affected $rows rows");

            if (!$is_append_only) {
                $this->info($this->definition->name() . ": Updating deleted records (iteration $count)...");
                $this->logSql($updateSql);
                $rows = $this->liveDb->affectingStatement($updateSql);
                $this->debug("Affected $rows rows");
            }
        }

        $end = Carbon::now();
        $diff = $end->diffForHumans($start, true);
        $sql = $this->compileUpdateLastImport($end->diffInSeconds($start, true));
        $this->logSql($sql);
        $this->liveDb->affectingStatement($sql);

        $this->notice("Completed importing table in $diff total");
    }

    protected function isThereUpdatedData($joinedTables) {
        foreach ($joinedTables as $joinedTable) {
            $this->debug("Checking $joinedTable for new data...");
            $joinDefinition = $this->definition->schema()->tables()->get($joinedTable);
            $joinFromTable = $joinDefinition->isSummaryTable() ? $joinDefinition->getSummarySourceTable() : $joinedTable;
            $recordsInLive = $this->liveDb->table($joinedTable)->value(DB::raw('COUNT(*)'));
            $recordsToIngest = $this->importDb->table("{$joinFromTable}_latest")->value(DB::raw('COUNT(*)'));
            $this->debug("$joinedTable ingested record count: $recordsInLive");
            $this->debug("$joinFromTable ingested record count: $recordsToIngest");
            if ($recordsToIngest < 1) {
                if (!$this->definition->getIsAppendOnly()) {
                    // It is expected that IsAppendOnly tables will be empty fairly often, no need to report it as an error
                    $this->error("$joinFromTable has no records, which means the transfer operation failed!");
                }
                return false;
            }
            $lastImported = $this->liveDb->table($joinedTable)->value(DB::raw('MAX(date_modified)'));
            $lastIngested = $this->importDb->table("{$joinFromTable}_latest")->value(DB::raw('MAX(date_modified)'));
            $this->debug("Last $joinedTable imported time: $lastImported");
            $this->debug("Last $joinFromTable ingested time: $lastIngested");
            if ($recordsInLive != $recordsToIngest || $lastIngested > $lastImported) {
                $this->lastIngested = $lastIngested = $this->importDb->table($this->fromTable)->value(DB::raw('MAX(date_modified)'));
                return true;
            }
        }
        return false;
    }

    protected function updateForeignKeys() {
        $fJoins = $this->definition->getForeignDependentJoins();
        $commitSequences = [];

        foreach ($fJoins as $field) {
            if ($field->table()->name() == $this->definition->name()) continue;
            $toTable = $field->table()->name();

            $joins = $field->getJoins();
            foreach ($joins as $join) {
                $start = Carbon::now();
                if ($field->table()->isSyncedDown() && !array_has($commitSequences, $toTable)) {
                    $this->info("Getting commit sequence number for $toTable");
                    $commitSequences[$toTable] = $this->liveDb->table($toTable)->lockForUpdate()->max('commit_sequence') + 1;
                    $this->info("New commit sequence number for $toTable: $commitSequences[$toTable]");
                }
                $sql = $this->compileUpdateFk($join, $toTable, $this->definition->name(), $field->name(), $field->foreign(), array_get($commitSequences, $toTable, false), $field->needsContinuousUpdate());
                $this->info("Updating foreign key $toTable.{$field->name()} from {$this->definition->name()}.{$field->foreign()}");
                $this->logSql($sql);
                $rows = $this->liveDb->affectingStatement($sql);
                $end = Carbon::now();
                $diff = $end->diffForHumans($start, true);
                $this->notice("Affected $rows rows in $diff while updating foreign key $toTable.{$field->name()} from {$this->definition->name()}.{$field->foreign()}");
            }
        }
    }

    protected function compileInsert($newCommitSequence) {
        $insertString = implode(",\n  ", $this->insertSelectPairs->keys()->toArray());
        $selectString = implode(",\n  ", $this->insertSelectPairs->values()->toArray());
        $joinsString = implode("\n", $this->joins->toArray());
        $updatesString = implode(",\n  ", $this->updates->values()->toArray());
        $wheresString = implode(' AND ', $this->wheres->values()->toArray());

        // if unique_key on the table was foolishly created with deleted_at as part of the index,
        // need to include a (<table>.deleted_at = 0 OR <table>.deleted_at IS NULL) check as part of
        // the main where clause ($whereString)
        $deletedAtWhereString = "";
        $uniqueKey = $this->definition->key();
        if (in_array('deleted_at', $uniqueKey)) {
            $deletedAtWhereString = '(' . $this->definition->name() . '.deleted_at = 0 OR ' . $this->definition->name() . '.deleted_at IS NULL' . ')';
        }

        $commitSequenceInsertString = $newCommitSequence ? ', commit_sequence' : '';
        $commitSequenceSelectString = $newCommitSequence ? ", $newCommitSequence" : '';
        $commitSequenceUpdateString = $newCommitSequence ? ", `commit_sequence` = $newCommitSequence" : '';

        if (strlen($wheresString)) {
            $wheresString = "WHERE $wheresString";
        }

        if (strlen($deletedAtWhereString)) {
            if (strlen($wheresString)) {
                $wheresString = "$wheresString AND $deletedAtWhereString";
            } else {
                $wheresString = "WHERE $deletedAtWhereString";
            }
        }

        $sql = <<<SQL
INSERT IGNORE INTO `{$this->liveDb->getDatabaseName()}`.`{$this->definition->name()}` (
  $insertString $commitSequenceInsertString
)
SELECT
  $selectString $commitSequenceSelectString
FROM `{$this->fromDb}`.`{$this->fromTable}`
$joinsString
$wheresString 
ON DUPLICATE KEY UPDATE
  $updatesString $commitSequenceUpdateString
;
SQL;
        return $sql;
    }
    
    protected function compileUpdateDeleted($newCommitSequence) {
        $destinationDb = $this->liveDb->getDatabaseName();
        $definition = $this->definition->name();
        $sourceDb = $this->fromDb;
        $sourceTable = $this->fromTable;

        $maxDateModified = $this->importDb->table($sourceTable)->value(DB::raw('MAX(date_modified)'));
        $dateModifiedString = ", `$definition`.`date_modified` = '$maxDateModified'";
        $commitSequenceString = $newCommitSequence ? ", `commit_sequence` = $newCommitSequence" : '';

        $joinStrings = [];
        foreach ($this->definition->key() as $keyColumn) {
            $joinStrings[] = "`$definition`.`$keyColumn` = `$sourceTable`.`$keyColumn`";
        }
        $joinString = implode(' AND ', $joinStrings);

        $sql = <<<SQL
UPDATE `$destinationDb`.`$definition`
LEFT OUTER JOIN `$sourceDb`.`$sourceTable`
ON ($joinString)
SET `$definition`.`deleted_at` = '{$this->lastIngested}' $dateModifiedString $commitSequenceString 
WHERE `$sourceTable`.`record_hash` IS NULL
AND `$definition`.`deleted_at` = 0;
SQL;
        return $sql;
    }

    protected function compileField($column, $table = false) {
        if ($table === false) $table = $this->fromTable;
        return "`$table`.`$column`";
    }

    protected function compileUpdateLastImport($duration) {
        $sql = <<<SQL
INSERT IGNORE INTO `{$this->liveDb->getDatabaseName()}`.`last_imported`
(table_name, last_imported, import_duration)
VALUES ('{$this->definition->name()}', '{$this->lastIngested}', $duration)
ON DUPLICATE KEY UPDATE last_imported = VALUES(last_imported), import_duration = VALUES(import_duration)
SQL;
        return $sql;
    }

    protected function compileUpdateFK($joinArray, $toTable, $fromTable, $toField, $fromField, $commitSequence, $needsContinuousUpdate) {
        $newJoinArray = array_except($joinArray, ['ons']);
        if (array_has($joinArray, 'ons')) {
            foreach ($joinArray['ons'] as $on) {
                $newJoinArray['ons'][] = [
                    'local' => $on['foreign'],
                    'foreign' => $on['local']
                ];
            }
        }

        $linkString = $this->compileJoinLinkString($newJoinArray, $toTable, $fromTable);
        $linkString = preg_replace('/\)$/', " AND `$fromTable`.`deleted_at` = '0000-00-00 00:00:00')", $linkString);

        $dateModifiedString = ", `$toTable`.`date_modified` = `$toTable`.`date_modified`";
        $commitSequenceString = $commitSequence ? ", `$toTable`.`commit_sequence` = $commitSequence" : '';

        if ($needsContinuousUpdate){
            $whereClause = <<<SQL
WHERE 
(`$toTable`.`$toField` IS NULL AND `$fromTable`.`$fromField` IS NOT NULL)
OR (`$toTable`.`$toField` IS NOT NULL AND `$fromTable`.`$fromField` IS NULL)
OR (`$toTable`.`$toField` != `$fromTable`.`$fromField`)
SQL;
        }else{
            $whereClause = <<<SQL
WHERE 
(`$toTable`.`$toField` IS NULL AND `$fromTable`.`$fromField` IS NOT NULL)
SQL;
        }

        $sql = <<<SQL
UPDATE `{$this->liveDb->getDatabaseName()}`.`$toTable`
JOIN `{$this->liveDb->getDatabaseName()}`.`$fromTable`
$linkString
SET `$toTable`.`$toField` = `$fromTable`.`$fromField` $dateModifiedString $commitSequenceString
$whereClause
SQL;
        return $sql;
    }

    protected function compileJoinLinkString($joinArray, $toTable, $fromTable) {
        $linkString = '';
        if (array_key_exists('using', $joinArray)) {
            $linkString = 'USING (' . implode(', ', $joinArray['using']) . ')';
        } else {
            $onStrings = [];
            foreach ($joinArray['ons'] as $on) {
                $onStrings[] = "`$toTable`.`$on[foreign]` = `$fromTable`.`$on[local]`";
            }
            $linkString = 'ON (' . implode(' AND ', $onStrings) . ')';
        }

        return $linkString;
    }

    protected function compileJoin($joinArray, $alias) {
        $linkString = $this->compileJoinLinkString($joinArray, $alias, $this->fromTable);

        return "LEFT OUTER JOIN `{$this->liveDb->getDatabaseName()}`.`$joinArray[table]` AS $alias $linkString";
    }

    private function logSql($sql) {
//        fputs($this->sqlLogFile, "$sql\n\n");
        $this->debug($sql);
    }
}