<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 4/16/16
 * Time: 2:29 PM
 */

namespace Visionware\DataManager\Grammars;

use Visionware\DataManager\Info\ColumnInfo;
use Visionware\DataManager\Info\IndexInfo;
use Visionware\DataManager\Info\TableInfo;
use Visionware\DataManager\Info\ViewInfo;

class MysqlGrammar extends Grammar {

    public function createTable(TableInfo $table) {
        $columnStrings = $this->compileEach($table->columns(), 'defineColumn');
        $indexStrings = $this->compileEach($table->indices(), 'defineIndex');
        $strings = implode(",\n  ", array_merge($columnStrings, $indexStrings));
        $tableName = $table->name();
        return <<<SQL
CREATE TABLE `$tableName` (
  $strings
) ENGINE=InnoDB DEFAULT CHARSET=utf8;
SQL;
    }

    protected function defineColumn(ColumnInfo $column) {
        $name = '`' . $column->name() . '` ';
        $type = $column->qualifiedType();
        $null = $column->nullable() ? "" : " NOT NULL";
        $default = strlen($column->default()) ? " DEFAULT {$column->default()}" : ($column->nullable() ? ' DEFAULT NULL' : '');
        $extra = strlen($column->extra()) ? ' ' . $column->extra() : '';
        return implode('', [$name, $type, $null, $default, $extra]);
    }

    protected function defineIndex(IndexInfo $index) {
        $name = $index->name();
        $name_string = strlen($name) ? " `$name`" : '';
        $columns_string = ' (`' . implode('`, `', $index->columns()) . '`)';
        return implode('', [$index->type(), $name_string, $columns_string]);
    }

    public function dropTable(TableInfo $table) {
        $name = $table->name();
        return "DROP TABLE IF EXISTS `$name`;";
    }

    public function createColumn(ColumnInfo $column) {
        $tableName = $column->table()->name();
        $columnString = $this->defineColumn($column);
        return "ALTER TABLE `$tableName` ADD COLUMN $columnString;";
    }

    public function dropColumn(ColumnInfo $column) {
        $tableName = $column->table()->name();
        $columnName = $column->name();
        return "ALTER TABLE `$tableName` DROP COLUMN $columnName;";
    }

    public function createIndex(IndexInfo $index) {
        $tableName = $index->table()->name();
        $indexString = $this->defineIndex($index);
        return "ALTER TABLE `$tableName` ADD $indexString;";
    }

    public function dropIndex(IndexInfo $index) {
        $tableName = $index->table()->name();
        $indexName = $index->name();
        return "DROP INDEX `$indexName` ON `$tableName`;";
    }

    public function alterColumn(ColumnInfo $column) {
        $tableName = $column->table()->name();
        $columnName = $column->name();
        $columnString = $this->defineColumn($column);
        return "ALTER TABLE `$tableName` CHANGE COLUMN `$columnName` $columnString;";
    }
    
    public function createView(ViewInfo $view) {
        $viewName = $view->name();
        $statement = $view->statement();
        return "CREATE VIEW `$viewName` AS $statement;";
    }

    public function dropView(ViewInfo $view) {
        $viewName = $view->name();
        return "DROP VIEW `$viewName`;";
    }

    public function createSchema($name) {
        return "CREATE DATABASE IF NOT EXISTS `$name`;";
    }

    public function dropSchema($name) {
        return "DROP DATABASE IF EXISTS `$name`;";
    }

    public function useSchema($name) {
        return "USE $name;";
    }

    public function createAlphaNumFunction() {
        return <<<SQL
DELIMITER |
CREATE FUNCTION alphanum( str VARCHAR(255) )
RETURNS VARCHAR(255)
  LANGUAGE SQL
  DETERMINISTIC
  CONTAINS SQL
  SQL SECURITY INVOKER
BEGIN
  DECLARE i, len SMALLINT DEFAULT 1;
  DECLARE ret VARCHAR(255) DEFAULT '';
  DECLARE c CHAR(1);
  IF str IS NULL THEN 
    SET ret = '';
  ELSE 
	  SET len = CHAR_LENGTH( str );
	  REPEAT
		BEGIN
		  SET c = MID( str, i, 1 );
		  IF c REGEXP '[[:alnum:]]' THEN
			SET ret=CONCAT(ret,c);
		  END IF;
		  SET i = i + 1;
		END;
	  UNTIL i > len END REPEAT;
  END IF;
  RETURN ret;
END |
DELIMITER ;
SQL;
    }

    public function createUuidToBinFunction() {
        return <<<SQL
CREATE FUNCTION UuidToBin(_uuid BINARY(36))
  RETURNS BINARY(16)
  LANGUAGE SQL
  DETERMINISTIC
  CONTAINS SQL
  SQL SECURITY INVOKER
  RETURN UNHEX(
    CONCAT(
      SUBSTR(_uuid, 15, 4),
      SUBSTR(_uuid, 10, 4),
      SUBSTR(_uuid,  1, 8),
      SUBSTR(_uuid, 20, 4),
      SUBSTR(_uuid, 25)
    )
  );
SQL;

    }

    public function createUuidFromBinFunction() {
        return <<<SQL
CREATE FUNCTION UuidFromBin(_bin BINARY(16))
  RETURNS CHAR(36)
  LANGUAGE SQL
  DETERMINISTIC
  CONTAINS SQL
  SQL SECURITY INVOKER
  RETURN LCASE(
    CONCAT_WS(
      '-',
      HEX(SUBSTR(_bin,  5, 4)),
      HEX(SUBSTR(_bin,  3, 2)),
      HEX(SUBSTR(_bin,  1, 2)),
      HEX(SUBSTR(_bin,  9, 2)),
      HEX(SUBSTR(_bin, 11))
    )
  );
SQL;

    }

//    public function replace($table, $columns, $values) {
//        if (!is_array($columns)) $columns = [$columns];
//        $columns_string = ' (`' . implode('`, `', $columns) . '`)';
//        if (!is_array($values)) $values = [$values];
//        $values_string = ' (' . implode(', ', $values) . ')';
//        return "REPLACE INTO `$table` $columns_string VALUES $values_string;";
//
//    }
}