<?php
/**
 * Created by PhpStorm.
 * User: cbarranco
 * Date: 5/20/16
 * Time: 12:43 PM
 */

namespace Visionware\DataManager\Definition;

use Illuminate\Support\Collection;

class FieldDefinition extends Definition {
    private $table;
    /**
     * @var string
     */
    private $name;
    /**
     * @var string
     */
    private $type;

    public function setName($name) {
        $this->name = $name;
    }

    public function setType($type) {
        $this->type = $type;
    }

    public function setTable($table) {
        $this->table = $table;
    }

    public function name() {
        return $this->name;
    }

    public function getQualifiedName() {
        return "`{$this->table()->name()}`.`{$this->name()}`";
    }
    
    public function type() {
        return $this->type;
    }

    /**
     * @return TableDefinition
     */
    public function table() {
        return $this->table;
    }

    public function onTable() {
        return $this->getOther('on_table', false);
    }

    public function importedFrom() {
        return $this->getOther('imported_from', false);
    }
    
    public function hasImportedFrom() {
        return $this->importedFrom() !== false;
    }

    public function foreign() {
        return $this->getOther('foreign', false);
    }

    public function hasForeign() {
        return $this->foreign() !== false;
    }

    public function hasJoin() {
        return $this->getOther('on_table', false) !== false;
    }

    public function getJoins() {
        if (!$this->hasJoin()) return false;
        $joinTable = $this->getOther('on_table');
        $joins = [];
        foreach ($this->getOther('extra_joins', []) as $extra) {
            $extraTable = $extra['imported_from'];
            if (array_key_exists('using', $extra)) {
                $using = $extra['using'];
                if (!is_array($using)) $using = [$using];
                $joins[] = ['table' => $extraTable, 'using' => $using];
            } else {
                $joins[] = ['table' => $extraTable, 'ons' => [['local' => $extra['local'], 'foreign' => $extra['foreign']]]];
            }

        }
        if ($this->getOther('using')) {
            $using = $this->getOther('using');
            if (!is_array($using)) $using = [$using];
            $joins[] = ['table' => $joinTable, 'using' => $using];
        } else if ($this->getOther('join_on')) {
            $ons = [];
            foreach ($this->getOther('join_on') as $on) {
                $ons[] = $on;
            }
            $joins[] = ['table' => $joinTable, 'ons' => $ons];
        }
        return $joins;
    }

    /**
     * @return TableDefinition|bool
     */
    public function getDependentTable() {
        $dependent = false;
        if ($this->onTable()) $dependent = $this->table()->schema()->tables()->get($this->onTable());
        else if ($this->importedFrom()) $dependent = $this->table()->schema()->tables()->get($this->importedFrom());
        if ($dependent && $dependent->name() != $this->name()) return $dependent;
        else return false;
    }

    public function hasDependentTable() {
        return $this->getDependentTable() !== false;
    }
    
    public function isNullable() {
        if ($this->isPartOfUniqueKey()) return false;
        return $this->getOther('nullable', false);
    }

    public function hasTransformation() {
        return $this->getOther('transformation', false) !== false;
    }

    public function getUntransformedName() {
        return $this->name() . '__untransformed';
    }

    public function getTransformation($qualifiedName = false) {
        $qName = $qualifiedName ? $this->getQualifiedName() : $this->name();
        if (!$this->hasTransformation()) return $qName;
        $transformation = $this->getOther('transformation');
        switch ($transformation['type']) {
            case 'date':
                return "STR_TO_DATE($qName, \"$transformation[format]\")";
            case 'boolean':
                return "CASE WHEN $qName = '$transformation[true_value]' THEN 1 WHEN $qName = '$transformation[false_value]' THEN 0 END";
            case 'round':
                return "ROUND($qName, $transformation[value])";
        }
    }

    public function getExtraJoins() {
        return $this->getOther('extra_joins', []);
    }

    public function joinOns() {
        return $this->getOther('join_on', []);
    }

    public function hasJoinOns() {
        return $this->getOther('join_on', false) !== false;
    }

    public function hasUsing() {
        return $this->getOther('using', false) !== false;
    }

    public function using() {
        return $this->getOther('using', false);
    }

    public function default() {
        $value = $this->getOther('default', '');
        if (!$this->isNullable() && $value === 'NULL') $value = '';
        return $value;
    }

    public function getImportField() {
        return $this->getOther('import_field', false);
    }

    public function hasImportField() {
        return $this->getOther('import_field', false) !== false;
    }

    public function shouldImport() {
        if ($this->foreign()) return false;
        $noImport = $this->getOther('no_import', false);
        return !$noImport;
    }

    public function skipImport() {
        return $this->getOther('no_import', false) === true;
    }

    public function extra() {
        return $this->getOther('extra');
    }

    public function isPartOfUniqueKey() {
        return $this->table()->isPartOfUniqueKey($this);
    }

    public static function hydrate($array) {
        $instance = new static();
        $instance->setName($array['name']);
        $instance->setType($array['type']);
        $instance->setOther(collect(array_except($array, ['name', 'type'])));
        return $instance;
    }

    public function validate() {
        
    }
}